// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file TrialMover
/// @brief performs a move and accepts it according to Monte Carlo accept/reject criterion.
/// @author Monica Berrondo


#ifndef INCLUDED_protocols_moves_TrialMover_HH
#define INCLUDED_protocols_moves_TrialMover_HH

// Unit headers
#include <protocols/moves/TrialMover.fwd.hh>

// Package headers
#include <protocols/moves/MonteCarlo.hh> // THIS SHOULD BE REMOVED
#include <protocols/moves/Mover.hh>
#include <protocols/moves/MoverStatistics.hh>

#include <core/scoring/ScoreType.hh> /// BAD
#include <core/scoring/ScoreFunction.hh> /// BAD

#include <core/types.hh>

#include <core/pose/Pose.fwd.hh>

// ObjexxFCL Headers

// C++ Headers
#include <map>
#include <string>

// Utility Headers


namespace protocols {
namespace moves {

///////////////////////////////////////////////////////////////////////////////
/// @brief:
/// 	A trial takes a mover and an mc object and applies the move then accepts
///   according to monte carlo object.
///
/// @detailed:
/// 	Each derived class should define its own apply() statement
/// 	the apply (mc) which requires a monte carlo object and only keeps
/// 	the move if the monte carlo test allows it
///
/// @authors Monica Berrondo
///
/// @last_modified August 16 2007
///////////////////////////////////////////////////////////////////////////////

// silly little enum to keep track of how we want to
// keep track of statistics from this Mover. It's
// important with the current design that these are
// kept in a hierarchically increasing order, so that
// the StatsType of i-1 keeps all of the same
// statistics as StatsType.
enum StatsType {
	all_stats = 1, // scores, accepts and rejects
	accept_reject, // only accept and reject counts
	no_stats	     // no stats. Always keep last.
};

class TrialMover : public Mover {
public:

	typedef core::Real Real;

public:

	// default constructor -- BAD, MOVE TO .CC
	// TODO: There is no setter for mover - so do we need this constructor?
	TrialMover() :
		start_weight_( 0.0 ),
		original_weight( 0.0 ),
		ramp_weight( 0.0 ),
		delta( 0.0 ),
		stats_type_( all_stats )
	{}

	// constructur with arguments -- BAD, MOVE TO .CC
	TrialMover( MoverOP mover_in, MonteCarloOP mc_in ) :
		start_weight_( 0.0 ),
		original_weight( 0.0 ),
		ramp_weight( 0.0 ),
		delta( 0.0 ),
		stats_type_( all_stats )
	{
		mover_ = mover_in;
		mc_ = mc_in;
	}

	// set the weights for the score_type for ramping -- BAD, MOVE TO .CC
	virtual void initialize_weights(
		Real const start_weight,
		Real const end_weight,
		core::scoring::ScoreType score_type,
		int const ramp_cycles
	) {
		original_weight = mc_->score_function().get_weight( score_type );
		delta = ( end_weight - start_weight ) / ramp_cycles;
		ramp_weight = start_weight;
		start_weight_ = start_weight;
	}

	/// apply does a single trial (which is a mover apply and a boltzmann)
	virtual void apply( core::pose::Pose & pose );
	Real acceptance_rate() const;
	/// @brief Returns the number of accepts made by this TrialMover.
	int num_accepts() const;

	// replace MonteCarlo Object of trial mover -- BAD, MOVE TO .CC
	void set_mc( MonteCarloOP mc_in ) {
		mc_ = mc_in;
	}

	/// @brief return underlying mover
	MoverOP mover() {
		return mover_;
	}

	MonteCarlo const& mc() {
		return *mc_;
	}

	StatsType keep_stats_type() const {
		return stats_type_;
	}

	void keep_stats_type( StatsType setting ) {
		stats_type_ = setting;
	}

	// @brief sets the input pose also for the contained mover
	virtual void set_input_pose( PoseCOP pose );

	// @brief sets the native pose also for the contained mover
	virtual void set_native_pose( PoseCOP pose );

protected:
	MoverOP mover_;
	MonteCarloOP mc_;
	MoverStatistics stats_;

private:
	Real start_weight_;
	Real original_weight;
	Real ramp_weight;
	Real delta;
	StatsType stats_type_;
}; // TrialMover base class

} // moves
} // protocols


#endif //INCLUDED_protocols_moves_TrialMover_HH
