// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Liz Kellogg

#include <protocols/moves/Mover.hh>
#include <protocols/moves/ddGMover.hh>

#include <core/types.hh>

#include <core/io/database/open.hh>

#include <core/chemical/AA.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/ResidueMatcher.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/ResidueSelector.hh>
#include <core/conformation/ResidueFactory.hh>
#include <core/conformation/Interface.hh> //added ek for interface ddgs

#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/pack/pack_rotamers.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>

#include <core/kinematics/MoveMap.hh>

#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>

//constraint stuff
#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/ConstraintSet.fwd.hh>
#include <core/scoring/constraints/AtomPairConstraint.hh>
#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/constraints/Func.hh>

#include <core/id/AtomID_Map.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/id/AtomID.hh>
#include <core/id/DOF_ID.hh>

#include <core/pose/Pose.hh>
#include <core/pose/Pose.fwd.hh>

#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/BinaryProteinSilentStruct.hh>

#include <core/options/util.hh>
#include <core/options/after_opts.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/options/keys/score.OptionKeys.gen.hh>
#include <core/options/keys/ddg.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>

#include <core/init.hh>
#include <core/io/pdb/pose_io.hh>

#include <numeric/xyzVector.hh>
#include <numeric/random/random.hh>
#include <core/pack/task/ResfileReader.hh>

#include <protocols/moves/RigidBodyMover.hh>

#include <fstream>
#include <iostream>
#include <sstream>
#include <ios>
#include <utility/io/izstream.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ headers
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>
#include <sstream>
#include <sys/stat.h>

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;
static core::util::Tracer TR("protocols.moves.ddGMover");

namespace protocols {
namespace moves {

numeric::random::RandomGenerator RG(15430); // <- Magic number, do not change it!!!

using namespace core;
using namespace scoring;

typedef std::vector<double> ddGs;

	ddGMover::ddGMover():
		Mover("ddGMover"),
    mean_(false),
		min_(true),
		min_cst_(true),
		nbr_cutoff_(8.0),
		restrict_to_nbrhood_(false),
		interface_ddg_(false),
		scorefxn_(0),
		min_cst_sfxn_(0),
		residues_to_mutate_(0,core::chemical::aa_unk),
		num_iterations_(20),
		dmp_pdb_(false),
		dbg_output_(false),
		wt_components_(1,1,-999.99),
		wt_unbound_components_(1,1,-999.99),//only used for interface ddg
		mutant_components_(1,1,-999.99),
		mutant_unbound_components_(1,1,-999.99),//only used for interface ddg
		natives_(),
		mutants_()
		{}

	ddGMover::ddGMover(
										 core::scoring::ScoreFunctionOP s,
										 core::scoring::ScoreFunctionOP m,
										 utility::vector1<core::chemical::AA> res_to_mutate
										 ):
		Mover("ddGMover"),
		mean_(false),
		min_(true),
		min_cst_(true),
		nbr_cutoff_(8.0),
		restrict_to_nbrhood_(false),
		interface_ddg_(false),
		scorefxn_(s),
		min_cst_sfxn_(m),
		residues_to_mutate_(res_to_mutate),
		num_iterations_(20),
		dmp_pdb_(false),
		dbg_output_(false),
		wt_components_(1,1,-999.99),
		wt_unbound_components_(1,1,-999.99),//only used for interface ddg
		mutant_components_(1,1,-999.99),
		mutant_unbound_components_(1,1,-999.99),//only used for interface ddg
		natives_(),
		mutants_()
	{}

ddGMover::~ddGMover(){}


bool
ddGMover::is_complete(ObjexxFCL::FArray2D<double> to_check){
	bool is_complete=true;
	for(int i = to_check.l1(); i <= to_check.u1();i++){
		for(int j = to_check.l2(); j <= to_check.u2();j++){
			if(to_check(i,j) == -999.99){
				is_complete=false;
			}
		}
	}
	return is_complete;
}

double
ddGMover::sum(ddGs &scores_to_sum)
{
	double sum=0;
	for(unsigned int i =0;i<scores_to_sum.size();i++){
		sum+=scores_to_sum[i];
	}
	return sum;
}

double
ddGMover::average( utility::vector1<double> &scores_to_average)
{
	double sum = 0;
	for(unsigned int i =1;i<=scores_to_average.size();i++){
		sum+=scores_to_average[i];
	}
	return (sum/scores_to_average.size());
}

int
ddGMover::store_energies( ObjexxFCL::FArray2D< double > &two_d_e_arrays,
								scoring::ScoreFunction &s,
								pose::Pose &p, int next_index , int size_to_expect)
{
	p.update_residue_neighbors();
	s(p);

	//hack. we don't want to consider atom-pair constraints in computing ddgs!
	core::scoring::ScoreType const atom_pair_cst = score_type_from_name("atom_pair_constraint");
	core::Real saved_weight = 0;
	if(s.get_weight( atom_pair_cst) > 0){
		saved_weight = s.get_weight(atom_pair_cst);
	}
	s.set_weight( atom_pair_cst, (core::Real)0);
	//

	/// Now handled automatically.  s.accumulate_residue_total_energies(p);
	//s.show(TR,p);

	//all this to determine how many non-zero weights there are
	int num_score_components = 0;
	EnergyMap::const_iterator it = s.weights().begin();
	while(it != (s.weights()).end()){
		it++;
		if(*it != 0){
			num_score_components++;
		}
	}
	//
	if((two_d_e_arrays.u1() != num_score_components) ||
		 (two_d_e_arrays.u2() != size_to_expect)){
		two_d_e_arrays.dimension(num_score_components,size_to_expect,-999.99);
	}
	int current_score_component=0;
	int j =1;
	for( EnergyMap::const_iterator i = (s.weights()).begin(); i != s.weights().end();i++){
		//get score component of pose, then store in next slot of two_d_e_arrays
		current_score_component++;
		if(*i != 0){
			two_d_e_arrays(j++,next_index)=(*i) *
				((p.energies()).total_energies())[ScoreType(current_score_component)];
		}
	}

	s.set_weight(atom_pair_cst,saved_weight);

	return 0;
}

int
ddGMover::average_score_components( ObjexxFCL::FArray2D< double > &scores_to_average,
													utility::vector1<double> &averaged_scores )
{
	averaged_scores = utility::vector1<double>(scores_to_average.u1()-scores_to_average.l1()+1);
	for(int i = scores_to_average.l1(); i <= scores_to_average.u1(); i++){
		double sum_score_component = 0;
		for(int j = scores_to_average.l2(); j <= scores_to_average.u2(); j++){
			sum_score_component += scores_to_average(i,j);
			//std::cout << "score being averaged at index " << j << " is " << scores_to_average(i,j) << std::endl;
		}
		//std::cout << "total sum is " << sum_score_component << " modulated by " << (scores_to_average.u2()-scores_to_average.l2()+1)<< std::endl;
		averaged_scores[i]=
			sum_score_component/(scores_to_average.u2()-scores_to_average.l2()+1);
	}
	return 0;
}

void
ddGMover::calculate_interface_unbound_energy(pose::Pose & p,ScoreFunctionOP s,core::pack::task::PackerTaskOP pt){

	using namespace protocols::moves;
	using namespace core::pack;
	//from sarel. thank you!
	int const rb_jump(1);
	RigidBodyTransMoverOP separate_partners( new RigidBodyTransMover(p,rb_jump));
	separate_partners->step_size(1000.0);
	separate_partners->apply(p);
	pack::pack_rotamers(p,(*s),pt);
}

void
ddGMover::setup_constraints(pose::Pose & pose, ScoreFunctionOP sfxn, float const cst_tol){
		//create constraints for all residues
	//type: HARMONIC
	using namespace core::scoring;
	using namespace core::scoring::constraints;
	using namespace core::options::OptionKeys;
	using namespace core::options;
	static float const CA_cutoff(9.0);

	double const CONSTRAINT_WEIGHT = 1.0;

	if(options::option[core::options::OptionKeys::constraints::cst_file].user()){
		core::scoring::constraints::ConstraintSetOP cstset(new
		core::scoring::constraints::ConstraintSet());
		cstset = core::scoring::constraints::ConstraintIO::read_constraints(
						  option[core::options::OptionKeys::constraints::cst_file][1],cstset,
							pose);
		pose.constraint_set(cstset);
	}
	else{
		int nres = pose.total_residue();
		for(int i = 1; i <= nres; i++){
			if(pose.residue(i).is_protein()){
				Vector const CA_i( pose.residue(i).xyz(" CA "));
				for(int j = 1; j <=nres; j++){
					if(pose.residue(j).is_protein()){
						Vector const CA_j(pose.residue(j).xyz(" CA "));
						Real const CA_dist = (CA_i - CA_j).length();
						if(CA_dist < CA_cutoff){
							ConstraintCOP cst(new AtomPairConstraint( core::id::AtomID(pose.residue(i).atom_index(" CA "),i),
																												core::id::AtomID(pose.residue(j).atom_index(" CA "),j),
																												new HarmonicFunc(CA_dist, cst_tol)));
							pose.add_constraint(cst);
						}
					}
				}
			}
		}
	}

	sfxn->set_weight(atom_pair_constraint, CONSTRAINT_WEIGHT);

}

void
ddGMover::minimize_with_constraints(pose::Pose & pose, ScoreFunctionOP s){
	using namespace core::options::OptionKeys;
	using namespace core::options;

	core::optimization::AtomTreeMinimizer min_struc;
	float const minimizer_tol = 0.00001;
	core::optimization::MinimizerOptions options( "dfpmin_armijo_nonmonotone", minimizer_tol,
																								true /*use_nb_list*/,
																								false /*deriv_check_in*/,
																								true /*deriv_check_verbose_in*/);
	options.nblist_auto_update( true );
	options.max_iter(5000); //otherwise, they don't seem to converge
	core::kinematics::MoveMap mm;
	mm.set_bb(true);
	mm.set_chi(true);

	if(options::option[ddg::sc_min_only]()){
		mm.set_bb(false);
	}

	/**
	if(options::option[ddg::ramp_repulsive]()){
		//set scorefxn fa_rep to 1/10 of original weight and then minimize
		ScoreFunction one_tenth_orig(*s);
		reduce_fa_rep(0.1,one_tenth_orig);
		//min_struc.run(p,mm,s,options);
		min_struc.run(p,mm,one_tenth_orig,options);

		//then set scorefxn fa_rep to 1/3 of original weight and then minimize
		ScoreFunction one_third_orig(*s);
		reduce_fa_rep(0.33,one_third_orig);
		min_struc.run(p,mm,one_third_orig,options);
		//then set scorefxn fa_rep to original weight and then minimize
	}
	**/
	(*s)(pose);
	s->show(std::cout,pose);
	pose::Pose before(pose);
	min_struc.run(pose,mm,*s,options);
	//(*s)(pose);
	while(std::abs((*s)(pose)-(*s)(before)) > 1 ){
		//std::cout << "running another iteration of minimization. difference is: " << ((*s)(pose)-(*s)(before)) << std::endl;
	  before=pose;
		min_struc.run(pose,mm,*s,options);
	}
	s->show(std::cout, pose);
}

void
ddGMover::neighbor_cutoff(double cutoff){
	nbr_cutoff_ = cutoff;
}

void
ddGMover::restrict_to_nbrs(bool truefalse){
	restrict_to_nbrhood_ = truefalse;
}

void
ddGMover::set_minimization_score_function( core::scoring::ScoreFunctionOP s){
	min_cst_sfxn_=s;
}

void
ddGMover::score_function( core::scoring::ScoreFunctionOP s){
	scorefxn_=*s;
}

void
ddGMover::num_iterations( int num ){
	num_iterations_=num;
	//std::cout << "number of iterations set to " << num_iterations_ << std::endl;//DEBUG
}

void
ddGMover::dump_pdbs(bool truefalse){
	dmp_pdb_=truefalse;
	//std::cout << "dump pdbs set to " << dmp_pdb << std::endl;//DEBUG
}

void
ddGMover::debug_output(bool truefalse){
	dbg_output_=truefalse;
	//std::cout << "debug output set to " << dbg_output << std::endl; //DEBUG
}

void
ddGMover::is_interface_ddg(bool truefalse){
	interface_ddg_=truefalse;
}

void
ddGMover::wt_score_components(ObjexxFCL::FArray2D<double> wsc){
	wt_components_=wsc;
}

void
ddGMover::wt_unbound_score_components(ObjexxFCL::FArray2D<double> wusc){
	wt_unbound_components_=wusc;
}

void
ddGMover::mutant_score_components(ObjexxFCL::FArray2D<double> msc){
	mutant_components_=msc;
}

void
ddGMover::residues_to_mutate(utility::vector1<core::chemical::AA> residues){
		residues_to_mutate_=residues;
		for(unsigned int i =1;i<=residues_to_mutate_.size();i++){
			//std::cout << "at position " << i << " residue to mutate is: " << core::chemical::name_from_aa(residues[i]) << std::endl; //DEBUG
		}
}

void
ddGMover::set_min_cst(bool truefalse){
	min_cst_ = truefalse;
}

void
ddGMover::set_mean(bool truefalse){
	mean_ = truefalse;
}

void
ddGMover::set_min(bool truefalse){
	min_  = truefalse;
}

Real
ddGMover::neighbor_cutoff(){
	return nbr_cutoff_;
}

bool
ddGMover::restrict_to_nbrs(){
	return restrict_to_nbrhood_;
}

core::scoring::ScoreFunctionOP
ddGMover::score_function(){
	return scorefxn_;
}

core::scoring::ScoreFunctionOP
ddGMover::minimization_score_function(){
	return min_cst_sfxn_;
}

int
ddGMover::num_iterations(){return num_iterations_;}

bool
ddGMover::is_interface_ddg(){return interface_ddg_;}

ObjexxFCL::FArray2D<double>
ddGMover::wt_score_components(){
	return wt_components_;
}

ObjexxFCL::FArray2D<double>
ddGMover::wt_unbound_score_components(){
	return wt_unbound_components_;
}

ObjexxFCL::FArray2D<double>
ddGMover::mutant_score_components(){
	return mutant_components_;
}

utility::vector1<core::chemical::AA>
ddGMover::residues_to_mutate(){
	return residues_to_mutate_;
}

utility::vector1<double>
ddGMover::get_wt_min_score_components(){
	//std::cout << "computing minimum energy components" << std::endl;
	utility::vector1<double> wt_min_score_components;
	utility::vector1<double> total_scores;
	//sum all components for each structure
	for(int i = wt_components_.l2(); i <= wt_components_.u2(); i++){
		double total_score = 0.0;
		for(int j = wt_components_.l1(); j <= wt_components_.u1(); j++){
			total_score += wt_components_(j,i);
		}
		//std::cout << "total score computed for wt number " << i << " is " << total_score << std::endl;
		total_scores.push_back(total_score);
	}
	//num_score_components, size_to_expect
	//find minimum index
	int min_index = -1;
	double min_score = 10000;
	for(unsigned int i = 1; i <= total_scores.size(); i++){
		if(total_scores[i] < min_score){
			min_index = i;
			min_score = total_scores[i];
		}
	}
	assert(min_index != -1);
	//std::cout << "min_index is " << min_index << " and corresponds to an energy " << min_score  << std::endl;
	//return score components of min energy structure
	for(int i = wt_components_.l1(); i <= wt_components_.u1(); i++){
		wt_min_score_components.push_back(wt_components_(i,min_index));
	}
	return wt_min_score_components;
}

utility::vector1<double>
ddGMover::get_wt_averaged_score_components(){
	//std::cout << "computing wt averaged score components" << std::endl;
	utility::vector1<double> wt_averaged_score_components;
	if(!interface_ddg_){
		average_score_components(wt_components_,wt_averaged_score_components);
	}else{
		assert((wt_components_.u1()-wt_components_.l1())== (wt_unbound_components_.u1()-wt_unbound_components_.l1()) &&
					 (wt_components_.u2()-wt_components_.l2()) == (wt_unbound_components_.u2()-wt_unbound_components_.l2()));
		ObjexxFCL::FArray2D<double> dG_bound_unbound((wt_components_.u1()-wt_components_.l1()+1),
																				 (wt_components_.u2()-wt_components_.l2()+1),-999.99);
		for(int i =wt_components_.l1();i<=wt_components_.u1();i++){
			for(int j = wt_components_.l2();j<= wt_components_.u2();j++){
				dG_bound_unbound(i,j)=wt_components_(i,j)- wt_unbound_components_(i,j);
			}
		}
		average_score_components(dG_bound_unbound,wt_averaged_score_components);
	}
	return wt_averaged_score_components;
}

utility::vector1<double>
ddGMover::get_mutant_min_score_components(){
	//std::cout << "computing minimum energy score components for mutant" << std::endl;
	utility::vector1<double> mutant_min_score_components;
	utility::vector1<double> total_scores;
	//sum all components for each structure
	for(int i = mutant_components_.l2(); i <= mutant_components_.u2(); i++){
		double total_score = 0.0;
		for(int j = mutant_components_.l1(); j <= mutant_components_.u1(); j++){
			total_score += mutant_components_(j,i);
		}
		//std::cout << "total score computed for mut number " << i << " is " << total_score << std::endl;
		total_scores.push_back(total_score);
	}
	//num_score_components, size_to_expect
	//find minimum index
	int min_index = -1;
	double min_score = 10000;
	for(unsigned int i = 1; i <= total_scores.size(); i++){
		if(total_scores[i] < min_score){
			min_index = i;
			min_score = total_scores[i];
		}
	}
	assert(min_index != -1);
	//std::cout << "MUT min_index is " << min_index << " and corresponds to an energy " << min_score  << std::endl;
	//return score components of min energy structure
	for(int i = mutant_components_.l1(); i <= mutant_components_.u1(); i++){
		mutant_min_score_components.push_back(mutant_components_(i,min_index));
	}
	return mutant_min_score_components;
}

utility::vector1<double>
ddGMover::get_mutant_averaged_score_components(){
	//std::cout << "computing average mutant score components" << std::endl;
	utility::vector1<double> mutant_averaged_score_components;
	if(!interface_ddg_){
		average_score_components(mutant_components_,
														 mutant_averaged_score_components);
	}else{
		assert((mutant_components_.u1()-mutant_components_.l1())== (mutant_unbound_components_.u1()-mutant_unbound_components_.l1()) &&
					 (mutant_components_.u2()-mutant_components_.l2()) == (mutant_unbound_components_.u2()-mutant_unbound_components_.l2()));
		ObjexxFCL::FArray2D<double> dG_bound_unbound((mutant_components_.u1()-mutant_components_.l1()+1),
																				 (mutant_components_.u2()-mutant_components_.l2()+1),-999.99);
		for(int i =mutant_components_.l1();i<=mutant_components_.u1();i++){
			for(int j = mutant_components_.l2();j<= mutant_components_.u2();j++){
				dG_bound_unbound(i,j)=mutant_components_(i,j)- mutant_unbound_components_(i,j);
			}
		}
		average_score_components(dG_bound_unbound,mutant_averaged_score_components);
	}
	return mutant_averaged_score_components;
}

utility::vector1<double>
ddGMover::get_delta_energy_components(){
	utility::vector1<double> wt;
	utility::vector1<double> mut;
	if(mean_){
		wt = this->get_wt_averaged_score_components();
		mut = this->get_mutant_averaged_score_components();
	}else if(min_){
		wt = this->get_wt_min_score_components();
		mut = this->get_mutant_min_score_components();
	}
	utility::vector1<double> delta_energy;
	assert(wt.size() == mut.size());
	for(unsigned int i=1;i<=wt.size();i++){
		delta_energy.push_back(mut[i]-wt[i]);
	}
	return delta_energy;
}

utility::vector1<std::string>
ddGMover::get_scorefunction_header(){
	utility::vector1<std::string> components;
	int score_component=0;
	if(scorefxn_ != 0){
		for( EnergyMap::const_iterator i = (scorefxn_->weights()).begin();
				 i != scorefxn_->weights().end(); i++){
			score_component++;
			if(*i != 0 ){
				components.push_back(name_from_score_type(ScoreType(score_component)));
			}
		}
	}
	return components;
}

std::string
ddGMover::mutation_label(pose::Pose & pose){
	std::string mutation_label="";
	if(residues_to_mutate_.size() != pose.total_residue()){
		return mutation_label; //residues_to_mutate_ hasn't been initialized
	}
	else{
		for(unsigned int i=1; i <= residues_to_mutate_.size(); i++){
			if(residues_to_mutate_[i] != core::chemical::aa_unk){
				std::ostringstream q;
				q << i;
				mutation_label = mutation_label + (pose.residue(i)).name1() + q.str() + core::chemical::oneletter_code_from_aa(residues_to_mutate_[i]);
				//std::cout << "at position " << i << " the mutation label now becomes " << mutation_label << std::endl; //DEBUG
			}
		}
	}
	return mutation_label;
}

double
ddGMover::get_wt_averaged_totals(){
	utility::vector1<double> wt = this->get_wt_averaged_score_components();
	double sum=0;
	for(unsigned int i =1;i<=wt.size();i++){
		sum=sum+wt[i];
	}
	return sum;
}

double
ddGMover::get_wt_min_totals(){
	utility::vector1<double> wt = this->get_wt_min_score_components();
	double sum = 0;
	for(unsigned int i =1;i <= wt.size();i++){
		sum = sum + wt[i];
	}
	return sum;
}


double
ddGMover::get_mutant_averaged_totals(){
	utility::vector1<double> mut = this->get_mutant_averaged_score_components();
	double sum=0;
	for(unsigned int i =1;i<=mut.size();i++){
		sum=sum+mut[i];
	}
	return sum;
}

double
ddGMover::get_mutant_min_totals(){
	utility::vector1<double> mut = this->get_mutant_min_score_components();
	double sum = 0;
	for(unsigned int i =1;i <= mut.size();i++){
		sum = sum + mut[i];
	}
	return sum;
}


double
ddGMover::ddG(){
	if(mean_){
		return this->get_mutant_averaged_totals()-this->get_wt_averaged_totals();
	}else if(min_){
		return this->get_mutant_min_totals()-this->get_wt_min_totals();
	}else{
		std::cerr << "neither mean or min set, reverting to default" << std::endl;
		min_=true; mean_=false;
		return this->get_mutant_min_totals()-this->get_wt_min_totals();
	}
}

bool
ddGMover::is_wt_calc_complete(){
	return is_complete(wt_components_);
}

bool
ddGMover::is_mutant_calc_complete(){
	return is_complete(mutant_components_);
}

bool
ddGMover::is_properly_initialized(pose::Pose & pose){
	bool is_initialized=true;
	//check scorefxn
	//check residues_to_mutate_
	if(residues_to_mutate_.size() != pose.total_residue()){
		is_initialized=false;
	}
	//check num_iterations_
	//check dmp_pdbs
	//check dbg_output
	return is_initialized;

}

bool
ddGMover::get_min_cst(){
	return min_cst_;
}

bool
ddGMover::use_mean(){
	return mean_;
}

bool
ddGMover::use_min(){
	return min_;
}

void
ddGMover::apply(core::pose::Pose & pose)
{
	using namespace pose;
	using namespace scoring;
	using namespace conformation;

	using namespace core::options;
	using namespace core::options::OptionKeys;
	using namespace core::pack::task;

	std::string wt_traj = "wt_traj";
	std::string mutant_traj = "mutant_traj"+this->mutation_label(pose);

	core::conformation::Interface protein_interface(1); // need to define this outside if, to prevent compilation errors
	//check for properly initialized variables

	//initialize output_silent variable. should go as boolean in private class data?
	bool output_silent = false;
	//force output of silent file
	if(options::option[ddg::output_silent].user()){
		output_silent = options::option[ddg::output_silent]();
	}


	if(this->is_properly_initialized(pose)){
		//debug output?
		if(dbg_output_){
			TR << "weights being used: " <<
				scorefxn_->weights() << "\n";
		}

		// initialize the scoring function stuff
		(*scorefxn_)(pose);
		/// Now handled automatically.  scorefxn_->accumulate_residue_total_energies(pose);

		if( interface_ddg_) {
			protein_interface.distance(10.0);
			protein_interface.calculate(pose);
		}

		bool dG_native_calculated = this->is_wt_calc_complete();
		if(!dG_native_calculated &&
			 !(options::option[ddg::mut_only].user() && (options::option[ddg::mut_only]()))){

			if(dbg_output_){
				TR << "dG_wildtype hasn't been calculated yet! " <<
					"starting to calculate dG for native structure\n";}

			//start filehandler for recording the native structure logfile
			std::ofstream record_trajectories;
			if(wt_traj != ""){
				record_trajectories.open(wt_traj.c_str());
			}else{
				TR << "wt_traj is emtpy, not recording trajectories.\n";
			}

			core::io::silent::SilentFileData sfd;
			std::string silentfilename;
			if(options::option[out::file::silent].user()){
				silentfilename = options::option[out::file::silent]();
				sfd.set_filename(silentfilename);
			}else{
				silentfilename = "wt_"+this->mutation_label(pose)+".out";
				sfd.set_filename(silentfilename); //setup in case we dump in form of silent file
			}

			//set fa_max_dis to 9.0
			options::option[ score::fa_max_dis ](9.0);

			//end add in interface specific stuff

			//measure dG of input structure by repacking n times and taking the average score
			//this corresponds to wt bound energies of protein complex
			pack::task::PackerTaskOP repack_native(pack::task::TaskFactory::create_packer_task(pose));
			//add option here to restrict to local region
			if(restrict_to_nbrhood_){
				//find mutation position
				utility::vector1<int> mutations;
				for(unsigned int i=1; i <= residues_to_mutate_.size(); i++){
					if(residues_to_mutate_[i] != core::chemical::aa_unk){
						mutations.push_back(i);
					}
				} //i is mutation position
				//find all neighbors within 8 angstrom
				if(mutations.size() == 0){
					TR << "FATAL ERROR no mutations specified, which are needed to determine neighbors" << std::endl;
					exit(1);
				}

				for(unsigned int i = 1; i <= mutations.size(); i++){
					numeric::xyzVector<double> i_pos;
					if(pose.residue(mutations[i]).name1() == 'G'){
						i_pos = pose.residue(mutations[i]).xyz(" CA ");
					}else{
						i_pos = pose.residue(mutations[i]).xyz(" CB ");
					}
					for(unsigned int j = 1; j <= pose.total_residue(); j++){
						numeric::xyzVector<double> j_pos;
						if(pose.residue(j).name1() == 'G'){
							j_pos = pose.residue(j).xyz(" CA ");
						}else{
							j_pos = pose.residue(j).xyz(" CB ");
						}
						//if c-beta is within x angstrom, set movemap(i) true
						if(i_pos.distance(j_pos) <= nbr_cutoff_){
							//TR << "residue " << j << " is " << i_pos.distance(j_pos) << " away from i " << mutations[i] << " and will be repacked " << std::endl;
							repack_native->nonconst_residue_task(j).restrict_to_repacking();
						}else{
							//TR << "residue " << j << " is " << i_pos.distance(j_pos) << " away from i " << mutations[i] << " and will NOT be repacked " << std::endl;
							repack_native->nonconst_residue_task(j).prevent_repacking();
						}
					}//for all j
				} // for all mutation positions
			}else{
				repack_native->restrict_to_repacking();
			}

			for( Size j =1;j<=pose.total_residue();j++){
				//by default use ex1 and ex2
				repack_native->nonconst_residue_task(j).or_include_current(true);
				repack_native->nonconst_residue_task(j).or_ex1(true);
				repack_native->nonconst_residue_task(j).or_ex2(true);
				if( interface_ddg_) {
					if( ! protein_interface.is_interface( j ) ) {
						repack_native->nonconst_residue_task(j).prevent_repacking();
					}
				}
			}
			utility::vector1< std::pair < Real,std::string > > results;
			utility::vector1<core::pose::PoseOP> poses;
			pose::Pose temporary_pose = pose;
			pack::pack_rotamers_loop(temporary_pose,(*scorefxn_),repack_native,num_iterations_,results,poses);

			for ( Size i = 1; i <= poses.size(); ++i) {

				std::ostringstream q;
				q << i;

				pose::Pose resulting_pose=(*poses[i]);
				if(min_cst_){
					setup_constraints(resulting_pose,min_cst_sfxn_,0.5);
					minimize_with_constraints(resulting_pose,min_cst_sfxn_);
					store_energies(wt_components_, (*min_cst_sfxn_), resulting_pose,i,num_iterations_);
				}
				if(!interface_ddg_ && !min_cst_){
					store_energies(wt_components_, (*scorefxn_), resulting_pose,i,num_iterations_);
					//end store components
				}else if(interface_ddg_){
					assert(resulting_pose.chain(resulting_pose.total_residue())-pose.chain(1) !=0);
					//dGinterface = Gbound-Gunbound
					//double debug_bound_score = (*scorefxn_)(temporary_pose);
					store_energies(wt_components_, (*scorefxn_), resulting_pose,i,num_iterations_);
					pose::Pose temporary_unbound_pose = resulting_pose;
					calculate_interface_unbound_energy(temporary_unbound_pose,scorefxn_,repack_native);
					//store in wt_components_
					store_energies(wt_unbound_components_, (*scorefxn_),temporary_unbound_pose,i,num_iterations_);
					//double debug_unbound_score = (*scorefxn_)(temporary_unbound_pose);
				}
				//store repacked pose
				natives_.push_back(temporary_pose);
				if(dmp_pdb_ && !restrict_to_nbrhood_ && !output_silent){
					std::string dump_repacked_wt = "repacked_wt_round_" + q.str() + ".pdb";
					resulting_pose.dump_pdb(dump_repacked_wt);
				}else if(restrict_to_nbrhood_ && dmp_pdb_ || output_silent){
					//if you optimize the local neighborhood, automatically dumps out in silent file form
					//otherwise there'll be too many files!!!
					core::io::silent::BinaryProteinSilentStruct ss(resulting_pose,"repacked_wt_round_" + q.str());
					sfd.write_silent_struct(ss,silentfilename,false);
				}
			}
		}


		std::string mutation_label= this->mutation_label(pose);
	} //for now can only store one. will be last listed mutation

	//also initialize packertask using this array
	pack::task::PackerTaskOP packer_task(pack::task::TaskFactory::create_packer_task(pose));

	bool contains_mutation=false;

	for(unsigned int i = 1;i<=pose.total_residue();i++){
		packer_task->nonconst_residue_task(i).or_include_current(true);
		if(residues_to_mutate_[i] != core::chemical::aa_unk){
			contains_mutation=true;
			//add option here to restrict to local neighborhood
			utility::vector1<bool> restrict_to_aa(20,false);
			restrict_to_aa[(int)residues_to_mutate_[i]]=true;
			packer_task->nonconst_residue_task(i).restrict_absent_canonical_aas(restrict_to_aa);
			packer_task->nonconst_residue_task(i).or_ex1(true);
			packer_task->nonconst_residue_task(i).or_ex2(true);
		}else{
			if(restrict_to_nbrhood_){
				//TR << "restricting to local mutant neighborhood" << std::endl;
				//find mutation position
				utility::vector1<int> mutations;
				for(unsigned int j=1; j <= residues_to_mutate_.size(); j++){
					if(residues_to_mutate_[j] != core::chemical::aa_unk){
						mutations.push_back(j);
					}
				} //j is mutation position
				//find all neighbors within 8 angstrom
				for(unsigned int ii = 1; ii <= mutations.size(); ii++){
					numeric::xyzVector<double> i_pos;
					if(pose.residue(mutations[ii]).name1() == 'G'){
						i_pos = pose.residue(mutations[ii]).xyz(" CA ");
					}else{
						i_pos = pose.residue(mutations[ii]).xyz(" CB ");
					}
					numeric::xyzVector<double> j_pos;
					if(pose.residue(i).name1() == 'G'){
						j_pos = pose.residue(i).xyz(" CA ");
					}else{
						j_pos = pose.residue(i).xyz(" CB ");
					}
					//if c-beta is within x angstrom, set movemap(i) true
					if(i_pos.distance(j_pos) <= nbr_cutoff_){
						packer_task->nonconst_residue_task(i).restrict_to_repacking();
						//TR << "MUT residue " << i << " is " << i_pos.distance(j_pos) << " away from i " << mutations[ii] << " and will be repacked " << std::endl;
					}else{
						//TR << "MUT residue " << i << " is " << i_pos.distance(j_pos) << " away from i " << mutations[ii] << " and will NOT be repacked " << std::endl;
						packer_task->nonconst_residue_task(i).prevent_repacking();
					}
				} // for all mutation positions
			}else{ //not a mutation position, not restricted by distance to mutation site, so allow all to repack
				packer_task->nonconst_residue_task(i).restrict_to_repacking();
				packer_task->nonconst_residue_task(i).or_ex1(true);
				packer_task->nonconst_residue_task(i).or_ex2(true);
			}
		}
	}
	if( interface_ddg_) { // further restrict non-interface aa's if we're in interface_ddg mode
		for( core::Size i = 1; i<= pose.total_residue(); ++i ) {
			if( ! protein_interface.is_interface( i ) ) {
				packer_task->nonconst_residue_task(i).prevent_repacking();
			}
		}
	}



	if(!contains_mutation){
		return; // no mutation specified, so no work to be done.
	}

	//apply all mutations in packertask to pose
	std::ofstream record_mutant_trajectories;

	bool any_pdb_empty = false;
	struct stat filesize;
	if(!restrict_to_nbrhood_ && !output_silent){
		for(int i =1; i <= num_iterations_; i++){
			std::ostringstream q;
			q << i;
			std::string file_to_check = ("mut_" + this->mutation_label(pose) + "_"
																	 + "round_" + q.str() + ".pdb");
			if(stat(file_to_check.c_str(), &filesize) == 0 && !options::option[ddg::suppress_checkpointing]()){
				//std::cout << "file " << file_to_check << " exists. checking to see if all files non-empty" << std::endl;
				if(filesize.st_size == 0){
					any_pdb_empty = true;
					std::cout << "file " << file_to_check << " is empty! re-doing calculations " << std::endl;
				}
			}else{
				any_pdb_empty = true;
			}
		}
	}else{
		//silentfile specific behavior when local repacks activated
		std::string file_to_check = "mut_"+this->mutation_label(pose)+".out";;
		if(!options::option[ddg::suppress_checkpointing]()){

			std::cout << "checking for decoys in silent file: " << file_to_check << std::endl;
			if(stat(file_to_check.c_str(), &filesize) == 0){
				std::cout << "silent file " << file_to_check << " exists. checking how many tags exist" << std::endl;
				core::io::silent::SilentFileData sfd(file_to_check,false,false,"binary");
				sfd.read_file(file_to_check);
				utility::vector1<std::string> tags = sfd.read_tags_fast(file_to_check);
				if(/**num tags**/ tags.size() < (Size) num_iterations_ ){
					std::cout << "number of tags " << tags.size() << " is less than number of iterations " << num_iterations_ << " re-doing calculations " << std::endl;
					any_pdb_empty = true;
				}
			}
		}else{
			std::cout << "mutant silent file " << file_to_check << " doesn't exist. starting calculations" << std::endl;
			any_pdb_empty=true;
		}
	}
	//if( fh.is_open() /** last pdb exists**/ &&
	//		!any_pdb_empty /** no pdbs are empty**/){
	if(!any_pdb_empty){
	}else{
		TR << "[DEBUG] file does not exist. "<< mutant_traj << ". Creating logfile now.\n";

		record_mutant_trajectories.open( mutant_traj.c_str()  );
		if(dbg_output_){
			record_mutant_trajectories << "beginning logfile" <<std::endl;
		}

		core::io::silent::SilentFileData sfd;
		std::string silentfilename;
		if(options::option[out::file::silent].user()){
			silentfilename = options::option[out::file::silent]();
			sfd.set_filename(silentfilename);
		}else{
			silentfilename = "mut_"+this->mutation_label(pose)+".out";
			sfd.set_filename(silentfilename); //setup in case we dump in form of silent file
		}

		utility::vector1< std::pair< Real,std::string > > results;
		utility::vector1< pose::PoseOP > poses;

		pose::Pose temporary_pose = pose;
		pack::pack_rotamers_loop(temporary_pose,(*scorefxn_),packer_task,num_iterations_,results,poses);

		for(unsigned int i = 1; i <= poses.size(); i++){
			pose::Pose resulting_pose = (*poses[i]);
			if(min_cst_){
				setup_constraints(resulting_pose,min_cst_sfxn_,0.5);
				minimize_with_constraints(resulting_pose,min_cst_sfxn_);
				store_energies(mutant_components_, (*min_cst_sfxn_),resulting_pose, i,num_iterations_);
			}
			Real final_score = -1;
			if(!min_cst_){
				final_score=(*scorefxn_)( resulting_pose ) ;
			}else{
				final_score=(*min_cst_sfxn_)(resulting_pose);
			}
			//store scores
			if(!interface_ddg_ && !min_cst_){
				store_energies(mutant_components_, (*scorefxn_),resulting_pose, i,num_iterations_);
			}else if(interface_ddg_){
				assert(pose.chain(pose.total_residue())-pose.chain(1) !=0);
				//dGinterface = Gbound-Gunbound
				/*double debug_bound = */ (*scorefxn_)(resulting_pose);
				store_energies(mutant_components_, (*scorefxn_), resulting_pose,i,num_iterations_);
				pose::Pose temporary_unbound_pose = resulting_pose;
				calculate_interface_unbound_energy(temporary_unbound_pose,scorefxn_,packer_task);
				//store in wt_components_
				/*double debug_unbound = */ (*scorefxn_)(temporary_unbound_pose);
				store_energies(mutant_unbound_components_, (*scorefxn_),temporary_unbound_pose,i,num_iterations_);
				//TR << "debug unbound " << debug_unbound << " debug bound is " << debug_bound << " so dg is " << (debug_bound-debug_unbound) << std::endl;
			}

			TR <<
				"score after mutation: residue " << this->mutation_label(pose) << " "
				 << final_score << " ";
			if(!min_cst_){
				//TR << resulting_pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
			}else{
				//TR << resulting_pose.energies().total_energies().weighted_string_of( min_cst_sfxn_->weights()) << std::endl;
			}

			//output the mutated pdb
			std::ostringstream q;
			q << i;
			//record_mutant_trajectories << "round " << q.str()
			TR << "round " << q.str()
																 << " mutate " << this->mutation_label(pose)
																 << " "  << (final_score) << std::endl;

			if(dmp_pdb_ && !restrict_to_nbrhood_ && !output_silent){
				std::string output_pdb = "mut_" +
					this->mutation_label(pose) + "_" + "round_" + q.str() + ".pdb";
				resulting_pose.dump_pdb(output_pdb);
			}else if(dmp_pdb_ && restrict_to_nbrhood_ || output_silent){
				core::io::silent::BinaryProteinSilentStruct ss(resulting_pose,"mut_" + this->mutation_label(pose) + "_round_" + q.str());
				sfd.write_silent_struct(ss,silentfilename,false);
			}
			mutants_.push_back(resulting_pose);
		}

		/**
		for(int k = 1; k <= num_iterations_; k++){

			//initialize packer and copy pose
			pose::Pose temporary_pose = pose;

			//debug output
			Real start_score_dG_mutant( (*scorefxn_)( temporary_pose ) ); //debug ek

			if(dbg_output_){
				record_mutant_trajectories << "round: " <<
					k << " score before mutation " <<
					start_score_dG_mutant << ' ' << std::endl;
			}//debug

			//then do the mutation
			if(dbg_output_){
				record_mutant_trajectories <<
					"start packing pose round: " << k <<
					std::endl;
			}

			pack::pack_rotamers(temporary_pose,(*scorefxn_),packer_task);

			if(min_cst_){
				setup_constraints(temporary_pose,min_cst_sfxn_,0.5);
				minimize_with_constraints(temporary_pose,min_cst_sfxn_);
				store_energies(mutant_components_, (*min_cst_sfxn_),temporary_pose, k,num_iterations_);
			}

			if(dbg_output_){
				record_mutant_trajectories <<
					"end packing pose round: " << k <<
					std::endl;
			}
			Real final_score = -1;
			if(!min_cst_){
				final_score=(*scorefxn_)( temporary_pose ) ;
			}else{
				final_score=(*min_cst_sfxn_)(temporary_pose);
			}
			//store scores
			if(!interface_ddg_ && !min_cst_){
				store_energies(mutant_components_, (*scorefxn_),temporary_pose, k,num_iterations_);
			}else if(interface_ddg_){
				assert(pose.chain(pose.total_residue())-pose.chain(1) !=0);
				//dGinterface = Gbound-Gunbound
				double debug_bound = (*scorefxn_)(temporary_pose);
				store_energies(mutant_components_, (*scorefxn_), temporary_pose,k,num_iterations_);
				pose::Pose temporary_unbound_pose = temporary_pose;
				calculate_interface_unbound_energy(temporary_unbound_pose,scorefxn_,packer_task);
				//store in wt_components_
				double debug_unbound = (*scorefxn_)(temporary_unbound_pose);
				store_energies(mutant_unbound_components_, (*scorefxn_),temporary_unbound_pose,k,num_iterations_);
				//TR << "debug unbound " << debug_unbound << " debug bound is " << debug_bound << " so dg is " << (debug_bound-debug_unbound) << std::endl;
			}

			TR <<
				"score after mutation: residue " << this->mutation_label(pose) << " "
				 << final_score << " ";
			if(!min_cst_){
				//TR << temporary_pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
			}else{
				//TR << temporary_pose.energies().total_energies().weighted_string_of( min_cst_sfxn_->weights()) << std::endl;
			}

			//output the mutated pdb
			std::ostringstream q;
			q << k;
			//record_mutant_trajectories << "round " << q.str()
			TR << "round " << q.str()
																 << " mutate " << this->mutation_label(pose)
																 << " "  << (final_score) << std::endl;

			if(dmp_pdb_ && !restrict_to_nbrhood_ && !output_silent){
				std::string output_pdb = "mut_" +
					this->mutation_label(pose) + "_" + "round_" + q.str() + ".pdb";
				temporary_pose.dump_pdb(output_pdb);
			}else if(dmp_pdb_ && restrict_to_nbrhood_ || output_silent){
				core::io::silent::BinaryProteinSilentStruct ss(temporary_pose,"mut_" + this->mutation_label(pose) + "_round_" + q.str());
				sfd.write_silent_struct(ss,silentfilename,false);
			}
			mutants_.push_back(temporary_pose);
		}
		**/


		//record_mutant_trajectories << "mutate " << mutation_label <<
		TR << "mutate " << this->mutation_label(pose) << //DEBUG
			" " << " "  << " wildtype_dG is: "
															 << this->get_wt_averaged_totals() << " and mutant_dG is: "
															 << this->get_mutant_averaged_totals() << " ddG is: " << (this->get_mutant_averaged_totals()-this->get_wt_averaged_totals())
															 << std::endl;
		record_mutant_trajectories.close();

	}
}
}
}




