// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author

// type headers
#include <core/types.hh>

// unit headers
#include <protocols/moves/mc_convergence_checks/Pool_ConvergenceCheck.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <utility/excn/Exceptions.hh>

// package headers
#include <protocols/toolbox/DecoySetEvaluation.hh>
#include <core/io/silent/SilentFileData.hh>
#include <protocols/toolbox/superimpose.hh>
#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/Job.hh>

#include <core/pose/Pose.fwd.hh>
#include <core/scoring/ScoreFunction.fwd.hh>

// utility headers
#include <utility/pointer/ReferenceCount.hh>
// #include "utility/basic_sys_util.h"

// C++ headers
#include <map>
#include <string>

// Forward declarations

namespace protocols {
namespace moves {
namespace mc_convergence_checks {

using namespace ObjexxFCL;
using namespace core;
void Pool_RMSD::fill_pool( std::string const& silent_file ) {
	tags_.clear();
	pool_.clear();

	io::silent::SilentFileData sfd;
	sfd.read_file( silent_file );

	for ( io::silent::SilentFileData::iterator it=sfd.begin(), eit=sfd.end(); it!=eit; ++it ) {
		pose::Pose pose;
		std::string tag = it->decoy_tag();
		it->fill_pose( pose );

		if ( pool_.n_decoys_max() < pool_.n_decoys() + 1 ) {
			pool_.reserve( pool_.n_decoys() + 100 );
		}
		pool_.push_back( pose );
		tags_.push_back( tag );
	}
	pool_.superimpose(); //this centers all structures
}

void Pool_RMSD::evaluate( core::pose::Pose const& fit_pose, std::string& best_decoy, core::Real& best_rmsd ) const {
	FArray1D_double const weights( pool_.n_atoms(), 1.0 );
	FArray2D_double coords( 3, pool_.n_atoms(), 0.0 );
	// 	if ( pool_.n_atoms() != fit_pose.total_residue() ) throw utility::excn::EXCN_BadInput( "the known_structures have wrong number of residues: pool"
// 		+ObjexxFCL::string_of( pool_.n_atoms() )+" pose: "+ObjexxFCL::string_of( fit_pose ) );
	//	std::cerr << "pool: " << pool_.n_atoms() << "  fit_pose " << fit_pose.total_residue() << std::endl;
	runtime_assert( pool_.n_atoms() == fit_pose.total_residue() );
	toolbox::fill_CA_coords( fit_pose, coords );
	FArray1D_double transvec( 3 );
	toolbox::reset_x( pool_.n_atoms(), coords, weights, transvec );//center coordinates

	Real invn( 1.0 / pool_.n_atoms() );
	best_rmsd = 1000000;
	Size best_index( 0 );
	for ( Size i = 1; i <= pool_.n_decoys(); i++ ) {
		toolbox::Matrix R;
		ObjexxFCL::FArray2Dp_double xx2( pool_.coords( i ) );
		toolbox::fit_centered_coords( pool_.n_atoms(), weights, xx2, coords, R );
		Real rmsd( 0 );
		for ( Size n = 1; n <= pool_.n_atoms(); n++) {
			for ( Size d = 1; d<=3; ++d ) {
				rmsd += ( coords( d, n ) -  xx2( d, n ) ) * ( coords( d, n ) - xx2( d, n ) ) * invn;
			}
		}
		rmsd = sqrt( rmsd );
		if ( rmsd <= best_rmsd ) {
			best_index = i;
			best_rmsd = rmsd;
		}
	}

	best_decoy = tags_[ best_index ];
}

bool Pool_ConvergenceCheck::operator() ( core::pose::Pose const & fit_pose, moves::MonteCarlo const&, bool reject ) {
	if ( reject ) return true; //dont bother for these

	core::Real best_rmsd;
	std::string best_decoy;

	rmsd_pool_->evaluate( fit_pose, best_decoy, best_rmsd );

	//store in Job-Object:
	protocols::jd2::JobDistributor::get_instance()->current_job()->add_string_string_pair( "pool_converged_tag", best_decoy );
	protocols::jd2::JobDistributor::get_instance()->current_job()->add_string_real_pair( "pool_converged_rmsd", best_rmsd );
	if ( best_rmsd <= threshold_ ) throw EXCN_Pool_Converged();
	return best_rmsd >= threshold_;
}

///@brief evaluate pose and store values in Silent_Struct
void Pool_Evaluator::apply( core::pose::Pose& fit_pose, std::string, core::io::silent::SilentStruct &pss) const {
	core::Real best_rmsd;
	std::string best_decoy;

	rmsd_pool_->evaluate( fit_pose, best_decoy, best_rmsd );

	pss.add_string_value( name( 1 ), best_decoy );
	pss.add_energy( name( 2 ), best_rmsd );
// //store in Job-Object:
// 	protocols::jd2::JobDistributor::get_instance()->current_job()->add_string_string_pair( "pool_converged_tag", best_decoy );
// 	protocols::jd2::JobDistributor::get_instance()->current_job()->add_string_real_pair( "pool_converged_rmsd", best_rmsd );

}

} // mc_convergence_check
} // moves
} // rosetta
