// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file MultiStateEntity.hh
/// @brief
/// @author Colin A. Smith

#ifndef INCLUDED_protocols_multistate_design_MultiStateEntity_HH
#define INCLUDED_protocols_multistate_design_MultiStateEntity_HH

#include <protocols/genetic_algorithm/Entity.hh>
#include <protocols/multistate_design/SingleStateEntityData.hh>

#include <core/types.hh>
#include <core/util/MetricValue.hh>

// Utility Headers
#include <utility/pointer/ReferenceCount.hh>

#include <map>
#include <utility>

namespace protocols {
namespace multistate_design {

template <typename T>
class MultiStateEntity : public protocols::genetic_algorithm::Entity<T> {

public:
	typedef typename protocols::genetic_algorithm::Entity<T>::OP EntityOP;

	MultiStateEntity() : protocols::genetic_algorithm::Entity<T>() {}
	MultiStateEntity( MultiStateEntity const & entity );

	virtual EntityOP clone() const { return new MultiStateEntity(*this); }

	virtual void show( std::ostream & os ) const
	{
		os << "MultiStateEntity with traits:";
		utility::vector1<T> const & seq( this->traits() );
		for ( typename utility::vector1<T>::const_iterator it( seq.begin() ), end( seq.end() );
					it != end; ++it ) {
			os << " " << *it;
		}
		os << " and fitness " << fmt::F(6,3,this->fitness());
		for (core::Size i = 1; i <= single_state_entity_data_.size(); ++i) {
			os << '\n' << " SingleState " << i << " with fitness: " << single_state_entity_data_[i].fitness();
		}
	}

	virtual void write_checkpoint( std::ostream & os ) const;
	virtual bool read_checkpoint( std::istream & is );

	utility::vector1<SingleStateEntityData> const & single_state_entity_data() const { return single_state_entity_data_; }
	utility::vector1<SingleStateEntityData> & single_state_entity_data() { return single_state_entity_data_; }

private:
	utility::vector1<SingleStateEntityData> single_state_entity_data_;
};

template <typename T>
MultiStateEntity<T>::MultiStateEntity( MultiStateEntity const & entity ) :
	protocols::genetic_algorithm::Entity<T>(entity),
	single_state_entity_data_(entity.single_state_entity_data())
{}

template <typename T>
void
MultiStateEntity<T>::write_checkpoint(
	std::ostream & os
) const
{
	protocols::genetic_algorithm::Entity<T>::write_checkpoint(os);

	os << " states " << single_state_entity_data_.size();
	for (core::Size i = 1; i <= single_state_entity_data_.size(); ++i) {
		os << "\n ";
		single_state_entity_data_[i].write_checkpoint(os);
	}
}

template <typename T>
bool
MultiStateEntity<T>::read_checkpoint(
	std::istream & is
)
{
	if (!protocols::genetic_algorithm::Entity<T>::read_checkpoint(is)) return false;

	std::string word;
	if (!(is >> word)) return false;
	if (word != "states") return false;

	core::Size num_states;
	if (!(is >> num_states)) return false;
	single_state_entity_data_.resize(num_states);

	for (core::Size i = 1; i <= single_state_entity_data_.size(); ++i) {
		if (!single_state_entity_data_[i].read_checkpoint(is)) return false;
	}

	return true;
}

} // namespace multistate_design
} // namespace protocols

#endif
