// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file MultiStateFitnessFunction.hh
/// @brief
/// @author ashworth

#ifndef INCLUDED_protocols_multistate_design_MultiStateFitnessFunction_HH
#define INCLUDED_protocols_multistate_design_MultiStateFitnessFunction_HH

#include <protocols/genetic_algorithm/FitnessFunction.hh>
#include <protocols/multistate_design/SingleState.hh>
#include <protocols/multistate_design/MultiStateEntity.hh>

#include <protocols/viewer/viewers.hh>

#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.fwd.hh>

#include <protocols/multistate_design/MultiStateAggregateFunction.hh>
#include <protocols/toolbox/PoseMetricCalculators/MetricValueGetter.hh>

#include <core/types.hh>
#include <utility/vector1.hh>
#include <utility/pointer/owning_ptr.hh>

namespace protocols {
namespace multistate_design {

template <typename T>
class MultiStateFitnessFunction : public genetic_algorithm::FitnessFunction<T> {

public:
	typedef utility::pointer::owning_ptr< MultiStateFitnessFunction<T> > OP;

	MultiStateFitnessFunction()
		: genetic_algorithm::FitnessFunction<T>(),
			aggregate_function_(NULL),
			scorefxn_(0),
			best_fitness_(0.) {}

	virtual ~MultiStateFitnessFunction(){}

	virtual void add_state( SingleStateOP state );
	virtual void add_state( core::pose::Pose const & pose, bool is_positive )
		{ add_state( new SingleState( pose, is_positive ) ); }

	virtual core::Real evaluate( protocols::genetic_algorithm::Entity<T> & entity );
	virtual core::Real evaluate( protocols::genetic_algorithm::Entity<T> & entity, core::Size single_state_num ) = 0;
	virtual core::Real evaluate_positive_states( protocols::genetic_algorithm::Entity<T> & entity );

	virtual void set_aggregate_function( typename MultiStateAggregateFunction<T>::COP aggregate_function )
	{ aggregate_function_ = aggregate_function; }
	virtual typename MultiStateAggregateFunction<T>::COP aggregate_function() const { return aggregate_function_; }

	virtual void set_scorefxn( core::scoring::ScoreFunctionCOP sf ) { scorefxn_ = sf; }
	virtual core::scoring::ScoreFunctionCOP scorefxn() const { return scorefxn_; }

	///@brief true const (read only) access to states
	virtual SingleStateCOPs const_states( bool positive_only = false ) const;
	virtual SingleStateCOPs positive_states() const { return const_states( true ); }
	virtual core::Size num_states() const { return states_.size(); }
	virtual core::Size num_states( bool pos_neg ) const;
	virtual core::Size num_positive_states() const { return num_states( true ); }
	virtual core::Size num_negative_states() const { return num_states( false ); }

	virtual void add_metric_value_getter(
		std::string const & name,
		protocols::toolbox::PoseMetricCalculators::MetricValueGetter const & metric_value_getter
	);

protected:
	typedef std::map<std::string, protocols::toolbox::PoseMetricCalculators::MetricValueGetter> MetricValueGetterMap;

	virtual SingleStateOPs & states() { return states_; }
	MetricValueGetterMap const & metric_value_getters() const { return metric_value_getters_; }

private:
	SingleStateOPs states_;
	typename MultiStateAggregateFunction<T>::COP aggregate_function_;
	core::scoring::ScoreFunctionCOP scorefxn_;
	// pose(s) to track most fit trait set in real time (such as for graphics)
	utility::vector1< core::pose::PoseOP > best_entity_positive_states_;
	core::Real best_fitness_;
	MetricValueGetterMap metric_value_getters_;
};

////////////////////////////////////////////////////////////////////////////////////////////////////
// method definitions (must be defined in same file for templates)

template <typename T>
void
MultiStateFitnessFunction<T>::add_state( SingleStateOP state )
{
	states_.push_back( state );
	// for real-time pose tracking of best trait set vs. positive state pose(s) (graphics)
	if ( state->is_positive_state() ) {
		core::pose::PoseOP pose = new core::pose::Pose;
		*pose = state->pose();
// ja this is annoying during iterative protocols because there is no way(?) to close old viewers
//		protocols::viewer::add_conformation_viewer( pose->conformation(), "Best fitness" );
		best_entity_positive_states_.push_back( pose );
	}
}

template <typename T>
core::Real
MultiStateFitnessFunction<T>::evaluate( protocols::genetic_algorithm::Entity<T> & entity )
{
	runtime_assert(aggregate_function_);

	protocols::multistate_design::MultiStateEntity<T>* multi_state_entity =
		dynamic_cast< protocols::multistate_design::MultiStateEntity<T>* >( &entity );
	if (multi_state_entity) multi_state_entity->single_state_entity_data().resize(states().size());

	utility::vector1<core::Real> single_state_fitnesses(states_.size());
	for (core::Size i = 1; i <= states_.size(); ++i) {
		single_state_fitnesses[i] += evaluate(entity, i);
	}

	core::Real fitness = aggregate_function_->evaluate(single_state_fitnesses, *this);
	entity.set_fitness(fitness);

	if ( fitness < best_fitness_ ) {
		best_fitness_ = fitness;
		// real-time pose tracking of best traits vs. positive state pose(s) (graphics)
		utility::vector1< core::pose::PoseOP >::iterator pose( best_entity_positive_states_.begin() );
		for ( SingleStateOPs::iterator s( states_.begin() ), end( states_.end() ); s != end; ++s ) {
			if ( (*s)->is_positive_state() ) {
				**pose = (*s)->pose();
				++pose;
			}
		}
	}

	return fitness;
}

template <typename T>
core::Real
MultiStateFitnessFunction<T>::evaluate_positive_states( protocols::genetic_algorithm::Entity<T> & entity )
{
	core::Real fitness(0.);
	for ( SingleStateOPs::iterator s( states_.begin() ), end( states_.end() ); s != end; ++s ) {
		if ( !(*s)->is_positive_state() ) continue;
		fitness += evaluate( entity, *s );
	}
	return fitness;
}

///@brief true const (read only) access to states
template <typename T>
SingleStateCOPs
MultiStateFitnessFunction<T>::const_states( bool positive_only /* = false */ ) const
{
	SingleStateCOPs const_states;
	for ( SingleStateOPs::const_iterator s( states_.begin() ), end( states_.end() );
				s != end; ++s ) {
		if ( positive_only && !(*s)->is_positive_state() ) continue;
		const_states.push_back( *s );
	}
	return const_states;
}

template <typename T>
core::Size
MultiStateFitnessFunction<T>::num_states( bool pos_neg ) const
{
	core::Size n(0);
	for ( SingleStateOPs::const_iterator s( states_.begin() ), end( states_.end() );
				s != end; ++s ) {
		if ( (*s)->is_positive_state() != pos_neg ) continue;
		++n;
	}
	return n;
}

template <typename T>
void
MultiStateFitnessFunction<T>::add_metric_value_getter(
	std::string const & name,
	protocols::toolbox::PoseMetricCalculators::MetricValueGetter const & metric_value_getter
)
{
	metric_value_getters_[name] = metric_value_getter;
}

} // namespace multistate_design
} // namespace protocols

#endif
