// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file MultiStatePacker.hh
/// @brief
/// @author ashworth

#include <protocols/multistate_design/MultiStatePacker.hh>
#include <protocols/multistate_design/PackingState.hh>
#include <protocols/multistate_design/MultiStateEntity.hh>

#include <core/conformation/Residue.hh>
#include <core/chemical/ResidueType.hh>
#include <core/pose/Pose.hh>
#include <core/pack/rotamer_set/RotamerSets.hh>
#include <core/scoring/ScoreFunction.hh>

#include <utility/string_util.hh>
using utility::string_split;

#include <utility/vector1.hh>
using utility::vector1;

#include <core/util/Tracer.hh>
using core::util::t_info;
using core::util::t_debug;
using core::util::t_trace;
static core::util::Tracer TR("protocols.multistate_design.MultiStatePacker",t_info);

#include <ObjexxFCL/formatted.o.hh>
using namespace ObjexxFCL::fmt;

#include <boost/functional/hash.hpp>

#include <cmath> // std::exp
#include <vector>
#include <iostream>

namespace protocols {
namespace multistate_design {

using core::Size;
using core::Real;

PosType::PosType() : index_(0), type_(core::chemical::aa_unk) {}
PosType::~PosType(){}
PosType::PosType( Size index, core::chemical::AA type ) : index_(index), type_(type) {}
PosType::PosType( std::string const & word ) : index_(0), type_( core::chemical::aa_unk )
{
	vector1< std::string > split( string_split( word, '.' ) );
	if ( split.size() != 2 ) utility_exit_with_message( "invalid string " + word );
	std::istringstream ss( split.front() );
	ss >> index_;
	type_ = core::chemical::aa_from_name( split.back() );
}

Size PosType::index() const { return index_; }
core::chemical::AA PosType::type() const { return type_; }

bool PosType::operator < ( PosType const & other ) const
{
	if ( index_ < other.index() ) return true;
	if ( index_ > other.index() ) return false;
	// indices ==
	if ( type_ < other.type() ) return true;
	if ( type_ > other.type() ) return false;
	// == -> false (strict)
	return false;
}

bool PosType::operator == ( PosType const & other ) const
{
	return ( index_ == other.index() && type_ == other.type() );
}

std::ostream & operator << ( std::ostream & os, PosType const & pt )
{
	os << pt.index() << '.' << name_from_aa( pt.type() );
	return os;
}

std::size_t hash_value( PosType const & pt )
{
	return boost::hash_value( pt.type() );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
void
MultiStatePacker::single_state_design( bool restrict_to_canonical /* = true */ )
{
	for ( SingleStateOPs::iterator ss( states().begin() ), end( states().end() );
	      ss != end; ++ss ) {
		PackingStateOP state = dynamic_cast< PackingState* >( (*ss)() );
		runtime_assert( state );
		std::vector< int > rot_to_pack;
		// this is important if alternate states are represented in the rotamer set (e.g. for DNA)
		if ( restrict_to_canonical ) restrict_to_canonical_aas( *state, rot_to_pack );
		state->run_packer( rot_to_pack );
		( *scorefxn() )( state->nonconst_pose() );
		Real const score( state->fitness_function()->evaluate(state->nonconst_pose()) );
		state->set_best_score( score );
		TR(t_info) << "Best single-state design score: " << F(8,2,score) << std::endl;
	}
}

Real
MultiStatePacker::evaluate( protocols::genetic_algorithm::Entity<PosType> & entity, core::Size single_state_num )
{
	PackingStateOP state = dynamic_cast< PackingState* >( states()[single_state_num]() );
	runtime_assert( state );
	protocols::multistate_design::MultiStateEntity<PosType>* multi_state_entity =
		dynamic_cast< protocols::multistate_design::MultiStateEntity<PosType>* >( &entity );

	// Filter down to the rotamers needed for this single sequence
	std::vector<int> rot_to_pack;
	limit_rotamer_set( rot_to_pack, *state, entity.traits() );

	// optionally pack multiple times to find best energy
	Real E(0.), bestE(0.);
	for ( Size i(0); i < num_packs_; ++i ) {

		state->run_packer( rot_to_pack );

		( *scorefxn() )( state->nonconst_pose() );
		E = state->fitness_function()->evaluate(state->nonconst_pose());
		if ( E < bestE || i == 0 ) {
			bestE = E;
			if (multi_state_entity) {
				multi_state_entity->single_state_entity_data()[single_state_num].fitness(E);
				for (MetricValueGetterMap::const_iterator iter = metric_value_getters().begin();
				     iter != metric_value_getters().end(); ++iter) {
					multi_state_entity->single_state_entity_data()[single_state_num].metric_value(
						iter->first,
						iter->second.get(state->pose())
					);
				}
			}
		}
	}
	return bestE;
}

void
limit_rotamer_set(
  std::vector< int > & rot_to_pack,
  PackingState const & state,
  PosTypes const & seq
)
{
	rot_to_pack.clear();

	// Allocate enough to accomodate full rotamer set
	core::pack::rotamer_set::RotamerSets const & rotsets( *state.rotamersets() );
	Size const nrotamers( rotsets.nrotamers() );

	for ( Size rot_i(1); rot_i <= nrotamers; ++rot_i ) {

		Size const rot_pos( rotsets.res_for_rotamer( rot_i ) );
		core::chemical::ResidueTypeCAP rot_type( rotsets.rotamer( rot_i )->type() );

		core::chemical::AA seq_type( core::chemical::aa_unk );
		for ( vector1< PosType >::const_iterator it( seq.begin() ), end( seq.end() );
					it != end; ++it ) {
			if ( it->index() == rot_pos ) { seq_type = it->type(); break; }
		}

		if ( seq_type == core::chemical::aa_unk ) {
			// this is not a position whose mutation is controlled by genetic algorithm,
			// and should just repack for this state--allow the rotamer if it is not a mutation
			if ( rot_type->aa() != state.pose().residue_type( rot_pos ).aa() ) continue;
			rot_to_pack.push_back( rot_i );
		} else {
			// this is not a position whose mutation is controlled by genetic algorithm:
				// accept this rotamer only if its identity matches that which is specified in seq
			if ( rot_type->aa() == seq_type ) rot_to_pack.push_back( rot_i );
		}
	}
}

void
restrict_to_canonical_aas(
	PackingState const & state,
	std::vector< int > & rot_to_pack
)
{
	rot_to_pack.clear();
	core::pack::rotamer_set::RotamerSets const & rotsets( *state.rotamersets() );
	Size const nrot( rotsets.nrotamers() );

	for ( Size i(1); i <= nrot; ++i ) {
		core::conformation::Residue const & rot( *rotsets.rotamer(i) );
		if ( rot.aa() > core::chemical::num_canonical_aas ) {
			// if non-canonical rotamer, restrict to existing residue type at this position
			if ( rot.type().name3() ==
					state.pose().residue( rotsets.res_for_rotamer(i) ).type().name3() ) {
				rot_to_pack.push_back(i);
			}
		}
		else rot_to_pack.push_back(i);
	}
}

} // namespace multistate_design
} // namespace protocols
