// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file PartitionAggregateFunction.hh
/// @brief
/// @author Colin A. Smith

#ifndef INCLUDED_protocols_multistate_design_PartitionAggregateFunction_HH
#define INCLUDED_protocols_multistate_design_PartitionAggregateFunction_HH

#include <protocols/multistate_design/MultiStateAggregateFunction.hh>

#include <protocols/multistate_design/MultiStateFitnessFunction.fwd.hh>
#include <protocols/multistate_design/MultiStateFitnessFunction.hh>
#include <core/types.hh>
#include <core/util/Tracer.hh>
#include <ObjexxFCL/formatted.o.hh>

// Utility Headers
#include <utility/pointer/ReferenceCount.hh>

namespace protocols {
namespace multistate_design {

template <typename T>
class PartitionAggregateFunction : public MultiStateAggregateFunction<T> {

public:
	typedef utility::pointer::owning_ptr< PartitionAggregateFunction<T> > OP;
	typedef utility::pointer::owning_ptr< PartitionAggregateFunction<T> const > COP;

	PartitionAggregateFunction() : MultiStateAggregateFunction<T>(), temp_(1), anchor_offset_(0) {}
	PartitionAggregateFunction(core::Real temp, core::Real anchor_offset) :
		MultiStateAggregateFunction<T>(), temp_(temp), anchor_offset_(anchor_offset) {}

	virtual ~PartitionAggregateFunction() {}

	virtual core::Real temp() const { return temp_; }
	virtual void set_temp( core::Real temp ) { temp_ = temp; }

	virtual core::Real anchor_offset() const { return anchor_offset_; }
	virtual void set_anchor_offset( core::Real offset ) { anchor_offset_ = offset; }

	virtual
	core::Real
	evaluate(
		utility::vector1<core::Real> const & single_state_fitnesses,
		MultiStateFitnessFunction<T> & fitness_function
	) const;

private:
	core::Real temp_;
	core::Real anchor_offset_;
	static core::util::Tracer TR;
};

template <typename T>
core::Real
PartitionAggregateFunction<T>::evaluate(
	utility::vector1<core::Real> const & single_state_fitnesses,
	MultiStateFitnessFunction<T> & fitness_function
) const
{
	using namespace ObjexxFCL::fmt;

	utility::vector1<SingleStateCOP> single_states(fitness_function.const_states());
	runtime_assert(single_state_fitnesses.size() == single_states.size());

	core::Real const inv_temp( -1.0 / temp_ );
	core::Real normalize(0.), numer(0.), denom(0.);

	for (core::Size i = 1; i <= single_state_fitnesses.size(); ++i) {

		TR(core::util::t_trace) << "State fitness " << F(8,2,single_state_fitnesses[i]);
		// 'normalize' is just a constant value to subtract from energies of large magnitude, in order to take exponents of
		// smaller numbers (exact value used should not be important for the calculations)
		if ( i == 1 ) normalize = single_state_fitnesses[i];
		core::Real const exp_term( std::exp( ( single_state_fitnesses[i] - normalize ) * inv_temp ) );
		TR(core::util::t_trace) << " exp. term " << F(6,2,exp_term);
		denom += exp_term;
		if ( single_states[i]->is_positive_state() ) {
			TR(core::util::t_trace) << " (POSITIVE STATE)";
			numer += exp_term;
			// also add 'affinity anchor(s)' to denominator for each positive state
			core::Real const anchor_term(
				std::exp( ( single_states[i]->best_score() + anchor_offset_ - normalize ) * inv_temp )
			);
			TR(core::util::t_trace) << " anchor exp. term " << F(6,2,anchor_term);
			denom += anchor_term;
		}
		TR(core::util::t_trace) << std::endl;
	}
	TR(core::util::t_trace) << "numer " << F(5,2,numer) << " / denom " << F(5,2,denom) << std::endl;
	// flip sign on the Boltzmann probability for proper ranking elsewhere (more negative is better)
	core::Real const prob_target( numer/denom );
	TR(core::util::t_debug) << "Boltzmann prob. for target state(s) vs. competitor(s): "
	                        << ObjexxFCL::fmt::F(5,2,prob_target) << std::endl;

	return -1.*prob_target; // in genetic algorithm, better fitnesses are more negative
}

template <typename T>
core::util::Tracer
PartitionAggregateFunction<T>::TR("protocols.multistate_design.PartitionAggregateFunction");

} // namespace multistate_design
} // namespace protocols

#endif
