// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_protocols
/// @brief protocols that are specific to relax
/// @detailed
/// @author Mike Tyka, Monica Berrondo

#include <protocols/jobdist/JobDistributors.hh> // keep first
#include <protocols/jobdist/Jobs.hh>

#include <protocols/relax/FastMultiRelax.hh>

#include <protocols/ScoreMap.hh>

#include <core/chemical/util.hh>
#include <core/scoring/rms_util.hh>
#include <protocols/evaluation/RmsdEvaluator.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/ScoringManager.fwd.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/scoring/constraints/util.hh>

#include <core/io/pdb/pose_io.hh>

#include <core/pose/Pose.hh>
#include <core/pose/util.hh>

#include <protocols/moves/ShakeStructureMover.hh>
#include <protocols/moves/RampingMover.hh>
#include <protocols/moves/BackboneMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/MoverContainer.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/JumpOutMover.hh>
#include <protocols/moves/RepeatMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/jumping/Dssp.hh>
#include <protocols/moves/TrialMover.hh>


#include <protocols/abinitio/GunnCost.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/options/option.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/after_opts.hh>
#include <core/fragment/ConstantLengthFragSet.hh>
#include <protocols/moves/WobbleMover.hh>
#include <protocols/checkpoint/Checkpoint.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <protocols/jobdist/standard_mains.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

#ifdef BOINC_GRAPHICS
#include <protocols/boinc/boinc.hh>
#endif

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.relax.ClassicRelax");

using namespace core;
using io::pdb::dump_pdb;
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace protocols {
namespace relax {


FastMultiRelax::FastMultiRelax(
	core::scoring::ScoreFunctionOP scorefxn_in
) :
	parent(),
	scorefxn_(scorefxn_in)
{
}

FastMultiRelax::FastMultiRelax( FastMultiRelax const & other ) :
	parent( other ),
	scorefxn_( other.scorefxn_ ) /// THIS MIMICS THE DEFAULT COPY-CTOR BUT IS ARGUABLY THE WRONG BEHAVIOR.  SHOULD THIS BE A SHALLOW COPY?
{}

FastMultiRelax::~FastMultiRelax() {}

protocols::moves::MoverOP
FastMultiRelax::clone() const
{
	return new FastMultiRelax(*this);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
void FastMultiRelax::apply( core::pose::Pose & pose ) {
	using namespace moves;
	using namespace scoring;

	if( !pose.is_fullatom() ){
		core::chemical::switch_to_residue_type_set(
			pose, core::chemical::FA_STANDARD
		);
		std::cerr << "Fullatom mode .... " << std::endl;
	}

	// initialize secondary structure from DSSP.
	protocols::jumping::Dssp dssp_obj( pose );
	dssp_obj.insert_ss_into_pose( pose );

 	core::kinematics::MoveMapOP movemap = get_movemap()->clone();
	//	movemap->set_bb ( true );
	// 	movemap->set_chi( true );
	/// Call base-class bond-angle and bond-length move-map intialization function
	initialize_movemap( pose, *movemap );

	// minimiser
	moves::MinMoverOP min_mover_;

	core::pack::task::PackerTaskOP task_;
	task_ = pack::task::TaskFactory::create_packer_task( pose );
	task_->initialize_from_command_line().restrict_to_repacking();
	task_->or_include_current( true );
	moves::PackRotamersMoverOP pack_full_repack_ = new moves::PackRotamersMover( scorefxn_, task_ );
	(*scorefxn_)( pose );

	std::vector<pose::Pose> initial_repack_store;
	pose::Pose original_pose;
	original_pose = pose;

	core::Real original_weight = scorefxn_->get_weight( scoring::fa_rep );

	/*core::Real step1_tolerance = 0.025;*/
	core::Real step2_tolerance = 0.0025;
	core::Real step3_tolerance = 0.00025;

	// create soft version of scoring function
	core::scoring::ScoreFunctionOP scorefxn_soft( new ScoreFunction() );
	scorefxn_soft->set_etable( FA_STANDARD_SOFT );
	for ( int i=1; i<= n_score_types; ++i ) {
		if ( scorefxn_->get_weight( ScoreType(i) ) != 0.0 ) {
			scorefxn_soft->set_weight(  ScoreType(i) , scorefxn_->get_weight(  ScoreType(i) ) );
		}
	}
	(*scorefxn_soft)(pose);
	//scorefxn_soft->show( TR, pose );

	/// Repack 1
	int const n_initial_repacks = 5;
	scorefxn_->set_weight    ( scoring::fa_rep , original_weight * 0.08 );
	scorefxn_soft->set_weight( scoring::fa_rep , original_weight * 0.08 * 3 );

	min_mover_ = new moves::MinMover( movemap, scorefxn_soft, "dfpmin_armijo_nonmonotone", step2_tolerance, true );

	for( int i = 0; i < n_initial_repacks ; i++ ){
		pose = original_pose;
		pack_full_repack_->apply( pose );
		min_mover_->apply( pose );
		core::Real score = (*scorefxn_soft)(pose );
		if( score < -10 )
		scorefxn_soft->show( TR, pose );
		//pose.dump_pdb( "stage1" + right_string_of(i,2,'0') + ".pdb" );

		initial_repack_store.push_back( pose );
	}

	for( unsigned int i =  0; i < initial_repack_store.size(); i ++ ){
		core::Real score;
		pose = initial_repack_store[i];
		TR << "---------------------------------------------" << std::endl;

		scorefxn_->set_weight( scoring::fa_rep , original_weight * 0.20 );
		scorefxn_soft->set_weight( scoring::fa_rep , original_weight * 0.50 );
		pack_full_repack_->apply( pose );
		min_mover_ = new moves::MinMover( movemap, scorefxn_soft, "dfpmin_armijo_nonmonotone", step2_tolerance, true );
		min_mover_->apply( pose );
		score = (*scorefxn_soft)(pose );
		//scorefxn_soft->show( TR, pose );

		scorefxn_->set_weight( scoring::fa_rep , original_weight * 0.50 );
		scorefxn_soft->set_weight( scoring::fa_rep , original_weight * 1.0 );
		pack_full_repack_->apply( pose );
		min_mover_ = new moves::MinMover( movemap, scorefxn_soft, "dfpmin_armijo_nonmonotone", step2_tolerance, true );
		min_mover_->apply( pose );
		score = (*scorefxn_soft)(pose );
		//scorefxn_soft->show( TR, pose );

		scorefxn_->set_weight( scoring::fa_rep , original_weight * 1.00 );
		scorefxn_soft->set_weight( scoring::fa_rep , original_weight * 1.00 );
		pack_full_repack_->apply( pose );
		min_mover_ = new moves::MinMover( movemap, scorefxn_, "dfpmin_armijo_nonmonotone", step3_tolerance, true );
		min_mover_->apply( pose );
		score = (*scorefxn_)(pose );
		//scorefxn_->show( TR, pose );
	}
}

}
} // namespace protocols


