// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_protocols
/// @brief protocols that are specific to relax
/// @detailed
/// @author Mike Tyka, Monica Berrondo

#include <protocols/jobdist/JobDistributors.hh> // keep first
#include <protocols/jobdist/Jobs.hh>

#include <protocols/relax/RelaxProtocolBase.hh>
#include <protocols/ScoreMap.hh>

#include <core/chemical/util.hh>
#include <core/scoring/rms_util.hh>
#include <protocols/evaluation/RmsdEvaluator.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/ScoringManager.fwd.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/scoring/constraints/util.hh>

#include <core/io/pdb/pose_io.hh>

#include <core/pose/Pose.hh>
#include <core/pose/util.hh>

#include <protocols/moves/ShakeStructureMover.hh>
#include <protocols/moves/RampingMover.hh>
#include <protocols/moves/BackboneMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/MoverContainer.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/JumpOutMover.hh>
#include <protocols/moves/RepeatMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/jumping/Dssp.hh>
#include <protocols/moves/TrialMover.hh>


#include <protocols/abinitio/GunnCost.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/options/option.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/after_opts.hh>
#include <core/fragment/ConstantLengthFragSet.hh>
#include <protocols/moves/WobbleMover.hh>
#include <protocols/checkpoint/Checkpoint.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <protocols/jobdist/standard_mains.hh>

//*only for debug structures
#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/SilentStruct.hh>
#include <core/io/silent/SilentStructFactory.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/abinitio.OptionKeys.gen.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

#ifdef BOINC_GRAPHICS
#include <protocols/boinc/boinc.hh>
#endif

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.relax.ClassicRelax");

using namespace core;
using io::pdb::dump_pdb;
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace protocols {
namespace relax {


void RelaxProtocolBase::output_debug_structure( core::pose::Pose & pose, std::string prefix ) {
	using namespace core::io::silent;
	using namespace core::options;
	if ( option[ core::options::OptionKeys::out::file::silent ].user() ) {
		std::string silent_file="_"+prefix;
		if ( get_current_job() && get_current_job()->output_file_name() != "" ) {
			silent_file = get_current_job()->output_file_name()+silent_file;
		} else silent_file = "bla"+silent_file;

		SilentFileData sfd;
		//filename might have been changed -- e.g., to also have an MPI rank in there

		//		ProteinSilentStruct pss;
		io::silent::SilentStructOP pss = io::silent::SilentStructFactory::get_silent_struct_out();
		pss->fill_struct( pose, get_current_tag() );

		sfd.write_silent_struct( *pss, silent_file, false /* bWriteScoresOnly */ );
	} // if option[ out::file::silent ].user()
}

RelaxProtocolBase::RelaxProtocolBase( core::scoring::ScoreFunctionOP score_in ) :
	parent(),
	minimize_bond_lengths_( core::options::option[ core::options::OptionKeys::relax::minimize_bond_lengths ] ),
	minimize_bond_angles_( core::options::option[ core::options::OptionKeys::relax::minimize_bond_angles ] ),
	minimize_mainchain_bond_lengths_( core::options::option[ core::options::OptionKeys::relax::minimize_mainchain_bond_lengths ] ),
	minimize_mainchain_bond_angles_( core::options::option[ core::options::OptionKeys::relax::minimize_mainchain_bond_angles ] ),
	scorefxn_( score_in )
{
	set_default_movemap();
}

RelaxProtocolBase::RelaxProtocolBase() :
	parent(),
	minimize_bond_lengths_( core::options::option[ core::options::OptionKeys::relax::minimize_bond_lengths ] ),
	minimize_bond_angles_( core::options::option[ core::options::OptionKeys::relax::minimize_bond_angles ] ),
	minimize_mainchain_bond_lengths_( core::options::option[ core::options::OptionKeys::relax::minimize_mainchain_bond_lengths ] ),
	minimize_mainchain_bond_angles_( core::options::option[ core::options::OptionKeys::relax::minimize_mainchain_bond_angles ] )
{
	set_default_movemap();
}

RelaxProtocolBase::RelaxProtocolBase( std::string const & movername, core::scoring::ScoreFunctionOP score_in ) :
	parent( movername ),
	minimize_bond_lengths_( core::options::option[ core::options::OptionKeys::relax::minimize_bond_lengths ] ),
	minimize_bond_angles_( core::options::option[ core::options::OptionKeys::relax::minimize_bond_angles ] ),
	minimize_mainchain_bond_lengths_( core::options::option[ core::options::OptionKeys::relax::minimize_mainchain_bond_lengths ] ),
	minimize_mainchain_bond_angles_( core::options::option[ core::options::OptionKeys::relax::minimize_mainchain_bond_angles ] ),
	scorefxn_(score_in )
{
	set_default_movemap();
}

RelaxProtocolBase::RelaxProtocolBase( RelaxProtocolBase const & other ) :
	parent( other ),
	minimize_bond_lengths_( other.minimize_bond_lengths_ ),
	minimize_bond_angles_( other.minimize_bond_angles_ ),
	minimize_mainchain_bond_lengths_( other.minimize_mainchain_bond_lengths_ ),
	minimize_mainchain_bond_angles_( other.minimize_mainchain_bond_angles_ ),
	movemap_( other.movemap_ ),
	scorefxn_( other.scorefxn_ )
{
	set_default_movemap();
}

RelaxProtocolBase::~RelaxProtocolBase() {}

RelaxProtocolBase const & RelaxProtocolBase::operator = ( RelaxProtocolBase const & rhs ) {
	if ( this == &rhs ) return *this;

	minimize_bond_lengths_ = rhs.minimize_bond_lengths_;
	minimize_bond_angles_ = rhs.minimize_bond_angles_;
	minimize_mainchain_bond_lengths_ = rhs.minimize_mainchain_bond_lengths_;
	minimize_mainchain_bond_angles_ = rhs.minimize_mainchain_bond_angles_;
	return *this;
}

void RelaxProtocolBase::initialize_movemap(
	core::pose::Pose const & pose,
	core::kinematics::MoveMap & movemap
)
{
	using namespace core::id;
	if ( minimize_bond_lengths_ ) {
		movemap.set( core::id::D, true );
	} else if ( minimize_mainchain_bond_lengths_ ) {
		for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
			core::chemical::AtomIndices const & ii_mainchain_atoms( pose.residue(ii).mainchain_atoms() );
			for ( Size jj = 1; jj <= ii_mainchain_atoms.size(); ++jj ) {
				if ( jj == 1 ) {
					if ( ii > 1 && pose.residue(ii).is_bonded( ii-1 ) && !pose.residue(ii).has_variant_type("CUTPOINT_UPPER")) {
						movemap.set( DOF_ID( AtomID( ii_mainchain_atoms[ jj ], ii ), core::id::D ), true );
					}
				} else {
					movemap.set( DOF_ID( AtomID( ii_mainchain_atoms[ jj ], ii ), core::id::D ), true );
				}
			}
		}
	}

	if ( minimize_bond_angles_ ) {
		movemap.set( core::id::THETA, true );
	} else if ( minimize_mainchain_bond_angles_ ) {
		for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
			core::chemical::AtomIndices const & ii_mainchain_atoms( pose.residue(ii).mainchain_atoms() );
			for ( Size jj = 1; jj <= ii_mainchain_atoms.size(); ++jj ) {
				if ( jj == 1 ) {
					if ( ii > 1 && pose.residue(ii).is_bonded( ii-1 ) && !pose.residue(ii).has_variant_type("CUTPOINT_UPPER")) {
						movemap.set( DOF_ID( AtomID( ii_mainchain_atoms[ jj ], ii ), core::id::THETA ), true );
					}
				} else {
					movemap.set( DOF_ID( AtomID( ii_mainchain_atoms[ jj ], ii ), core::id::THETA ), true );
				}
			}
		}
	}

}

void RelaxProtocolBase::set_scorefxn( core::scoring::ScoreFunctionOP score ) {
	scorefxn_ = score;
}

core::scoring::ScoreFunctionOP RelaxProtocolBase::scorefxn() {
	return scorefxn_;
}

core::scoring::ScoreFunctionCOP RelaxProtocolBase::get_scorefxn() const {
	return scorefxn_;
}


void RelaxProtocolBase::set_default_movemap(){
	movemap_ = new core::kinematics::MoveMap();
	movemap_->set_jump( core::options::option[ core::options::OptionKeys::relax::jump_move ]() );
	movemap_->set_bb( core::options::option[ core::options::OptionKeys::relax::bb_move ]() );
	movemap_->set_chi( core::options::option[ core::options::OptionKeys::relax::chi_move ]() );
}

void RelaxProtocolBase::set_movemap( core::kinematics::MoveMapOP movemap ) {
	movemap_ = movemap;
}

void RelaxProtocolBase::minimize_bond_lengths( bool setting ) { minimize_bond_lengths_ = setting; }
void RelaxProtocolBase::minimize_bond_angles( bool setting ) { minimize_bond_angles_ = setting; }
void RelaxProtocolBase::minimize_mainchain_bond_lengths( bool setting ) { minimize_mainchain_bond_lengths_ = setting; }
void RelaxProtocolBase::minimize_mainchain_bond_angles( bool setting ) { minimize_mainchain_bond_angles_ = setting; }

bool RelaxProtocolBase::minimize_bond_lengths() const { return minimize_bond_lengths_;}
bool RelaxProtocolBase::minimize_bond_angles() const { return minimize_bond_angles_;}
bool RelaxProtocolBase::minimize_mainchain_bond_lengths() const { return minimize_mainchain_bond_lengths_;}
bool RelaxProtocolBase::minimize_mainchain_bond_angles() const { return minimize_mainchain_bond_angles_;}

void RelaxProtocolBase::register_options()
{
	using namespace core::options;
	using namespace OptionKeys;

	static bool runonce( false );
	if ( runonce ) return;

	runonce = true;

	option.add_relevant( OptionKeys::relax::minimize_bond_lengths );
	option.add_relevant( OptionKeys::relax::minimize_bond_angles );
	option.add_relevant( OptionKeys::relax::minimize_mainchain_bond_lengths );
	option.add_relevant( OptionKeys::relax::minimize_mainchain_bond_angles );

}

}
}
 // namespace protocols


