// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_protocols
/// @brief protocols that are specific to relax
/// @detailed
/// @author Mike Tyka, Monica Berrondo

#include <protocols/jobdist/JobDistributors.hh> // keep first
#include <protocols/jobdist/Jobs.hh>

#include <protocols/relax/SequenceRelax.hh>
#include <protocols/ScoreMap.hh>

#include <core/chemical/util.hh>
#include <core/scoring/rms_util.hh>
#include <protocols/evaluation/RmsdEvaluator.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/ScoringManager.fwd.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/scoring/constraints/util.hh>

#include <core/io/pdb/pose_io.hh>

#include <core/pose/Pose.hh>
#include <core/pose/util.hh>

#include <protocols/moves/ShakeStructureMover.hh>
#include <protocols/moves/ShakeStructureMover.fwd.hh>
#include <protocols/moves/RampingMover.hh>
#include <protocols/moves/BackboneMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/MoverContainer.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/JumpOutMover.hh>
#include <protocols/moves/RepeatMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/jumping/Dssp.hh>
#include <protocols/moves/TrialMover.hh>


#include <protocols/abinitio/GunnCost.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/options/option.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/after_opts.hh>
#include <core/fragment/ConstantLengthFragSet.hh>
#include <protocols/moves/WobbleMover.hh>
#include <protocols/checkpoint/Checkpoint.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <protocols/jobdist/standard_mains.hh>

#include <core/kinematics/util.hh>

//symmetry
#include <core/conformation/symmetry/util.hh>
#include <protocols/moves/symmetry/SymShakeStructureMover.hh>
#include <protocols/moves/symmetry/SymShakeStructureMover.fwd.hh>
#include <protocols/moves/symmetry/SymPackRotamersMover.hh>
#include <protocols/moves/symmetry/SymMinMover.hh>
#include <core/options/keys/evaluation.OptionKeys.gen.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>


#ifdef BOINC_GRAPHICS
#include <protocols/boinc/boinc.hh>
#endif

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.relax.SequenceRelax");

using namespace core;
using io::pdb::dump_pdb;
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace protocols {
namespace relax {
////////////////////////////////////////////////////////////////////////////////////////////////////






SequenceRelax::SequenceRelax(
	core::scoring::ScoreFunctionOP scorefxn_in,
	const std::string &          sequence_file
) :
	parent("SequenceRelax", scorefxn_in ),
	checkpoints_("SequenceRelax")
{
	set_to_default();
	read_sequence_file( sequence_file );
}

SequenceRelax::SequenceRelax(	SequenceRelax const & other ) :
	parent( other ),
	checkpoints_( other.checkpoints_ ),
	sequence_( other.sequence_ )
{}

SequenceRelax::~SequenceRelax() {}


protocols::moves::MoverOP
SequenceRelax::clone() const {
	return new SequenceRelax(*this);
}



////////////////////////////////////////////////////////////////////////////////////////////////////
void SequenceRelax::set_to_default( )
{
}

////////////////////////////////////////////////////////////////////////////////////////////////////
void SequenceRelax::apply( core::pose::Pose & pose ){
	using namespace core::options;
	using namespace core::options::OptionKeys;
	TR.Info  << "================== SequenceRelax: " << sequence_.size() << " ===============================" << std::endl;

	if( !pose.is_fullatom() ){
		core::chemical::switch_to_residue_type_set( pose, core::chemical::FA_STANDARD);
		std::cerr << "Fullatom mode .. " << std::endl;
	}
	// not used for some reason.
	//int max_accepts = option[ OptionKeys::relax::sequence_max_accept ]();
	//	pose.dump_pdb("sq_relax_step_"+string_of( 112 )+".pdb" );
	/// SETUP
	core::scoring::ScoreFunctionOP scorefxn( get_scorefxn()->clone() );
	core::Real start_rep_weight = scorefxn->get_weight( scoring::fa_rep );


	core::pack::task::PackerTaskOP task_;
	task_ = pack::task::TaskFactory::create_packer_task( pose ); //!!!!!!!!!!!
	bool const repack = core::options::option[ core::options::OptionKeys::relax::chi_move]();
	utility::vector1<bool> allow_repack( pose.total_residue(), repack);

	if ( !core::options::option[ core::options::OptionKeys::relax::chi_move].user() ) {
		for ( Size pos = 1; pos <= pose.total_residue(); pos++ ) {
			allow_repack[ pos ] = get_movemap()->get_chi( pos );
		}
	}

	task_->initialize_from_command_line().restrict_to_repacking().restrict_to_residues(allow_repack);
	task_->or_include_current( true );
  moves::PackRotamersMoverOP pack_full_repack_ = new moves::PackRotamersMover( scorefxn, task_ );
  if ( core::conformation::symmetry::is_symmetric( pose ) )  {
    pack_full_repack_ = new moves::symmetry::SymPackRotamersMover( scorefxn, task_ );
	}
	(*scorefxn)( pose );

	//	pose.dump_pdb("sq_relax_step_"+string_of( 42 )+".pdb" );
	core::kinematics::MoveMapOP movemap = get_movemap()->clone();
	//	movemap->set_bb ( true );
	//	movemap->set_chi( true );
	initialize_movemap( pose, *movemap );

  // Make sure we only allow symmetrical degrees of freedom to move
  if ( core::conformation::symmetry::is_symmetric( pose )  )  {
    core::conformation::symmetry::make_symmetric_movemap( pose, *movemap );
  }


/// @brief prints something like this ***1***C***1*********2***C********3****C****2********3*****
///                                   **********xxxxxxxxxxxxx************************************
	kinematics::simple_visualize_fold_tree_and_movemap_bb_chi( pose.fold_tree(),  *movemap, TR );

	pose::Pose start_pose=pose;
	pose::Pose best_pose=pose;
	core::Real best_score=100000000.0;
	core::Size accept_count = 0;

	core::Size chk_counter = 0;

	int repeat_step=0;
	int repeat_count=-1;

	core::pose::PoseCOP native_pose = get_native_pose();
	/// RUN


	std::vector< core::Real > best_score_log;
	std::vector< core::Real > curr_score_log;

	//	pose.dump_pdb("sq_relax_step_zero.pdb" );

	if( option[ core::options::OptionKeys::run::dry_run ]() ){
		(*scorefxn)( pose );
		return;
	}

	int total_count=0;
	for ( core::Size ncmd = 0; ncmd < sequence_.size(); ncmd ++ ){
		total_count++;
		int command_starttime = time(NULL);
		if ( TR.Debug.visible() ) scorefxn->show( TR, pose );
		if ( TR.Debug.visible() ) pose.constraint_set()->show_numbers( TR.Debug );

// No MC is used, so update graphics manually
#ifdef BOINC_GRAPHICS
		boinc::Boinc::update_graphics_current( pose );
#endif

		TR << std::endl;
		SequenceCommand cmd = sequence_[ncmd];

		if( cmd.command == "repeat" ){
			repeat_count = (int) cmd.param1;
			repeat_step = ncmd;
		} else

		if( cmd.command == "repeat_A" ){
			repeat_count = option[ OptionKeys::relax::sequence_repeat_A ]();
			repeat_step = ncmd;
		} else

		if( cmd.command == "endrepeat" ){
			TR << "CMD:  Repeat: " << repeat_count << std::endl;
			repeat_count -- ;
			if( repeat_count <= 0 ){}
			else{
				ncmd = repeat_step;
			}
		} else
		if( cmd.command == "dump" ){
			if( cmd.nparams < 1 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			pose.dump_pdb( "dump_" + right_string_of( (int) cmd.param1, 4, '0' ) );
		}	else

		if( cmd.command == "shake" ){
			if( cmd.nparams < 2 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			//THIS DOESN'T HONE THE MOVEMAP YET !!!

			chk_counter++;
			std::string checkpoint_id = "chk" + string_of( chk_counter );
			if (!checkpoints_.recover_checkpoint( pose, get_current_tag(), checkpoint_id, true, true )){
				protocols::moves::ShakeStructureMoverOP ssm = new protocols::moves::ShakeStructureMover;
				if (  conformation::symmetry::is_symmetric( pose ) ) {
					ssm = new protocols::moves::symmetry::SymShakeStructureMover;
				}
//				protocols::moves::ShakeStructureMover ssm; //don't initialize with a scorefunction,
				ssm->set_sc_min(true); //sc min after perturbing backbone
				ssm->set_nrounds( (int)cmd.param1 );
				ssm->set_mc_temperature( cmd.param2 );
				ssm->apply(pose);
				checkpoints_.checkpoint( pose, get_current_tag(), checkpoint_id,  true );
			}

		}	else

		if( cmd.command == "repweight" ){
			if( cmd.nparams < 1 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			scorefxn->set_weight( scoring::fa_rep, start_rep_weight * cmd.param1 );
		}	else

		if( cmd.command == "repack" ){
			//if( cmd.nparams < 0 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			chk_counter++;
			std::string checkpoint_id = "chk" + string_of( chk_counter );
			if (!checkpoints_.recover_checkpoint( pose, get_current_tag(), checkpoint_id, true, true )){
				pack_full_repack_->apply( pose );
				checkpoints_.checkpoint( pose, get_current_tag(), checkpoint_id,  true );
			}
		}	else

		if( cmd.command == "min" ){
			if( cmd.nparams < 1 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }

			chk_counter++;
			std::string checkpoint_id = "chk" + string_of( chk_counter );
			if (!checkpoints_.recover_checkpoint( pose, get_current_tag(), checkpoint_id, true, true )){
				moves::MinMoverOP min_mover = new moves::MinMover( movemap, scorefxn, "dfpmin_armijo_nonmonotone", cmd.param1, true );
				if ( core::conformation::symmetry::is_symmetric( pose ) )  {
					min_mover = new moves::symmetry::SymMinMover( movemap, scorefxn, "dfpmin_armijo_nonmonotone", cmd.param1, true );
				}
				min_mover->apply( pose );
				checkpoints_.checkpoint( pose, get_current_tag(), checkpoint_id,  true );
			}
		}	else


		if( cmd.command == "ramp_repack_min" ){
			if( cmd.nparams < 2 ){ utility_exit_with_message( "More parameters expected after : " + cmd.command  ); }
			scorefxn->set_weight( scoring::fa_rep, start_rep_weight * cmd.param1 );
			chk_counter++;
			std::string checkpoint_id = "chk" + string_of( chk_counter );
			if (!checkpoints_.recover_checkpoint( pose, get_current_tag(), checkpoint_id, true, true )){
				pack_full_repack_->apply( pose );
				moves::MinMoverOP min_mover = new moves::MinMover( movemap, scorefxn, "dfpmin_armijo_nonmonotone", cmd.param2, true );
				if ( core::conformation::symmetry::is_symmetric( pose ) )  {
					min_mover = new moves::symmetry::SymMinMover( movemap, scorefxn, "dfpmin_armijo_nonmonotone", cmd.param2, true );
				}
				min_mover->apply( pose );
				checkpoints_.checkpoint( pose, get_current_tag(), checkpoint_id,  true );
			}
		}	else


		if( cmd.command == "accept_to_best" ){
			// grab the score and remember the pose if the score is better then ever before.
			core::Real score = (*scorefxn)( pose );
			if( ( score < best_score) || (accept_count == 0) ){
				best_score = score;
				best_pose = pose;
			}
#ifdef BOINC_GRAPHICS
		boinc::Boinc::update_graphics_low_energy( best_pose, best_score  );
		boinc::Boinc::update_graphics_last_accepted( pose, score );
		boinc::Boinc::update_mc_trial_info( total_count , "SequenceRelax" );
#endif
			if ( native_pose ) {
				if ( core::conformation::symmetry::is_symmetric( pose ) && option[ core::options::OptionKeys::evaluation::symmetric_rmsd ].user() ) {
						core::Real rms = core::scoring::CA_rmsd_symmetric( *native_pose , best_pose );
						core::Real irms = core::scoring::CA_rmsd_symmetric( start_pose , best_pose );
						TR << "MRP: " << accept_count << "  " << score << "  " << best_score << "  "
						<< rms << "  "
						<< irms << "  "
						<< std::endl;

				} else {
					core::Real rms =  protocols::evaluation::native_CA_rmsd( *native_pose , best_pose );
					core::Real irms =  protocols::evaluation::native_CA_rmsd( start_pose , best_pose );
					TR << "MRP: " << accept_count << "  " << score << "  " << best_score << "  "
						 << rms << "  "
						 << irms << "  "
						 << std::endl;
				}
			}
			best_score_log.push_back( best_score );
			curr_score_log.push_back( score );

			accept_count++;
			if ( static_cast< int > ( accept_count ) >
				  option[ OptionKeys::relax::sequence_max_accept ]()
			)
			{
				break;
			}

			if( 	option[ core::options::OptionKeys::run::test_cycles ]() ||
 		        option[ core::options::OptionKeys::run::dry_run ]()
			){
				break;
			}

		}	else

		if( cmd.command == "load_best" ){
			pose = best_pose;
		}	else
		if( cmd.command == "load_start" ){
			pose = start_pose;
		}	else

		if( cmd.command == "exit" ){
			utility_exit_with_message( "EXIT INVOKED" );
		}	else

		{
			utility_exit_with_message( "Unknown command: " + cmd.command );
		}

		core::Real rms( -1.0 );
		if ( native_pose ) {
			rms =  protocols::evaluation::native_CA_rmsd( *native_pose , pose );
			if ( core::conformation::symmetry::is_symmetric( pose ) &&
					option[ core::options::OptionKeys::evaluation::symmetric_rmsd ].user() ) {
				rms = core::scoring::CA_rmsd_symmetric( *native_pose , best_pose );
			}
		}
		core::Real irms =  protocols::evaluation::native_CA_rmsd( start_pose , pose );
		if ( core::conformation::symmetry::is_symmetric( pose ) &&
					option[ core::options::OptionKeys::evaluation::symmetric_rmsd ].user() ) {
			irms =  protocols::evaluation::native_CA_rmsd( start_pose , best_pose ); //need to make a symmetrical verision?
		}
		int command_endtime = time(NULL);
		TR << "CMD: " <<  cmd.command << "  "
			<< command_endtime - command_starttime << "  "
			<< (*scorefxn)( pose ) << "  "
			<< rms << "  "
			<< irms << "  "
			<< scorefxn->get_weight( scoring::fa_rep	)
			<< std::endl;
		//		pose.dump_pdb("sq_relax_step_"+string_of( ncmd )+".pdb" );
		//output_debug_structure( pose, "step_"+string_of( ncmd ) );
	}

	pose = best_pose;
	(*scorefxn)( pose );

	if(	option[ core::options::OptionKeys::relax::sequence_details ]() ){
		for( Size j = 0; j < best_score_log.size(); j++ )
			core::pose::setPoseExtraScores( pose, "B" + right_string_of(j,3,'0'), best_score_log[j]);

		for( Size j = 0; j < curr_score_log.size(); j ++ )
			core::pose::setPoseExtraScores( pose, "S" + right_string_of(j,3,'0'), curr_score_log[j] );
	}

	checkpoints_.clear_checkpoints();
}


void SequenceRelax::read_sequence_file( const std::string &sequence_file ){
	using namespace ObjexxFCL;
	sequence_.clear();
	std::vector< std::string > filelines;
	std::string line;

	if( sequence_file == "" ){
		TR << "================== Using default sequence ==================" << std::endl;
		filelines.push_back( "repeat 15"                       );
		filelines.push_back( "ramp_repack_min 0.02  0.01"      );
		filelines.push_back( "ramp_repack_min 0.250 0.01"      );
		filelines.push_back( "ramp_repack_min 0.550 0.01"      );
		filelines.push_back( "ramp_repack_min 1     0.00001"   );
		filelines.push_back( "accept_to_best"                  );
		filelines.push_back( "endrepeat "                      );
	}else{
		std::ifstream infile( sequence_file.c_str() );

		TR << "================== Reading sequence file: ==================" << std::endl;
		if (!infile.good()) {
			utility_exit_with_message( "[ERROR] Error opening sequence file '" + sequence_file + "'" );
		}
		while( getline(infile,line) ) {
			filelines.push_back( line );
		}
		infile.close();
	}

	int linecount=0;

	core::Size i;
	for( i =0; i< filelines.size(); i++ ){
		line = filelines[i];
		TR << line << std::endl;
		linecount++;
		std::vector< std::string > tokens ( utility::split( line ) );

		if ( tokens.size() > 0 ) {
			SequenceCommand newcmd;
			newcmd.command = tokens[0];

			if (tokens.size() > 1) {newcmd.param1 = atof(tokens[1].c_str()); newcmd.nparams = 1;}
			if (tokens.size() > 2) {newcmd.param2 = atof(tokens[2].c_str()); newcmd.nparams = 2;}
			if (tokens.size() > 3) {newcmd.param3 = atof(tokens[3].c_str()); newcmd.nparams = 3;}
			if (tokens.size() > 4) {newcmd.param4 = atof(tokens[4].c_str()); newcmd.nparams = 4;}

			sequence_.push_back( newcmd );
		}
	}


}

}

} // namespace protocols


