// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_protocols
/// @brief protocols that are specific to relax
/// @detailed
/// @author Mike Tyka, Monica Berrondo

#include <protocols/jobdist/JobDistributors.hh> // keep first
#include <protocols/jobdist/Jobs.hh>

#include <protocols/relax/SimpleMultiRelax.hh>
#include <protocols/relax/ClassicRelax.hh>
#include <protocols/ScoreMap.hh>

#include <core/chemical/util.hh>
#include <protocols/loops/util.hh>
#include <core/scoring/rms_util.hh>
#include <protocols/evaluation/RmsdEvaluator.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/ScoringManager.fwd.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/scoring/constraints/util.hh>

#include <core/io/pdb/pose_io.hh>

#include <core/pose/Pose.hh>
#include <core/pose/util.hh>

#include <protocols/moves/ShakeStructureMover.hh>
#include <protocols/moves/RampingMover.hh>
#include <protocols/moves/BackboneMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/MoverContainer.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/JumpOutMover.hh>
#include <protocols/moves/RepeatMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/jumping/Dssp.hh>
#include <protocols/moves/TrialMover.hh>


#include <protocols/abinitio/GunnCost.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/options/option.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/loops.OptionKeys.gen.hh>
#include <core/options/after_opts.hh>
#include <core/fragment/ConstantLengthFragSet.hh>
#include <protocols/moves/WobbleMover.hh>
#include <protocols/checkpoint/Checkpoint.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <protocols/jobdist/standard_mains.hh>

// Symmetry
#include <core/conformation/symmetry/util.hh>
#include <protocols/moves/symmetry/SymMinMover.hh>
#include <core/options/keys/symmetry.OptionKeys.gen.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

#ifdef BOINC_GRAPHICS
#include <protocols/boinc/boinc.hh>
#endif

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.relax.SimpleMultiRelax");

using namespace core;
using io::pdb::dump_pdb;
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace protocols {
namespace relax  {

SimpleMultiRelax::SimpleMultiRelax(
	core::scoring::ScoreFunctionOP scorefxn_in
) :
	parent("FastRelax", scorefxn_in),
	checkpoints_("FastRelax")
{
	set_to_default();
}

SimpleMultiRelax::SimpleMultiRelax( SimpleMultiRelax const & other ) :
	parent( other ),
	checkpoints_( other.checkpoints_ ),
	repeats_( other.repeats_ ),
	phase1_cycles_( other.phase1_cycles_ ),
	start_weight_low_( other.start_weight_low_ ),
	start_weight_hi_( other.start_weight_hi_ )
{}

SimpleMultiRelax::~SimpleMultiRelax() {}


protocols::moves::MoverOP
SimpleMultiRelax::clone() const {
	return new SimpleMultiRelax(*this);
}


////////////////////////////////////////////////////////////////////////////////////////////////////
void SimpleMultiRelax::set_to_default()
{
	repeats_         = core::options::option[ core::options::OptionKeys::relax::fastrelax_repeats];
	phase1_cycles_   = core::options::option[ core::options::OptionKeys::relax::fastrelax_rampcycles];
	start_weight_low_= core::options::option[ core::options::OptionKeys::relax::fastrelax_start_weight_low];
	start_weight_hi_ = core::options::option[ core::options::OptionKeys::relax::fastrelax_start_weight_hi];
}

////////////////////////////////////////////////////////////////////////////////////////////////////
///@details registering of options that are relevant for AbrelaxApplication
void SimpleMultiRelax::register_options()
{
	using namespace core::options;
	using namespace OptionKeys;

	parent::register_options();

	option.add_relevant( OptionKeys::relax::fast );
	option.add_relevant( OptionKeys::relax::fastrelax_start_weight_low );
	option.add_relevant( OptionKeys::relax::fastrelax_start_weight_hi );
	option.add_relevant( OptionKeys::relax::fastrelax_repeats );
	option.add_relevant( OptionKeys::relax::fastrelax_rampcycles );
	option.add_relevant( OptionKeys::relax::fastrelax_startover );
	option.add_relevant( OptionKeys::relax::fastrelax_postrelax );
	option.add_relevant( OptionKeys::relax::fastrelax_postrelax_cycles );
	option.add_relevant( OptionKeys::relax::fastrelax_postmin );
	option.add_relevant( OptionKeys::relax::fastrelax_taper );
	protocols::relax::ClassicRelax::register_options();
}

////////////////////////////////////////////////////////////////////////////////////////////////////
void SimpleMultiRelax::apply( core::pose::Pose & pose )
{
	using namespace core::options;
	using namespace core::options::OptionKeys;
	using namespace moves;
	using namespace scoring;

	if( !pose.is_fullatom() ){
		core::chemical::switch_to_residue_type_set( pose, core::chemical::FA_STANDARD);
		std::cerr << "Fullatom mode .. " << std::endl;
	}

	if ( option[ core::options::OptionKeys::run::dry_run ]() ) return;
	//long starttime = time(NULL);

	if( 	option[ core::options::OptionKeys::run::test_cycles ]() ){
		repeats_ = 1;
		phase1_cycles_ = 2;
	}

	TR << "Running FASTRELAX " <<  get_current_tag()  << std::endl;
	pose::Pose best_pose=pose;
	core::Real best_score=100000000.0;
	std::vector < core::Real > score_set; // USE VECTOR0 INSTEAD OF STL VECTOR.  STL VECTOR DOES NOT DO BOUNDS CHECKING!

	// save the original pose
	pose::Pose original_pose;
	original_pose = pose;

	runtime_assert( scorefxn() );

	best_score = (*scorefxn())( pose );
	int repeat = 1;
	for(repeat = 1; repeat <= std::max(1,repeats_); repeat ++ ){

		core::Real pose_score = (*scorefxn())( pose );
		TR << "----- Cycle: " << repeat << " Score: " << pose_score << " BestScore: " << best_score << " ---------------" << std::endl;


		if( option[core::options::OptionKeys::relax::fastrelax_startover ]() > 0 ){
			if( ( repeat % option[core::options::OptionKeys::relax::fastrelax_startover ]() ) == 0 ){
				TR << "Restarting from starting_pose" << std::endl;
				pose = original_pose;
			}
		}

		// it appears to work better when we feed the last result back into the packer at
		// low repulsive weight. This differs somewhat from the original fast repack strategy.
		//pose = original_pose;    // i.e. dont do this.

		// do a quick ramping (misuse the Relax protocol for this purpose by setting the
		// stages 2 and 3 to 0 steps (the beauty of mini).
		// ramp cycles are reduced to 5. (can we get thi seven lower using smooth etable like
		// in rosetta++ ?

		/// Either taper the starting repulsive weight from low to high over the number of outer-loop cycles
		/// or choose the starting repulsive weight randomly between the two starting weights (hi and low)
		/// which by default are both 0.02 (as of 1/3/09)
		core::Real start_weight =
			option[core::options::OptionKeys::relax::fastrelax_taper ] ?
			start_weight_low_ + (0.5 - start_weight_low_) * ( ((Real) repeat) / repeats_ ) :
			start_weight_low_ + numeric::random::uniform() * (start_weight_hi_ - start_weight_low_);

		std::string checkpoint_id = "chk" + string_of( repeat );
		if (!checkpoints_.recover_checkpoint( pose, get_current_tag(), checkpoint_id, true, true )){
			protocols::relax::ClassicRelax myrelax( scorefxn() );
			myrelax.get_checkpoints().set_disabled( true );
			myrelax.set_native_pose( get_native_pose() );
			myrelax.set_current_tag( get_current_tag() );
			myrelax.set_lj_ramp_cycles( phase1_cycles_ );

			myrelax.set_start_rep_weight( start_weight );
			myrelax.set_stage2_cycles( 0 );
			myrelax.set_stage3_cycles( 0 );

			// Make sure we only allow symmetrical degrees of freedom to move
			if ( core::options::option[ core::options::OptionKeys::symmetry::symmetry_definition ].user() )  {
				core::kinematics::MoveMapOP symm_movemap = get_movemap()->clone();
		    core::conformation::symmetry::make_symmetric_movemap( pose, *symm_movemap );
				set_movemap( symm_movemap );
			}

			myrelax.set_movemap( get_movemap() );
			myrelax.apply(pose );

			checkpoints_.checkpoint( pose, get_current_tag(), checkpoint_id,  true );
		}
		// score. just in case. who knows.
		(*scorefxn())(pose);

		// grab the score and remember the pose if the score is better then ever before.
		core::Real score = (*scorefxn())( pose );
		TR << "MRP: " << repeat << "  " << score << "  " << best_score << "  " << start_weight << std::endl;

		if( ( score < best_score) || (repeat == 1) ){
			best_score = score;
			best_pose = pose;
		}


		score_set.push_back( score );

		if( option[core::options::OptionKeys::relax::fastrelax_remove_cst_after ].user()  ){
			if( option[core::options::OptionKeys::relax::fastrelax_remove_cst_after ]() >= repeat ){
				pose.constraint_set( NULL );
				score = (*scorefxn())( pose );
				best_pose.constraint_set( NULL );
				best_score = (*scorefxn())( best_pose );
			}
		}

	}


	pose = best_pose;

	core::Real score_average =0.0;
	core::Real score_sumsqr =0.0;
	for( Size i = 0; i < score_set.size(); i++ ){
		score_average += score_set[i];
		score_sumsqr  += score_set[i]*score_set[i];
	}
	score_average /= Real( score_set.size()) ;


	core::Real score_std = sqrt( std::max(0.0, score_sumsqr / Real( score_set.size())  - score_average * score_average) );

	// add the average score as an extra score to the silent file.. this may not make sense all that much if
	// we keep feeding the result of each repeat into the beginning.
	core::pose::setPoseExtraScores( pose, "Repack_average_score", score_average);
	core::pose::setPoseExtraScores( pose, "Repack_stdev_score", score_std);

	// print the time - this quick protocol should take about a 10th of the average full_atom_relax time
	// if not less.
	core::Real final_score  = best_score;
	if ( option[ OptionKeys::relax::fastrelax_postrelax]() ) {
		TR.Info << "fastrelax_postrelax enabled! " << std::endl;
		protocols::relax::ClassicRelax myrelax( scorefxn() );
		myrelax.set_lj_ramp_cycles( 0 );
		myrelax.set_stage2_cycles( option[ OptionKeys::relax::fastrelax_postrelax_cycles ]() );
		myrelax.set_stage3_cycles( 0 );

		// Make sure we only allow symmetrical degrees of freedom to move
		if ( core::options::option[ core::options::OptionKeys::symmetry::symmetry_definition ].user() )  {
			core::kinematics::MoveMapOP symm_movemap = get_movemap()->clone();
	    core::conformation::symmetry::make_symmetric_movemap( pose, *symm_movemap );
			set_movemap( symm_movemap );
		}

		myrelax.set_movemap( get_movemap() );
		myrelax.apply( pose );
		final_score = (*scorefxn())( pose );
	}

	if ( option[ OptionKeys::relax::fastrelax_postmin].user() ) {
		TR.Info << "fastrelax_postmin  enabled! " << std::endl;
		moves::MinMoverOP min_mover;
 		core::kinematics::MoveMapOP movemap = get_movemap()->clone();
		///		movemap->set_bb ( true );
		// 		movemap->set_chi( true );
		initialize_movemap( pose, *movemap ); //this changes bond-length and bond-angle settings

		// Make sure we only allow symmetrical degrees of freedom to move
		if ( core::options::option[ core::options::OptionKeys::symmetry::symmetry_definition ].user() )  {
	    core::conformation::symmetry::make_symmetric_movemap( pose, *movemap );
			min_mover = new moves::symmetry::SymMinMover( movemap, get_scorefxn(), "dfpmin_armijo_nonmonotone", option[ OptionKeys::relax::fastrelax_postmin](), true );
		} else {
			min_mover = new moves::MinMover( movemap, get_scorefxn(), "dfpmin_armijo_nonmonotone", option[ OptionKeys::relax::fastrelax_postmin](), true );
		}
		min_mover->apply( pose );
		final_score = (*scorefxn())( pose );
	}

	if ( option[ OptionKeys::loops::loopscores].user() ) {
		core::pose::Pose native_pose;
		if ( option[ in::file::native ].user() ) {
			core::io::pdb::pose_from_pdb( native_pose, option[ in::file::native ]() );
			core::pose::set_ss_from_phipsi( native_pose );
		} else	{
			native_pose = pose;
		}
		loops::Loops my_loops;
		my_loops.read_loop_file( option[ OptionKeys::loops::loopscores]() );
		addScoresForLoopParts( pose, my_loops, (*scorefxn()), native_pose, my_loops.size() );
	}

	final_score = (*scorefxn())( pose );
	TR << "FASTRELAX:  AV: " << score_average << " Lowscore: " << best_score << "Final: " << final_score << std::endl;

	checkpoints_.clear_checkpoints();
}
////////////////////////////////////////////////////////////////////////////////////////////////////
void SimpleMultiRelax::setPoseExtraScores( pose::Pose &pose ) {
	using namespace core::options;
	using namespace core::options::OptionKeys;
	if ( option[ OptionKeys::relax::fastrelax_postrelax]() ) {
		ClassicRelax().setPoseExtraScores( pose );
	}
	core::pose::setPoseExtraScores( pose, "Repack_average_score", 0);
	core::pose::setPoseExtraScores( pose, "Repack_stdev_score", 0);
}
////////////////////////////////////////////////////////////////////////////////////////////////////

}
} // namespace protocols


