// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_protocols
/// @brief protocols that are specific to relax
/// @detailed
/// @author Mike Tyka, Monica Berrondo

// #include <protocols/jobdist/JobDistributors.hh> // keep first
// #include <protocols/jobdist/Jobs.hh>

#include <protocols/relax/RelaxProtocolBase.hh>
#include <protocols/relax_protocols.hh>
#include <core/io/pdb/pose_io.hh>
// #include <protocols/ScoreMap.hh>

// #include <core/chemical/util.hh>
// #include <core/scoring/rms_util.hh>
// #include <protocols/evaluation/RmsdEvaluator.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
// #include <core/scoring/constraints/CoordinateConstraint.hh>
// #include <core/scoring/constraints/HarmonicFunc.hh>
// #include <core/scoring/ScoringManager.fwd.hh>
// #include <core/pack/task/PackerTask.hh>
// #include <core/pack/task/TaskFactory.hh>

#include <core/scoring/constraints/util.hh>
#include <core/scoring/electron_density/util.hh>
#include <protocols/moves/symmetry/SetupForSymmetryMover.hh>
#include <protocols/moves/SuperimposeMover.hh>
// #include <core/io/pdb/pose_io.hh>

#include <core/pose/Pose.hh>
//#include <core/pose/util.hh>

// #include <protocols/moves/ShakeStructureMover.hh>
// #include <protocols/moves/RampingMover.hh>
// #include <protocols/moves/BackboneMover.hh>
// #include <protocols/moves/PackRotamersMover.hh>
// #include <protocols/moves/MinMover.hh>
// #include <protocols/moves/MonteCarlo.hh>
// #include <protocols/moves/MoverContainer.hh>
// #include <protocols/moves/RotamerTrialsMover.hh>
// #include <protocols/moves/JumpOutMover.hh>
// #include <protocols/moves/RepeatMover.hh>
// #include <protocols/moves/PackRotamersMover.hh>
// #include <protocols/jumping/Dssp.hh>
// #include <protocols/moves/TrialMover.hh>


// #include <protocols/abinitio/GunnCost.hh>
// #include <core/chemical/ChemicalManager.hh>
#include <core/options/option.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/edensity.OptionKeys.gen.hh>
#include <core/options/keys/symmetry.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/after_opts.hh>
// #include <core/fragment/ConstantLengthFragSet.hh>
// #include <protocols/moves/WobbleMover.hh>
// #include <protocols/checkpoint/Checkpoint.hh>
// #include <utility/file/file_sys_util.hh>
// #include <utility/io/izstream.hh>
// #include <utility/io/ozstream.hh>

// #include <core/scoring/ScoreType.hh>
// #include <core/scoring/ScoreFunctionFactory.hh>


#include <protocols/jd2/JobDistributor.hh>

#include <protocols/moves/MoverContainer.hh>
#include <protocols/electron_density/util.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

#ifdef BOINC_GRAPHICS
#include <protocols/boinc/boinc.hh>
#endif

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer tr("protocols.relax");

using namespace core;
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace protocols {
namespace relax {
////////////////////////////////////////////////////////////////////////////////////////////////////

RelaxProtocolBaseOP generate_relax_from_cmd( bool NULL_if_no_flag ) {

	using namespace protocols::moves;
	using namespace scoring;
	using namespace core::options;
	using namespace core::options::OptionKeys;

	core::scoring::ScoreFunctionOP scorefxn;
	scorefxn = core::scoring::getScoreFunction();
	if( option[ in::file::fullatom ]() )
		core::scoring::constraints::add_fa_constraints_from_cmdline_to_scorefxn( *scorefxn );
	else
		core::scoring::constraints::add_constraints_from_cmdline_to_scorefxn( *scorefxn );

	// now add density scores
	if ( option[ edensity::mapfile ].user() ) {
		core::scoring::electron_density::add_dens_scores_from_cmdline_to_scorefxn( *scorefxn );
	}

	RelaxProtocolBaseOP protocol;
	if ( option[ OptionKeys::relax::fast ]() ) {
		protocol = new SimpleMultiRelax( scorefxn );
	} else if ( option[ OptionKeys::relax::sequence]() ) {
		protocol = new SequenceRelax( scorefxn, option[ OptionKeys::relax::sequence_file ]() );
	} else {
		if ( NULL_if_no_flag ){
			tr.Debug << "no relax protocol specified at command line" << std::endl;
			return NULL;
		}
		protocol = new ClassicRelax ( scorefxn );
	}

	return protocol;
}

int
Relax_main( bool ) {
	using namespace protocols::jobdist;
	using namespace core::options;
	using namespace core::options::OptionKeys;

	protocols::moves::MoverOP protocol = generate_relax_from_cmd();

	// set pose for density scoring if a map was input
	// (potentially) dock map into density
	if ( option[ OptionKeys::edensity::mapfile ].user() ) {
		protocols::moves::SequenceMoverOP seqmov = new protocols::moves::SequenceMover;
		seqmov->add_mover( new protocols::electron_density::SetupForDensityScoringMover );
		seqmov->add_mover( protocol );
		protocol = seqmov;
	}

	// setup symmetry mover ... this should happen _before_ SetupForDensityScoringMover
	//   to avoid adding extra VRTs
	if ( option[ OptionKeys::symmetry::symmetry_definition ].user() )  {
			protocols::moves::SequenceMoverOP seqmov = new protocols::moves::SequenceMover;
	    seqmov->add_mover( new protocols::moves::symmetry::SetupForSymmetryMover );
	    seqmov->add_mover( protocol );
	    protocol = seqmov;
	}

	// superimpose input model to the native structure
	if ( option[ OptionKeys::relax::superimpose_to_native ].user() ) {
		if ( option[ OptionKeys::in::file::native ].user() ) {
				core::pose::Pose native;
				core::io::pdb::pose_from_pdb( native, option[ core::options::OptionKeys::in::file::native ] );
				protocols::moves::SequenceMoverOP seqmov = new protocols::moves::SequenceMover;
				protocols::moves::SuperimposeMover *sm  =  new protocols::moves::SuperimposeMover;
				sm->set_reference_pose( native );
				seqmov->add_mover( sm );
				seqmov->add_mover( protocol );
				protocol = seqmov;
		}
	}

	protocols::jd2::JobDistributor::get_instance()->go( protocol );
	//	universal_main( *protocol );

	return 0;
} // Looprelax_main


void relax_pose( pose::Pose& pose, core::scoring::ScoreFunctionOP scorefxn, std::string const& tag ) {
	using namespace core::options;
	RelaxProtocolBaseOP protocol( generate_relax_from_cmd() );
	protocol->set_current_tag( tag );
	protocol->set_scorefxn( scorefxn );
	protocol->apply( pose );
}


}
}
