// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_protocols
/// @brief protocols that are specific to RNA_LoopCloser
/// @detailed
/// @author Rhiju Das

#include <protocols/rna/RNA_LoopCloser.hh>
#include <core/conformation/Residue.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/rna/RNA_Util.hh>
#include <core/id/AtomID_Map.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/id/AtomID.hh>
#include <core/id/DOF_ID.hh>
#include <core/kinematics/AtomTree.hh>
#include <core/kinematics/DomainMap.hh>
#include <core/pose/Pose.hh>
#include <core/io/pdb/pose_io.hh>

#include <core/options/option.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/options/util.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>

#include <core/types.hh>
#include <core/util/Tracer.hh>

//#include <numeric/random/random.hh>

// External library headers


//C++ headers
#include <vector>
#include <string>
#include <sstream>
#include <fstream>
#include <ctime>

using namespace core;
using core::util::T;

static core::util::Tracer TR( "protocols.rna.rna_loop_closer" ) ;

typedef  numeric::xyzMatrix< Real > Matrix;

namespace protocols {
namespace rna {

RNA_LoopCloser::RNA_LoopCloser():
	verbose_( false ),
	NUM_ROUNDS_( 100 ),
	check_tolerance_( false ),
	ccd_tolerance_( 0.000001 ),
	absolute_ccd_tolerance_( 0.01 ),
	attempt_closure_cutoff_( 20.0 ),
	fast_scan_( false )
{
	Mover::type("RNA_LoopCloser");
}



////////////////////////////////////////////
////////////////////////////////////////////
// HEY NEED TO DEFINE A CLONER?
////////////////////////////////////////////
////////////////////////////////////////////

/// @details  Apply the RNA full atom minimizer.
///
void RNA_LoopCloser::apply( core::pose::Pose & pose )
{
	using namespace core::scoring;
	using namespace core::kinematics;
	using namespace core::optimization;
	using namespace core::options;
	using namespace core::options::OptionKeys;

	// Loop through all residues and look for potential chainbreaks to close --
	// marked by CUTPOINT_LOWER and CUTPOINT_UPPER variants.
	for (Size i = 1; i <= pose.total_residue(); i++ ) {

		if ( !pose.residue( i   ).has_variant_type( chemical::CUTPOINT_LOWER )  ) continue;
		if ( !pose.residue( i+1 ).has_variant_type( chemical::CUTPOINT_UPPER )  ) continue;

		// do_fast_scan will check if this cutpoint has changed since the
		//  last movement, or is the chainbreak is already well closed,
		//  or if the chainbreak is too big to close.
		if ( fast_scan_ && !passes_fast_scan( pose, i ) )  continue;

		apply( pose, i );

	}

}


///////////////////////////////////////////////////////////////////////////////////////////////
Real RNA_LoopCloser::apply( core::pose::Pose & pose, Size const & cutpoint	)
{

	if (verbose_) TR << "Closing loop at: " << cutpoint << std::endl;
	//	TR << "Closing loop at: " << cutpoint << std::endl;

	// In the future might have a separate option, e.g., for kinematic loop closure.
	return rna_ccd_close( pose, cutpoint );
}

////////////////////////////////////////////////////////////////////
// returns false if failure.
bool
RNA_LoopCloser::passes_fast_scan( core::pose::Pose & pose, Size const i ) const
{
	//Don't bother if there hasn't been any movement...
	core::kinematics::DomainMap domain_map;
	pose.conformation().update_domain_map( domain_map );
	if ( domain_map( i ) == domain_map( i+1 ) ) {
		if (verbose_) TR << "Skipping " << i << " due to domain map."  << std::endl;
		return false;
	}

	Real const current_dist_err =   get_dist_err( pose, i );
	//Don't bother if chain is really bad...
	if ( current_dist_err > attempt_closure_cutoff_ )		{
		if (verbose_) TR << "Cutpoint " << i << " will be tough to close: " << current_dist_err << std::endl;
		return false;
	}

	//Don't bother if chain already closed...
	if ( current_dist_err < absolute_ccd_tolerance_ )		{
		if (verbose_) TR << "Cutpoint " << i << " already pretty close to closed: " << current_dist_err << std::endl;
		return false;
	}

	return true;
}



///////////////////////////////////////////////////////////////////////////////////////////////
Real
RNA_LoopCloser::rna_ccd_close( core::pose::Pose & input_pose, Size const & cutpoint ) const
{
	using namespace core::scoring::rna;
	using namespace core::id;

	if ( !input_pose.residue( cutpoint ).is_RNA() ||
			 !input_pose.residue( cutpoint+1 ).is_RNA() ) {
		utility_exit_with_message( "RNA CCD closure at "+string_of( cutpoint )+" but residues are not RNA?");
	}

	if ( !input_pose.residue( cutpoint ).has_variant_type( chemical::CUTPOINT_LOWER ) ||
			 !input_pose.residue( cutpoint+1 ).has_variant_type( chemical::CUTPOINT_UPPER ) ) {
		utility_exit_with_message( "RNA CCD closure at "+string_of( cutpoint )+" but CUTPOINT_LOWER or CUTPOINT_UPPER variants not properly set up." );
	}

	/////////////////////////////////////////////////////////////////////
	//Just to get all the atom tree connectivities right, make a little scratch pose
	// to play with.
	/////////////////////////////////////////////////////////////////////
	// In the future (if refold could be faster?!), we won't need this
	// scratch pose.

	pose::Pose pose;
	pose.append_residue_by_bond( input_pose.residue( cutpoint)   );
	pose.append_residue_by_jump( input_pose.residue( cutpoint+1), 1 );

	kinematics::FoldTree f( 2 );
	f.new_jump( 1, 2, 1 );
	f.set_jump_atoms( 1,
										core::scoring::rna::chi1_torsion_atom( pose.residue(1) ),
										core::scoring::rna::chi1_torsion_atom( pose.residue(2) )   );
	pose.fold_tree(f );

//	pose.dump_pdb( "scratch.pdb" );

	//Vectors of the three atoms upstream or downstream of the cutpoint that need to be matched.
	utility::vector1 <Vector> upstream_xyzs, downstream_xyzs;
	Real mean_dist_err( -1.0 ), mean_dist_err_prev( 9999.9999 );
	mean_dist_err = get_chainbreak_xyz( pose, 1, upstream_xyzs, downstream_xyzs );
	// This get_chainbreak_xyz will get called repeatedly after each torsion change.

	// What torsions are in play? Note that this could be expanded (or contracted) at will.
	utility::vector1< TorsionID > tor_ids, input_tor_ids;

	// epsilon and zeta before the cutpoint.
	for (Size j = 5; j <=6; ++ j){
		tor_ids.push_back( TorsionID( 1 /*cutpoint*/, BB, j ) );
		input_tor_ids.push_back( TorsionID( cutpoint, BB, j ) );
	}
	// alpha, beta, gamma, delta after the cutpoint?
	for (Size j = 1; j <=3; ++ j){
		tor_ids.push_back( TorsionID( 2 /*cutpoint+1*/, BB, j ) );
		input_tor_ids.push_back( TorsionID( cutpoint+1, BB, j ) );
	}
	// Also include delta (sugar pucker!) from second residue if its a free floater (cutpoints before and after)
	if ( input_pose.residue( cutpoint+1 ).has_variant_type( chemical::CUTPOINT_LOWER )  ||
			 input_pose.residue( cutpoint+1 ).has_variant_type( chemical::UPPER_TERMINUS )  ) {
		Size const j = 4;
		tor_ids.push_back( TorsionID( 2 /*cutpoint+1*/, BB, j ) );
		input_tor_ids.push_back( TorsionID( cutpoint+1, BB, j ) );
	}


	//Need to make following user-settable.
	Size nrounds( 0 );

	/////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////

	while ( nrounds++ < NUM_ROUNDS_ ) {

		if (check_tolerance_ && ( (mean_dist_err_prev - mean_dist_err) < ccd_tolerance_ ) ) {
			if (verbose_ ) TR << "Reached tolerance of " << ccd_tolerance_ << " after " << nrounds << " rounds. " << std::endl;
			break;
		}
		mean_dist_err_prev = mean_dist_err;

		/////////////////////////////////////////////////////////////////////
		// One round of chain closure.
		// This doesn't have to be sequential ... allow random traversal,
		// and also biased towards "moveable torsions" (i.e., not beta or epsilon)
		// Also to do? -- check on beta/epsilon -- could perhaps make use of
		//   torsion potential as a simple and general screen.
		for (Size n = 1; n <= tor_ids.size(); n++ )
		{
			TorsionID const & tor_id( tor_ids[ n ] );
			AtomID id1,id2,id3,id4;
			pose.conformation().get_torsion_angle_atom_ids( tor_id, id1, id2, id3, id4 );

			//Is this torsion before or after the chainbreak?
			AtomID my_id;
			Real dir( 0.0 );
			if ( Size( tor_id.rsd() ) == 1 /*cutpoint*/ ){
				my_id = id3;
				dir = 1;
			} else {
				my_id = id2;
				dir = -1;
			}

			core::kinematics::tree::Atom const * current_atom ( & pose.atom_tree().atom( my_id ) );

			kinematics::Stub const & stub_i( current_atom->get_stub() );
			Matrix const & M_i( stub_i.M );
			Vector const & x_i = M_i.col_x();

			Real weighted_sine( 0.0 ), weighted_cosine( 0.0 );
			for (Size m = 1; m <= upstream_xyzs.size(); m++ ){
				Vector const current_xyz( current_atom->xyz() );

				Vector const r1 = upstream_xyzs[m] - current_xyz;
				Vector const rho1 = r1 - dot( r1, x_i) * x_i;

				Vector const r2 = downstream_xyzs[m] - current_xyz;
				Vector const rho2 = r2 - dot( r2, x_i) * x_i;

				Real const current_sine   = dir * dot( x_i, cross( rho1, rho2 ) );
				Real const current_cosine = dot( rho1, rho2 );
				//				std::cout << "PREFERRED ANGLE: " << numeric::conversions::degrees( std::atan2( current_sine, current_cosine) ) << std::endl;

				weighted_sine += current_sine;
				weighted_cosine += current_cosine;

				mean_dist_err = get_chainbreak_xyz( pose, 1 /*cutpoint*/, upstream_xyzs, downstream_xyzs );
		}

			Real const twist_torsion = numeric::conversions::degrees( std::atan2( weighted_sine, weighted_cosine) );

			//			std::cout << "CHECK: " << twist_torsion << std::endl;

			Real const current_val = pose.torsion( tor_id );
			pose.set_torsion( tor_id, current_val + twist_torsion );


		}

		if ( verbose_ ) std::cout << "Distance error: " << mean_dist_err << std::endl;

	}

	if (verbose_) pose.dump_pdb( "scratch_close.pdb" );

	/////////////////////////////////////////////////////////////////////
	// OK, done with mini_pose ... copy torsions back into main pose.
	for (Size n = 1; n <= tor_ids.size(); n++ ) {
		input_pose.set_torsion( input_tor_ids[n], pose.torsion( tor_ids[n] ) );
	}

	return mean_dist_err;

}



///////////////////////////////////////////////////////////////
Real
RNA_LoopCloser::get_dist_err( pose::Pose & pose,
										Size const cutpoint
										) const
{
	utility::vector1< Vector > upstream_xyzs;
	utility::vector1< Vector > downstream_xyzs;
	return get_chainbreak_xyz( pose, cutpoint, upstream_xyzs, downstream_xyzs );
}

///////////////////////////////////////////////////////////////
Real
RNA_LoopCloser::get_chainbreak_xyz( pose::Pose & pose,
										Size const cutpoint,
										utility::vector1< Vector > & upstream_xyzs,
										utility::vector1< Vector > & downstream_xyzs
										) const
{
	upstream_xyzs.clear();
	downstream_xyzs.clear();

	upstream_xyzs.push_back( pose.residue( cutpoint ).xyz( " O3*" ) );
	upstream_xyzs.push_back( pose.residue( cutpoint ).xyz( "OVL1" ) );
	upstream_xyzs.push_back( pose.residue( cutpoint ).xyz( "OVL2" ) );

	downstream_xyzs.push_back( pose.residue( cutpoint+1 ).xyz( "OVU1" ) );
	downstream_xyzs.push_back( pose.residue( cutpoint+1 ).xyz( " P  " ) );
	downstream_xyzs.push_back( pose.residue( cutpoint+1 ).xyz( " O5*" ) );

	Real mean_dist_err( 0.0 );
	for (Size m = 1; m <= upstream_xyzs.size(); m++ ){
		mean_dist_err += ( upstream_xyzs[m]  - downstream_xyzs[m] ).length();
	}
	mean_dist_err /= upstream_xyzs.size();

	return mean_dist_err;
}


} // namespace rna
} // namespace protocols
