// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   protocols/symmetric_docking/SymDockProtocol.cc
///
/// @brief
/// @author Ingemar Andre


#include <protocols/symmetric_docking/SymDockProtocol.hh>

////////////
#include <protocols/ScoreMap.hh>

#include <core/chemical/ChemicalManager.hh>

#include <core/options/option.hh>

#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>
#include <core/util/datacache/BasicDataCache.hh>
#include <core/pose/datacache/CacheableDataType.hh>

#include <core/scoring/rms_util.hh>
#include <core/scoring/symmetry/SymmetricScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/util/datacache/DiagnosticData.hh>

#include <core/types.hh>

#include <protocols/symmetric_docking/SymDockingInitialPerturbation.hh>
#include <protocols/symmetric_docking/SymDockingLowRes.hh>
#include <protocols/symmetric_docking/SymDockingHiRes.hh>
#include <protocols/symmetric_docking/SymRestrictTaskForDocking.hh>

#include <protocols/geometry/RB_geometry.hh>

#include <protocols/moves/Mover.fwd.hh>
#include <protocols/moves/OutputMovers.hh>
#include <protocols/moves/symmetry/SymPackRotamersMover.hh>
#include <protocols/moves/SwitchResidueTypeSetMover.hh>

#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/TaskFactory.fwd.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/operation/NoRepackDisulfides.hh>
#include <core/pack/task/operation/OperateOnCertainResidues.hh>
#include <core/pack/task/operation/ResLvlTaskOperations.hh> // PreventRepackingRLT
#include <core/pack/task/operation/ResFilters.hh> // ResidueLacksProperty

//#include <protocols/moves/ResidueMover.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/moves/ScoreMover.hh>
#include <protocols/moves/ReturnSidechainMover.hh>

#include <protocols/viewer/viewers.hh>
//for resfile reading
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/options/option.hh>

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/cluster.OptionKeys.gen.hh>

#include <utility/file/FileName.hh>
#include <utility/file/file_sys_util.hh>

#include <ObjexxFCL/FArray1D.hh>

#include <core/conformation/symmetry/util.hh>
#include <core/util/Tracer.hh>

namespace protocols {
namespace symmetric_docking {

using namespace core;

static core::util::Tracer TR("protocols.symmetric_docking.SymDockProtocol");

SymDockProtocol::SymDockProtocol():
  SymDockBaseProtocol()
{
  Mover::type( "SymDockProtocol" );

	set_default();

}

SymDockProtocol::~SymDockProtocol() {}

void
SymDockProtocol::set_default()
{

	using namespace core::options;
	using namespace core::scoring;

	// score function setup
//	docking_score_low_ = ScoreFunctionFactory::create_score_function( "interchain_cen" ) ;
//	docking_score_high_ = ScoreFunctionFactory::create_score_function( STANDARD_WTS, DOCK_PATCH ) ;
	fullatom_ = option[ OptionKeys::out::file::fullatom ]();
	local_refine_ = option[ OptionKeys::docking::docking_local_refine ]();

	if (local_refine_) fullatom_=true;
	view_ = option[ OptionKeys::docking::view ]();

 // options
	protocol_ = "standard";
	docking_low_ = new protocols::symmetric_docking::SymDockingLowRes( scorefxn_lowres_ );
	docking_high_ = new protocols::symmetric_docking::SymDockingHiRes( scorefxn_hires_ );

}
/*
void
SymDockProtocol::apply( core::pose::Pose & pose )
{

	using namespace protocols::moves;
	using namespace core::pack;

	assert( core::conformation::symmetry::is_symmetric( pose ));

	protocols::viewer::add_conformation_viewer( pose.conformation(), "start_pose", 450, 450 );

core::scoring::constraints::load_unboundrot(pose); // adds scoring bonuses for the "unbound
																										//" rotamers, if any

  if( protocol_ == "rescore" ) return;

 // Residue movers
  SwitchResidueTypeSetMover to_centroid( core::chemical::CENTROID );
  SwitchResidueTypeSetMover to_all_atom( core::chemical::FA_STANDARD );
//recover sidechains from starting structures
 protocols::moves::ReturnSidechainMover recover_sidechains( pose );

	to_centroid.apply( pose );

	MoverOP initialPerturb = new protocols::symmetric_docking::SymDockingInitialPerturbation(true);
	initialPerturb->apply(pose);
	MoverOP centroid_dock = new protocols::symmetric_docking::SymDockingLowRes( scorefxn_lowres_ );
	centroid_dock->apply(pose);
	assert( core::conformation::symmetry::is_symmetric( pose ));

	to_all_atom.apply(pose);

 recover_sidechains.apply( pose );

	//  Do initial pack over all residues within 1000A of the interface.
	//  Equates to repacking all residues, but doing so without RestrictTaskForDocking invalidates -norepack1 and -norepack2 flags
	task::TaskFactoryOP tf = new core::pack::task::TaskFactory;
	task::PreventRepackingOperation prevent_repacking_non_protein;


	for( core::Size residue_num=1; residue_num<=pose.total_residue(); ++residue_num )
		if( !pose.residue( residue_num ).is_protein() ) prevent_repacking_non_protein.include_residue( residue_num );
			tf->push_back( new task::PreventRepackingOperation( prevent_repacking_non_protein ) );
			tf->push_back( new task::InitializeFromCommandlineOperation );
			tf->push_back( new task::IncludeCurrentOperation );
			tf->push_back( new task::RestrictToRepackingOperation );
			tf->push_back( new protocols::docking::RestrictTaskForDocking( scorefxn_hires_, rb_jump_, true, 1000 ) );

			//reading resfile
			pack::task::ReadResfileOperationOP rrf_ = new task::ReadResfileOperation;
  			if( core::options::option[core::options::OptionKeys::packing::resfile].user() )
 			{
      				rrf_->default_filename();
      				tf->push_back( rrf_ );
      				TR<<"reading resfile ..." <<std::endl;
 			}

			moves::symmetry::SymPackRotamersMover pack( scorefxn_hires_ );
			pack.task_factory( tf );
			pack.apply( pose );

	pose.dump_pdb("intial_pert.pdb");
	to_centroid.apply( pose );

	std::cout << "Sym vs Asym score " << std::endl;
	scorefxn_lowres_->show(std::cout, pose );
	core::pose::Pose asym_pose;
	core::io::pdb::pose_from_pdb(asym_pose, "intial_pert.pdb");
	to_centroid.apply( asym_pose );
	scorefxn_lowres_->show(std::cout, asym_pose );

//	std::exit(0);


	assert( core::conformation::symmetry::is_symmetric( pose ));
	MoverOP hires_dock = new protocols::symmetric_docking::SymDockingHiRes( scorefxn_hires_ );
	hires_dock->apply(pose);

}
*/
void
SymDockProtocol::apply( pose::Pose & pose )
{
	using namespace scoring;
	using namespace options;
	using namespace moves;
	using namespace viewer;
	using namespace core::pack::task;
  using namespace core::pack::task::operation;
	using core::pose::datacache::CacheableDataType::SCORE_MAP;
	using utility::file::FileName;

	assert( core::conformation::symmetry::is_symmetric( pose ));

	core::pose::PoseOP input_pose = new core::pose::Pose();
	*input_pose = pose;
	docking_low_->set_input_pose( input_pose );
	docking_high_->set_input_pose( input_pose );
	set_input_pose( input_pose );

	// Residue movers
	SwitchResidueTypeSetMover to_centroid( core::chemical::CENTROID );
	SwitchResidueTypeSetMover to_all_atom( core::chemical::FA_STANDARD );

	core::scoring::ScoreFunctionOP docking_scorefxn;

	util::prof_reset();

	if ( option[ OptionKeys::run::score_only ]() ) {
		if ( fullatom_ ) {
			core::scoring::ScoreFunctionOP high_score = new core::scoring::symmetry::SymmetricScoreFunction( *scorefxn_hires_ );
			ScoreMover score_and_exit( high_score ) ;
			//score_and_exit.insert_rms( rmsd_no_super_subset( *get_native_pose(), pose, superpos_partner, is_protein_CA ) );
			score_and_exit.insert_rms( calc_symmetric_rms( pose ) );
			score_and_exit.apply( pose );
		} else {
			to_centroid.apply( pose );
			core::scoring::ScoreFunctionOP low_score = new core::scoring::symmetry::SymmetricScoreFunction( *scorefxn_lowres_ );
			ScoreMover score_and_exit( low_score );
			//score_and_exit.insert_rms( rmsd_no_super_subset( *get_native_pose(), pose, superpos_partner, is_protein_CA ) );
			score_and_exit.insert_rms( calc_symmetric_rms( pose ) );
			score_and_exit.apply( pose );
		}
		return;
	}

	if ( view_ ) add_conformation_viewer( pose.conformation(), "start_pose", 450, 450 );

	// TODO
	// some set up functions from pose_docking_standard_protocol
	// nominimize1, nominimize2, chainbreaks, bb_move
	//

	//temporary fix for docking filters
	bool passed_filters = false;
	Size max_repeats = 1000;

	core::pose::Pose starting_pose = pose;

	//start loop of decoy creation until filters are all passed
	for (Size r = 1; r <= max_repeats; r++){
		pose = starting_pose;

		MonteCarloOP mc;
		if ( !local_refine_ ) {
			docking_scorefxn = new core::scoring::symmetry::SymmetricScoreFunction( *scorefxn_lowres_ ) ;
			// first convert to centroid mode

			to_centroid.apply( pose );
			// make starting perturbations based on command-line flags
			SymDockingInitialPerturbation initial( true /*slide into contact*/ );
			initial.apply( pose );
			TR << "finished initial perturbation" << std::endl;

			//score_map_["st_rmsd"] = rmsd_no_super_subset( *get_native_pose(), pose, superpos_partner, is_protein_CA );
			score_map_["st_rmsd"] = calc_symmetric_rms( pose );

			// low resolution docking
			docking_low_->set_default_mc( pose );
			if ( view_ ) {
				/// uses conformation_viewer instead of monte_carlo_viewer so that each step (accepted or rejected)
				/// can be seen.  Also pauses every move....only use for debugging!
				if ( option[ OptionKeys::run::debug ]() ) {
					add_conformation_viewer( pose.conformation(), "start_pose", 450, 450, true );
				}
				mc_ = docking_low_->get_mc();
				//add_monte_carlo_viewer( *mc_, "", 450, 450 );
				add_monte_carlo_silent_viewer( *mc_, "", false );

			}
			TR.Debug << "SymDockingLowRes object created" << std::endl;

			docking_low_->apply( pose );

			//check low res filter to see if it should repeat low res or not
			passed_filters = docking_lowres_filter( pose );

			// add scores to map for output
			ScoreMap::nonzero_energies( score_map_, docking_scorefxn, pose );
			//score_map_["cen_rms"] = rmsd_no_super_subset( *get_native_pose(), pose, superpos_partner, is_protein_CA );
			score_map_["cen_rms"] = calc_symmetric_rms( pose );
		} else {
			passed_filters = true;
			}

		// only do this is full atom is true
		if ( fullatom_ && passed_filters ) {

			//(*scorefxn_lowres_)(pose);

			docking_scorefxn =  new core::scoring::symmetry::SymmetricScoreFunction( *scorefxn_hires_ ) ;
			if (!pose.is_fullatom()) to_all_atom.apply( pose );
			// to_all_atom.apply( pose );

			//recover sidechains from starting structures
			protocols::moves::ReturnSidechainMover recover_sidechains( *get_input_pose() );
			recover_sidechains.apply( pose );

			(*docking_scorefxn)( pose );
			docking_high_->set_default_mc( pose );

			if ( view_ ) {
				mc_ = docking_high_->get_mc();
				//add_monte_carlo_viewer( *mc_, "", 450, 450 );
				add_monte_carlo_silent_viewer( *mc_, "", true );
			}

			TR.Debug << "DockingHighRes object created" << std::endl;

			// TODO Add rtmin!!!  Need to check if this has been incorporated into the packer
			TR << "Doing initial repack of sidechains" << std::endl;
			(*docking_scorefxn)( pose );
			/// Now handled automatically.  docking_scorefxn->accumulate_residue_total_energies( pose );

			//  Do initial pack over all residues within 1000A of the interface.
			//  Equates to repacking all residues, but doing so without RestrictTaskForDocking invalidates -norepack1 and -norepack2 flags
			TaskFactoryOP tf = new TaskFactory;

		tf->push_back( new OperateOnCertainResidues( new PreventRepackingRLT, new ResidueLacksProperty("PROTEIN") ) );
		tf->push_back( new InitializeFromCommandline );
		tf->push_back( new IncludeCurrent );
		tf->push_back( new RestrictToRepacking );
		tf->push_back( new NoRepackDisulfides );
		tf->push_back( new SymRestrictTaskForDocking( docking_scorefxn, true, 1000 ) );

		if( option[OptionKeys::packing::resfile].user() ) tf->push_back( new ReadResfile );

			moves::symmetry::SymPackRotamersMover pack( docking_scorefxn );
			pack.task_factory( tf );
			pack.apply( pose );
			// run high resolution docking
			docking_high_->apply( pose );

			// add scores to map for output
			ScoreMap::nonzero_energies( score_map_, docking_scorefxn, pose );
			calc_interaction_energy( pose );

			// check highres docking filter
			passed_filters = docking_highres_filter( pose );
			}

		if ( passed_filters ) break;
		if (!passed_filters) TR<<"REPEAT STRUCTURE "<< r <<std::endl;

		}//end of repeat decoy creation

	// calculate and store the rms no matter which mode was used
	//score_map_["rms"] = rmsd_no_super_subset( *get_native_pose(), pose, superpos_partner, is_protein_CA );
	score_map_["rms"] = calc_symmetric_rms( pose );
	calc_interaction_energy( pose );

	if ( option[ OptionKeys::run::debug ]() ) util::prof_show();

	// cache the score map to the pose
	pose.data().set(SCORE_MAP, new core::util::datacache::DiagnosticData(score_map_));
}

bool
SymDockProtocol::docking_lowres_filter( core::pose::Pose & pose){

	using namespace scoring;
	using namespace options;

	bool passed_filter = true;
	if ( !option[ OptionKeys::docking::fake_native ]() ){
		//criterion for failure
		Real interchain_contact_cutoff	= 10.0;
		Real interchain_vdw_cutoff = 1.0;

		if( option[ OptionKeys::docking::dock_lowres_filter ].user() ) {
			utility::vector1< Real > dock_filters = option[ OptionKeys::docking::dock_lowres_filter ]();
			interchain_contact_cutoff = dock_filters[1];
			interchain_vdw_cutoff = dock_filters[2];
			}

		if (pose.energies().total_energies()[ interchain_contact ] >= interchain_contact_cutoff ) passed_filter = false;
		if (pose.energies().total_energies()[ interchain_vdw ] >= interchain_vdw_cutoff ) passed_filter = false;
		}

	return passed_filter;
}

bool
SymDockProtocol::docking_highres_filter( core::pose::Pose & pose){

	using namespace scoring;
	using namespace options;

	bool passed_filter = true;
	if ( option[ OptionKeys::docking::dock_ppk ]() ) return passed_filter;

	Real score_cutoff = option[ OptionKeys::cluster::output_score_filter ]();
	//criterion for failure
	if (pose.energies().total_energy() >= score_cutoff) passed_filter = false;
	if (score_map_["I_sc"] >= 0.0) passed_filter = false;

	return passed_filter;
}

void
SymDockProtocol::calc_interaction_energy( core::pose::Pose & pose ){
	using namespace scoring;
	using namespace core::conformation::symmetry;

	core::scoring::ScoreFunctionOP docking_scorefxn;
	core::pose::Pose complex_pose = pose;

	Real trans_magnitude = 1000;

	assert( is_symmetric( pose ) );
  SymmetricConformation & symm_conf (
        dynamic_cast<SymmetricConformation & > ( pose.conformation()) );
	//SymmetryInfo const & symm_info( symm_conf.Symmetry_Info() );

	std::map< Size, SymDof > dofs ( symm_conf.Symmetry_Info().get_dofs() );
	moves::RigidBodyDofSeqTransMoverOP translate_away ( new moves::RigidBodyDofSeqTransMover( dofs ) );
	translate_away->step_size( trans_magnitude );

	if ( fullatom_ ){
		docking_scorefxn = new core::scoring::symmetry::SymmetricScoreFunction( *scorefxn_hires_ ) ;
	} else {
		docking_scorefxn = new core::scoring::symmetry::SymmetricScoreFunction( *scorefxn_lowres_ ) ;
	}

	Real bound_energy = (*docking_scorefxn)( complex_pose );
	translate_away->apply( complex_pose );

	Real unbound_energy = (*docking_scorefxn)( complex_pose );

	score_map_["I_sc"] = bound_energy - unbound_energy;

}

core::Real
SymDockProtocol::calc_symmetric_rms( core::pose::Pose & pose ){

	using namespace core::conformation::symmetry;

	assert( core::conformation::symmetry::is_symmetric( pose ) );
  SymmetricConformation & SymmConf (
		dynamic_cast<SymmetricConformation &> ( pose.conformation()) );
  SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );

	FArray1D_bool superpos ( pose.total_residue(), false );
	for (Size res=1; res <= symm_info.num_total_residues_without_pseudo(); ++res )
	{
		superpos(res) = true;
	}

	if ( get_native_pose() ) {
		return core::scoring::rmsd_with_super_subset( *get_native_pose(), pose, superpos, core::scoring::is_protein_CA );
	}
	return -1;
}

} // symmetric_docking
} // protocols
