// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file SymDockingHiRes
/// @brief protocols that are specific to high resolution docking
/// @author Ingemar Andre

#include <protocols/symmetric_docking/SymDockingHiRes.hh>
#include <protocols/symmetric_docking/SymRestrictTaskForDocking.hh>
#include <protocols/symmetric_docking/SymRestrictTaskForDocking.fwd.hh>

// Rosetta Headers
#include <core/kinematics/MoveMap.hh>

//#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>

#include <core/pose/Pose.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/OutputMovers.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/moves/TrialMover.hh>
#include <protocols/moves/JumpOutMover.hh>
#include <protocols/moves/RepeatMover.hh>

//for resfile reading
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/options/option.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/options/option.hh>

//Symmetry
#include <core/conformation/symmetry/util.hh>
#include <protocols/moves/symmetry/SymPackRotamersMover.hh>
#include <protocols/moves/symmetry/SymRotamerTrialsMover.hh>
#include <protocols/moves/symmetry/SymMinMover.hh>
#include<core/conformation/symmetry/SymDof.hh>

// For symmetry
#include <core/conformation/symmetry/SymmetricConformation.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <string>

//Utility Headers
#include <numeric/conversions.hh>
//	#include <numeric/random.functions.hh>

#include <numeric/trig.functions.hh>
#include <numeric/xyzMatrix.fwd.hh>

#include <core/util/prof.hh>
#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.docking.SymDockingHiRes");

using namespace core;

namespace protocols {
namespace symmetric_docking {

// default constructor
SymDockingHiRes::SymDockingHiRes() : Mover()
{
	moves::Mover::type( "SymDockingHiRes" );
}

// constructor with arguments
SymDockingHiRes::SymDockingHiRes(
	core::scoring::ScoreFunctionOP scorefxn_in
) : Mover(), scorefxn_(scorefxn_in)
{
	moves::Mover::type( "SymDockingHiRes" );
}

moves::MoverOP
SymDockingHiRes::clone() const {
	return new SymDockingHiRes(*this);
}

SymDockingHiRes::~SymDockingHiRes(){}

void SymDockingHiRes::set_default_mc( pose::Pose & pose ) {
	// create the monte carlo object and movemap
	mc_ = new moves::MonteCarlo( pose, *scorefxn_, temperature_ );
}

void SymDockingHiRes::set_default_protocol( pose::Pose & /*pose*/ ){
	using namespace options;
}

void SymDockingHiRes::set_default_move_map( pose::Pose & pose ) {
	using namespace core::conformation::symmetry;

	assert( is_symmetric( pose ) );
  SymmetricConformation & symm_conf (
        dynamic_cast<SymmetricConformation & > ( pose.conformation()) );

	movemap_ = new kinematics::MoveMap();
	movemap_->set_bb( minimize_backbone_ );
	movemap_->set_chi( minimize_sidechains_ );
	movemap_->set_jump( false );
  std::map< Size, SymDof > dofs ( symm_conf.Symmetry_Info().get_dofs() );

	std::map< Size, SymDof >::iterator it;
  std::map< Size, SymDof >::iterator it_begin = dofs.begin();
  std::map< Size, SymDof >::iterator it_end = dofs.end();
  for ( it = it_begin; it != it_end; ++it ) {
    int jump_nbr ( (*it).first );
		SymDof dof( (*it).second );
		if ( dof.allow_dof( X_DOF ) ) {
			id::DOF_ID const & id
      	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,1)));
			movemap_->set( id, true );
		}
		if ( dof.allow_dof( Y_DOF ) ) {
			id::DOF_ID const & id
      	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,2)));
			movemap_->set( id, true );
		}
		if ( dof.allow_dof( Z_DOF ) ) {
			id::DOF_ID const & id
      	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,3)));
			movemap_->set( id, true );
		}
		if ( dof.allow_dof( X_ANGLE_DOF ) ) {
			id::DOF_ID const & id
      	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,4)));
			movemap_->set( id, true );
		}
		if ( dof.allow_dof( Y_ANGLE_DOF ) ) {
			id::DOF_ID const & id
      	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,5)));
			movemap_->set( id, true );
		}
		if ( dof.allow_dof( Z_ANGLE_DOF ) ) {
			id::DOF_ID const & id
      	( pose.conformation().dof_id_from_torsion_id(id::TorsionID(jump_nbr,id::JUMP,6)));
			movemap_->set( id, true );
		}
	}
}

void SymDockingHiRes::set_default_packers( pose::Pose & pose ) {
	// packer options
	protocols::symmetric_docking::SymRestrictTaskForDockingOP rtfd = new protocols::symmetric_docking::SymRestrictTaskForDocking( scorefxn_, true );
	pack::task::TaskFactory tf;
	tf.push_back( rtfd );
	//making docking sensitive to resfiles
	if( options::option[options::OptionKeys::packing::resfile].user() ) tf.push_back( new core::pack::task::operation::ReadResfile );

	// this task does double duty(?)
	pack::task::PackerTaskOP task = tf.create_task_and_apply_taskoperations( pose );

	(*scorefxn_)( pose );

	if ( repack_switch_ == true ) {
		pack_interface_repack_ = new moves::symmetry::SymPackRotamersMover( scorefxn_, task );
		pack_rottrial_ = new moves::symmetry::SymEnergyCutRotamerTrialsMover( scorefxn_, *task, mc_, energycut_ );
	}
}

void SymDockingHiRes::set_default_minimizer() {
	// options for the minimizer
	TR.Debug << "min_type: " << min_type_  << std::endl;
	min_mover_ = new moves::symmetry::SymMinMover( movemap_, scorefxn_, min_type_, loose_func_tol_, nb_list_ );
	min_mover_->threshold( minimization_threshold_ );
}

void SymDockingHiRes::set_min_type( std::string min_type_in ) {
	// helper to reset minimizer
	min_type_ = min_type_in;
	set_default_minimizer();
}

void SymDockingHiRes::set_repack( bool repack_switch ) {
	repack_switch_ = repack_switch;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking high resolution apply function
/// @brief
/// @detailed
///		decides what to call according to options
void SymDockingHiRes::apply( core::pose::Pose & pose )
{
	using namespace scoring;
	using namespace options;

	TR << "in SymDockingHiRes.apply" << std::endl;

 // jec sanity check to avoid overwriting newly-set minimizers on every apply
  if ( !min_mover_ ) {
    set_default( pose );
  }

		assert( core::conformation::symmetry::is_symmetric( pose ));
		mc_->reset( pose );

		dock_mcm_protocol( pose );

//		classic_mcm_protocol( pose, scorefxn_, mc_, 45, 8 );
		docking_highres_protocol_mover_->apply( pose );

}

void
SymDockingHiRes::dock_mcm_protocol( core::pose::Pose & pose ) {
	using namespace moves;
	using namespace conformation::symmetry;

	assert( is_symmetric( pose ));
  SymmetricConformation & symm_conf (
        dynamic_cast<SymmetricConformation & > ( pose.conformation()) );

  std::map< Size, SymDof > dofs ( symm_conf.Symmetry_Info().get_dofs() );

	//set up movers for mcm cycle
	RigidBodyDofSeqPerturbMoverOP rb_perturb = new RigidBodyDofSeqPerturbMover( dofs , rot_magnitude_, trans_magnitude_ );

	// the standard mcm cycle : rb perturbation, pack_rottrial_, minimization, MC
	SequenceMoverOP rb_mover = new SequenceMover;
	rb_mover->add_mover( rb_perturb );
	if ( repack_switch_ ) rb_mover->add_mover( pack_rottrial_ );

	JumpOutMoverOP rb_mover_min = new JumpOutMover( rb_mover, min_mover_, scorefxn_, minimization_threshold_ );
	TrialMoverOP rb_mover_min_trial = new TrialMover( rb_mover_min, mc_ );

	//the standard mcm cycle + intermittent repacking
	SequenceMoverOP repack_step = new SequenceMover;
	repack_step->add_mover(rb_mover_min_trial);
	if ( repack_switch_ ) repack_step->add_mover( pack_interface_repack_ );

	CycleMoverOP rb_mover_min_trial_repack  = new CycleMover;
	for ( Size i=1; i<repack_period_; ++i ) rb_mover_min_trial_repack->add_mover( rb_mover_min_trial );
	rb_mover_min_trial_repack->add_mover( repack_step );

	// set up intial and final minimizers
	moves::symmetry::SymMinMoverOP min_mover_loose = new moves::symmetry::SymMinMover( movemap_, scorefxn_, min_type_, loose_func_tol_, nb_list_ );
	TrialMoverOP minimize_trial_loose = new TrialMover( min_mover_loose, mc_ );

	moves::symmetry::SymMinMoverOP min_mover_tight = new moves::symmetry::SymMinMover( movemap_, scorefxn_, min_type_, tight_func_tol_, nb_list_ );
	TrialMoverOP minimize_trial_tight = new TrialMover( min_mover_tight, mc_ );

	RepeatMoverOP mcm_four_cycles = new RepeatMover( rb_mover_min_trial, four_cycles_ );
	RepeatMoverOP mcm_fortyfive_cycles = new RepeatMover( rb_mover_min_trial_repack, fortyfive_cycles_ );


	TR << "::::::::::::::::::DOCK_MCM:::::::::::::::::::" << std::endl;

	docking_highres_protocol_mover_ = new SequenceMover;
	docking_highres_protocol_mover_->add_mover( minimize_trial_loose );
	docking_highres_protocol_mover_->add_mover( mcm_four_cycles );
	docking_highres_protocol_mover_->add_mover( mcm_fortyfive_cycles );
	docking_highres_protocol_mover_->add_mover( minimize_trial_tight );

}

void SymDockingHiRes::classic_mcm_protocol(
  core::pose::Pose & pose,
  core::scoring::ScoreFunctionCOP /*scorefxn*/,
  protocols::moves::MonteCarloOP monteCarlo,
  core::Size num_cycles,
  core::Size repack_every_Nth
)
{
  using namespace protocols::moves;
	using namespace conformation::symmetry;

  assert(repack_every_Nth >= 2);

	assert( is_symmetric( pose ));
  SymmetricConformation & symm_conf (
        dynamic_cast<SymmetricConformation & > ( pose.conformation()) );

  std::map< Size, SymDof > dofs ( symm_conf.Symmetry_Info().get_dofs() );

  // Rigid body exploration
	MoverOP rb_mover_;
  rb_mover_ = new RigidBodyDofSeqPerturbMover( dofs , rot_magnitude_, trans_magnitude_ );

  for( core::Size cycle = 1; cycle <= num_cycles; ++cycle ) {

		pose.dump_pdb( "brb"+lead_zero_string_of( cycle,4 )+".pdb" );
		rb_mover_->apply( pose );
		pose.dump_pdb( "rb"+lead_zero_string_of( cycle,4 )+".pdb" );
		MoverOP pack_mover;
		if ( cycle % repack_every_Nth == 1 ) {
			pack_mover = pack_interface_repack_;
		} else {
			pack_mover = pack_rottrial_;
		}
		pack_mover->apply( pose );
		pose.dump_pdb( "pack"+lead_zero_string_of( cycle,4 )+".pdb" );
//		MoverOP min_mover = new moves::symmetry::SymMinMover( movemap_, scorefxn, "dfpmin_armijo_nonmonotone_atol", 1.0, true /*use_nblist*/ );
    //  min_mover->min_options()->nblist_auto_update(true); // does this cost us lots of time in practice?
		min_mover_->apply( pose );
		pose.dump_pdb( "min"+lead_zero_string_of( cycle,4 )+".pdb" );
/*		std::cout << "before" << std::endl;
		Real before = (*scorefxn)( pose );
		scorefxn->show(std::cout, pose );
		pose.dump_pdb("before_rot.pdb");
	  util::prof_reset();
    pack_mover->apply(pose);
		util::prof_show();
		std::cout << "after" << std::endl;
		Real after = (*scorefxn)( pose );
		scorefxn->show(std::cout, pose );
		if (after > before ) {
			std::cout << "score increase " << std::endl;
		}
      //min_mover->apply(pose);
      (*scorefxn)( pose ); // no effect at alla
			*/
      monteCarlo->boltzmann( pose );
    }
}

pose::Pose
SymDockingHiRes::make_one_jump_pose( core::pose::Pose & pose )
{

		using namespace core::conformation::symmetry;

		assert( is_symmetric( pose ));
	  SymmetricConformation & symm_conf (
        dynamic_cast<SymmetricConformation & > ( pose.conformation()) );

		Size nres_subunit ( symm_conf.Symmetry_Info().num_independent_residues() ); // Is this always going to be = subunit length
		Size tot_res_no_pseudo( symm_conf.Symmetry_Info().num_total_residues_without_pseudo() );
		Size tot_res( pose.conformation().size() );

		core::pose::Pose pose_one_jump( pose );
		pose_one_jump.conformation().delete_residue_range_slow( tot_res_no_pseudo+1, tot_res );
		pose_one_jump.copy_segment( tot_res_no_pseudo, pose, 1, 1 );

    Size jump_pos1 ( 1 );
    Size jump_pos2 ( nres_subunit + 1 );

		core::kinematics::FoldTree f;
    f.clear();
    f.add_edge( jump_pos1, jump_pos2, 1 );
    f.add_edge( jump_pos1, nres_subunit, kinematics::Edge::PEPTIDE );
    f.add_edge( jump_pos2, tot_res_no_pseudo, kinematics::Edge::PEPTIDE );
    f.reorder( 1 );

		std::cout << "mytree " << f << std::endl;
    pose_one_jump.fold_tree( f );
		return pose_one_jump;
}


///////////////////////////////////////////////////////////////////////////////////
/// @begin dock_mcm_protocol
///
/// @brief main entrance to do monte carlo minimization
/// @detailed
///			a total of 50 cycles of monte-carlo minimization will be
///			carried out if the minimized structure can pass the filter
///			after the first and fifth cycle.  Then it is rigid-body minimized
///			to a stringent tolerance.
///
/// @remarks
///
/// @references docking_mcm_protocol from docking_minimize.cc
///				pose_docking_monte_carlo_minimize from pose_docking.cc
///
/// @authors Monica Berrondo June 14 2007
///
/// @last_modified April 30 2008
/////////////////////////////////////////////////////////////////////////////////
/*void DockingHiRes::dock_mcm_protocol() {
	using namespace moves;
	CycleMoverOP repack_cycle = new CycleMover;
	// set up the movers
	// only repack if repack_ was set true (defaults to such) and the repack_period is set > 0
	if ( ( repack_switch_ == true ) && ( repack_period_ > 0 ) ) {
		for ( Size i=1; i<repack_period_; ++i )
			repack_cycle->add_mover( pack_rottrial_ );
		repack_cycle->add_mover( pack_interface_repack_ );
	}

	bool interface_rbcenter = true;
	RigidBodyPerturbMoverOP rb_mover = new RigidBodyPerturbMover( rb_jump_, rot_magnitude_, trans_magnitude_ , partner_downstream, interface_rbcenter);

	moves::MinMoverOP min_mover_tight = new moves::MinMover( movemap_, scorefxn_, min_type_, tight_func_tol_, nb_list_ );

	/// the mcm cycle for the first four iteration
	SequenceMoverOP mcm_cycle_4 = new SequenceMover;
	mcm_cycle_4->add_mover( rb_mover );
	// only repack if repack_ was set true (defaults to such) and the repack_period is set > 0
	if ( ( repack_switch_ == true ) && ( repack_period_ > 0 ) ) {
		mcm_cycle_4->add_mover( pack_rottrial_ );
	}

	/// the mcm cycle for the 45 iterations
	SequenceMoverOP mcm_cycle_45 = new SequenceMover;
	mcm_cycle_45->add_mover( rb_mover );
	// only repack if repack_ was set true (defaults to such) and the repack_period is set > 0
	if ( ( repack_switch_ == true ) && ( repack_period_ > 0 ) ) {
		mcm_cycle_45->add_mover( repack_cycle );
	}

	TrialMoverOP minimize_trial_loose = new TrialMover( min_mover_, mc_ );
	TrialMoverOP minimize_trial_tight = new TrialMover( min_mover_tight, mc_ );

	JumpOutMoverOP mcm_mover = new JumpOutMover( mcm_cycle_4, min_mover_, scorefxn_, minimization_threshold_ );
	TrialMoverOP mcm_trial = new TrialMover( mcm_mover, mc_ );
	RepeatMoverOP mcm_four_cycles = new RepeatMover( mcm_trial, four_cycles_ );

	mcm_mover = new JumpOutMover( mcm_cycle_45, min_mover_, scorefxn_, minimization_threshold_ );
	mcm_trial = new TrialMover( mcm_mover, mc_ );
	RepeatMoverOP mcm_fortyfive_cycles = new RepeatMover( mcm_trial, fortyfive_cycles_ );

	TR << "::::::::::::::::::DOCK_MCM:::::::::::::::::::" << std::endl;

	docking_highres_protocol_mover_ = new SequenceMover;
	docking_highres_protocol_mover_->add_mover( minimize_trial_loose );
	docking_highres_protocol_mover_->add_mover( mcm_four_cycles );
	docking_highres_protocol_mover_->add_mover( mcm_fortyfive_cycles );
	docking_highres_protocol_mover_->add_mover( minimize_trial_tight );
*/
/*
	/// the filter here can probably be made into some combination of JumpOutMovers
	// JumpOutMoverOP fab_filter = new JumpOutMover(
	// docking_mcm_filter(delta_before_mcm)
	if ( true ) { /// TODO make this a filter
		minimize_trial->apply( pose );
		// docking_mcm_filter(delta_after_one_min)
		if ( true ) { /// TODO make this a filter
			mcm_four_cycles->apply( pose );
			// docking_mcm_filter(delta_after_five_mcm)
			if ( true ) { /// TODO make this a filter
				mcm_fortyfive_cycles->apply( pose );
				min_mover_->tolerance( tight_func_tol_ );
				minimize_trial->apply( pose );
			}
		}
	}
*/
//}

} // namespace symmetric_docking
} // namespace protocols
