// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file DockingInitialPerturbation.cc
/// @brief initial position functions
/// @detailed
///		This contains the functions that create initial positions for docking
/// @author Ingemar Andre

#include <protocols/symmetric_docking/SymDockingInitialPerturbation.hh>

// Rosetta Headers
#include <protocols/moves/Mover.hh>
#include <protocols/moves/RigidBodyMover.hh>

#include <core/options/option.hh>
#include <core/options/keys/docking.OptionKeys.gen.hh>
#include <core/options/keys/symmetry.OptionKeys.gen.hh>

// for symmetry
#include <core/conformation/symmetry/util.hh>
#include <core/conformation/symmetry/SymmetricConformation.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <string>

//Utility Headers
#include <numeric/conversions.hh>

#include <numeric/trig.functions.hh>
#include <numeric/xyzMatrix.fwd.hh>

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.symmetric_docking.SymDockingInitialPerturbation");
static core::Size trans ( 1 ), rot ( 2 );

using namespace core;

namespace protocols {
namespace symmetric_docking {


// Symmetric version of initial perturbation on one of the partners
// the type of perturbation is defined in the options
// some of the options are randomize1 or randomize2 (they are the same),
// dock_pert
//------------------------------------------------------------------------------
//
//     there are several ways to perturb the structure before beginning
//     the search; they are controlled through command-line flags
//
//     at the end, partners are slid into contact and scored
//

////////////////////////////////////////////////////////////////////////////////
/// @begin initial_perturbation
///
/// @brief   Make starting perturbations for rigid body moves
///
/////////////////////////////////////////////////////////////////////////////////
void SymDockingInitialPerturbation::apply( core::pose::Pose & pose )
{
	using namespace moves;
	using namespace options;
	using namespace conformation::symmetry;
	//////////////////////////////////
	//The options are -symmetry::initialize_rigid_body_dofs
	//								-symmetry::perturb_rigid_body_dofs <trans> <rot>
	//						for dock_pert, also need to get the normal perturbation
	// In the future this is specified in the symmetry definition file
	//////////////////////////////////
	assert( is_symmetric( pose ));
	SymmetricConformation & symm_conf (
        dynamic_cast<SymmetricConformation & > ( pose.conformation()) );

	std::map< Size, SymDof > dofs ( symm_conf.Symmetry_Info().get_dofs() );

	TR << "Reading options..." << std::endl;
 	if( option[ OptionKeys::symmetry::initialize_rigid_body_dofs ]() ) {
		TR << "initialize_rigid_body_dofs: true" << std::endl;
 		RigidBodyDofSeqRandomizeMover mover( dofs );
 		mover.apply( pose );
 	}

	if( option[ OptionKeys::symmetry::perturb_rigid_body_dofs ].user() ) {
		TR << "perturb_rigid_body_dofs: true" << std::endl;
		/// read in dock_pert options from commandline.  the first value is the
		/// rotation magnitude and the second value is the translational value
		utility::vector1< Real > pert_mags = option[ OptionKeys::symmetry::perturb_rigid_body_dofs ]();
		TR << "option[ symmetry::perturb_rigid_body_dofs ]() rot=" << pert_mags[rot] << "  trans=" << pert_mags[trans] << std::endl;
		RigidBodyDofSeqPerturbMover mover( dofs, pert_mags[rot], pert_mags[trans] );
		mover.apply( pose );
	}
	// DO NOT do this for e.g. ligand docking
	if ( slide_ ) {
		SymDockingSlideIntoContact slide( dofs );
		slide.apply( pose );
	}
}

void SymDockingSlideIntoContact::apply( core::pose::Pose & pose )
{
	using namespace moves;
	using namespace conformation::symmetry;

	assert( is_symmetric( pose ));
	SymmetricConformation & symm_conf (
        dynamic_cast<SymmetricConformation & > ( pose.conformation()) );

  std::map< Size, SymDof > dofs ( symm_conf.Symmetry_Info().get_dofs() );
	RigidBodyDofRandomTransMover mover( dofs );
	( *scorefxn_ )( pose );
	TR.Debug << "score " << pose.energies().total_energies()[ scoring::interchain_vdw ]  << std::endl;
	TR.Debug << "sliding into contact" << std::endl;
	TR.Debug << "Moving away" << std::endl;
	// first try moving away from each other
	while ( pose.energies().total_energies()[ scoring::interchain_vdw ] > 0.1 ) {
		mover.apply( pose );
		( *scorefxn_ )( pose );
	TR.Debug << "score away " << pose.energies().total_energies()[ scoring::interchain_vdw ]  << std::endl;
	}
	// then try moving towards each other
	TR.Debug << "Moving together" << std::endl;
	mover.trans_axis().negate();
	while ( pose.energies().total_energies()[ scoring::interchain_vdw ] < 0.1 ) {
		mover.apply( pose );
		TR.Debug << "score together " << pose.energies().total_energies()[ scoring::interchain_vdw ]  << std::endl;
		( *scorefxn_ )( pose );
	}
	// move away again until just touching
	mover.trans_axis().negate();
	mover.apply( pose );

}

void FaSymDockingSlideTogether::apply( core::pose::Pose & pose )
{
	using namespace core::scoring;
	using namespace conformation::symmetry;

	assert( is_symmetric( pose ));
  SymmetricConformation & symm_conf (
        dynamic_cast<SymmetricConformation & > ( pose.conformation()) );

  std::map< Size, SymDof > dofs ( symm_conf.Symmetry_Info().get_dofs() );

	// A very hacky way of guessing whether the components are touching:
	// if pushed together by 1A, does fa_rep change at all?
	// (The docking rb_* score terms aren't implemented as of this writing.)
	(*scorefxn_)( pose );
	core::Real const initial_fa_rep = pose.energies().total_energies()[ fa_rep ];
	bool are_touching = false;
	moves::RigidBodyDofSeqTransMover trans_mover( dofs );

	//int i=1;
	// Take 2A steps till clash, then back apart one step.  Now you're within 2A of touching.
	// Repeat with 1A steps, 0.5A steps, 0.25A steps, etc until you're as close are you want.
	for( core::Real stepsize = 2.0; stepsize > tolerance_; stepsize /= 2.0 ) {
		trans_mover.trans_axis( trans_mover.trans_axis().negate() ); // now move together
		trans_mover.step_size(stepsize);
		do
		{
			trans_mover.apply( pose );
			(*scorefxn_)( pose );
			core::Real const push_together_fa_rep = pose.energies().total_energies()[ fa_rep ];
			//std::cout << "fa_rep = " << push_together_fa_rep << std::endl;
			are_touching = (std::abs(initial_fa_rep - push_together_fa_rep) > 1e-4);
			//std::ostringstream s;
			//s << "snapshot" << i << ".pdb";
			//pose.dump_pdb(s.str());
			//i += 1;
		} while( !are_touching );
		trans_mover.trans_axis( trans_mover.trans_axis().negate() ); // now move apart
		trans_mover.apply( pose );
	}
}

} // namespace docking
} // namespace protocols
