// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @brief initialization protocols for symmetrical docking
/// @author Ingemar Andre


#ifndef INCLUDED_protocols_symmetric_docking_SymDockingInitialPerturbation_HH
#define INCLUDED_protocols_symmetric_docking_SymDockingInitialPerturbation_HH

// Package headers
#include <core/pose/Pose.hh>

#include <core/scoring/symmetry/SymmetricScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <protocols/moves/Mover.hh>
#include <protocols/moves/RigidBodyMover.hh>

#include <core/conformation/symmetry/SymDof.hh>

#include <utility/pointer/owning_ptr.hh>
#include <string>

namespace protocols {
namespace symmetric_docking {

class SymDockingInitialPerturbation; // fwd declaration
typedef utility::pointer::owning_ptr< SymDockingInitialPerturbation > SymDockingInitialPerturbationOP;
typedef utility::pointer::owning_ptr< SymDockingInitialPerturbation const > SymDockingInitialPerturbationCOP;

class SymDockingInitialPerturbation : public moves::Mover
{
public:

	// default constructor
	SymDockingInitialPerturbation() : Mover()
	{
		slide_ = false;
		Mover::type( "SymmDockingInitialPerturbation" );
	}

	// constructor with arguments
	SymDockingInitialPerturbation(
		bool const slide_in
	) : Mover(),
			slide_(slide_in)
	{
		Mover::type( "SymmDockingInitialPerturbation" );
	}

	// protocol functions
	virtual void apply( core::pose::Pose & pose );

private:
	/// do slide into context?
	bool slide_;
};

/// @brief Contrary to the name, slides things apart first, then together.
class SymDockingSlideIntoContact : public moves::Mover
{
public:

	// default constructor
	SymDockingSlideIntoContact() : Mover() {	}

	// constructor with arguments
	SymDockingSlideIntoContact(
		std::map< Size, core::conformation::symmetry::SymDof > dofs
	) : Mover(),
			dofs_(dofs)
	{
		using namespace core::scoring;
		Mover::type( "SymDockingSlideIntoContact" );
		symmetry::SymmetricScoreFunction scorefxn_sym ( ScoreFunctionFactory::create_score_function( CENTROID_WTS, DOCK_LOW_PATCH ) );
		scorefxn_ = new symmetry::SymmetricScoreFunction( scorefxn_sym );
	}

	// protocol functions
	virtual void apply( core::pose::Pose & pose );

private:
	// dof to allow
	std::map< Size, core::conformation::symmetry::SymDof > dofs_;
	// scorefxn to use
	core::scoring::symmetry::SymmetricScoreFunctionOP scorefxn_;
};

class FaSymDockingSlideTogether; // fwd declaration
typedef utility::pointer::owning_ptr< FaSymDockingSlideTogether > FaSymDockingSlideTogetherOP;
typedef utility::pointer::owning_ptr< FaSymDockingSlideTogether const > FaSymDockingSlideTogetherCOP;

/// @brief Slides docking partners together by monitoring fa_rep.
/// @details
///		If partners are already touching, no change is made.
///		Separation will be 1A or less after calling this function.
class FaSymDockingSlideTogether : public moves::Mover
{
public:
	FaSymDockingSlideTogether(
		std::map< Size, core::conformation::symmetry::SymDof > dofs
	) : Mover(),
			dofs_(dofs),
			tolerance_(0.2)
	{
		Mover::type( "FaSymDockingSlideTogether" );
		scorefxn_ = new core::scoring::symmetry::SymmetricScoreFunction();
		scorefxn_->set_weight( core::scoring::fa_rep, 1.0 );
	}

	virtual void apply( core::pose::Pose & pose );

private:
	 // dof to allow
  std::map< Size, core::conformation::symmetry::SymDof > dofs_;
	// scorefxn to use
	core::scoring::symmetry::SymmetricScoreFunctionOP scorefxn_;
	core::Real tolerance_; ///< how accurate do you want to be?

};

} // docking
} // protocols

#endif
