// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/jd2/Job.hh
/// @brief  header file for ThreadingJob classes, part of August 2008 job distributor as planned at RosettaCon08.  This file is responsible for three ideas: "inner" jobs, "outer" jobs (with which the job distributor works) and job container (currently just typdefed in the .fwd.hh)
/// @author Steven Lewis smlewi@gmail.com


#include <protocols/toolbox/DecoySetEvaluation.hh>
#include <protocols/toolbox/superimpose.hh>

#include <core/types.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <utility/excn/Exceptions.hh>
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>


// ObjexxFCL Headers

// Utility headers
#include <core/util/Tracer.hh>
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <numeric/model_quality/rms.hh>
#include <numeric/model_quality/maxsub.hh>

//// C++ headers
#include <string>
#include <iostream>

namespace protocols {
namespace toolbox {

using namespace ObjexxFCL;

static core::util::Tracer tr("protocols.evaluation.PCA",core::util::t_info);

using namespace core;
using namespace numeric::model_quality; //for rms functions

typedef core::Real matrix[3][3];
typedef core::Real rvec[3];


DecoySetEvaluation::DecoySetEvaluation() : COM( 3 ),n_decoys_( 0 ), n_atoms_( 0 ), n_decoys_max_( 0 )
{}

void DecoySetEvaluation::reserve( core::Size n_resize ) {
  if ( n_resize == n_decoys_max_ ) return;
  n_decoys_max_ = n_resize;
  if ( n_atoms_ ) coords_.redimension( 3, n_atoms_, n_decoys_max_ );
}

void DecoySetEvaluation::push_back( core::pose::Pose& pose ) {
  if ( n_decoys_ == 0 ) {
    n_atoms_ = pose.total_residue();
    coords_.dimension( 3, n_atoms_, n_decoys_max_ );
    weights_.dimension( n_atoms_, 0.1 );
    ref_structure_.dimension( 3, n_atoms_ );
  } else {
    if ( n_atoms_ != pose.total_residue() ) {
      utility_exit_with_message( "can't insert poses with varying sizes into DecoySetEvaluation" );
    }
  }

  if ( n_decoys_ >= n_decoys_max_ ) {
    // we could also just resize... but I want the user to know.. because he should keep track which decoy is which...
    throw utility::excn::EXCN_RangeError( "you can't add any more decoys to DecoySetEvaluation ");
  }
  ++n_decoys_;

	// fill coords
  for ( core::Size i = 1; i <= n_atoms_; i++ ) {
    id::NamedAtomID idCA( "CA", i );
    PointPosition const& xyz = pose.xyz( idCA );
		for ( core::Size d = 1; d<=3; ++d ) {
      coords_( d, i, n_decoys_ ) = xyz[ d-1 ];
    }
  }

	if ( n_decoys_ == 1 ) {
		ref_pose_ = pose;
	}

}


void DecoySetEvaluation::superimpose() {
	FArray1D_double const weights( n_atoms_, 1.0 );
	superimpose( weights );
}

void DecoySetEvaluation::superimpose( FArray1DB_double const& weights ) {
	FArray2Dp_double xx_ref( coords_( 1, 1, 1 ), 3, n_atoms_ ); //proxy array provides view to coords_

	for ( Size n = 1; n<= n_decoys_; ++n ) {
		FArray2Dp_double xx( coords_( 1, 1, n ), 3, n_atoms_ ); //proxy array provides view to coords_
		FArray1D_double transvec( 3 );
		reset_x( n_atoms_, xx, weights, transvec );

		//fit
		if ( n > 1 ) {
			Matrix R;
			FArray2Dp_double xx2( coords_( 1, 1, n ), 3, n_atoms_ ); //proxy array provides view to coords_
			fit_centered_coords( n_atoms_, weights, xx_ref, xx2, R );
		}
	} // for n
}


Real DecoySetEvaluation::rmsf( core::Size pos ) {
	//Real rms( 0 );
	Vector rmsd_x( 0.0 );
	Vector rmsd_av( 0.0 );
	Real invn( 1.0 /n_decoys() );
	for ( Size idecoy = 1; idecoy <= n_decoys(); ++idecoy ) {
		for ( Size d = 1; d<=3; ++d ) {
			Real dx = coords_( d, pos, idecoy );
			rmsd_x( d ) += dx * dx * invn;
			rmsd_av( d ) += dx * invn;
		}
	}
	Real rms( 0 );
	for ( Size d = 1; d<=3; ++d ) {
		rms += rmsd_x( d ) - rmsd_av( d )*rmsd_av( d );
	}
	return sqrt( rms );
}

void DecoySetEvaluation::rmsf( utility::vector1< Real >& result ) {
  result.clear();
  result.reserve( n_atoms_ );

  for ( Size pos = 1; pos <= n_atoms_; ++pos ) {
		result.push_back( rmsf( pos ) );
  }
}

void DecoySetEvaluation::rmsf( FArray1DB_double& result ) {
  for ( Size pos = 1; pos <= n_atoms_; ++pos ) {
		result( pos ) = rmsf( pos );
  }
}

void DecoySetEvaluation::wRMSD( Real sigma2, Real tolerance, ObjexxFCL::FArray1DB_double& weights ) {
	//FArray1D_double weights( n_atoms_, 1.0 );
	Real wsum_old = 100;
	Real wsum ( 1.0 );
	Real invn ( 1.0/n_atoms_ );
	Size ct ( 0 );
	tr.Info << "wRMSD: iter  wsum   wRMSD" << std::endl;
	while ( std::abs( wsum_old - wsum ) > tolerance ) {
		superimpose( weights );
		rmsf( weights );
		wsum_old = wsum;
		wsum = 0.0;
		Real wMSD = 0.0;
		for ( Size i = 1; i <= n_atoms_; ++i ) {
			Real di2 = weights( i )*weights( i );
			weights( i ) = exp( - di2 / sigma2 );
			wsum += weights( i )*invn;
			wMSD += weights( i ) * di2;
		}
		tr.Info << "wRMSD: " << ++ct << " " << wsum << " " << sqrt( wMSD ) << std::endl;
	}
}

} //evaluation
} //protocols


