// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file /src/protocols/toolbox/PoseMetricCalculators/NeighborhoodByDistanceCalculator.cc
/// @brief NeighborhoodByDistanceCalculator can determine all the neighbors of group of residues within a certain distance.  It uses NeighborsByDistanceCalculator objects to do this.
/// @author Steven Lewis


//Unit headers
#include <protocols/toolbox/PoseMetricCalculators/NeighborhoodByDistanceCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/NeighborsByDistanceCalculator.hh>

#include <core/pose/Pose.hh>

#include <core/util/MetricValue.hh>
#include <core/pose/metrics/CalculatorFactory.hh>

//Utility headers
//#include <core/options/option.hh>
//#include <core/types.hh>
#include <core/util/Tracer.hh>
#include <utility/exit.hh>
#include <utility/string_util.hh>

// option key includes
#include <core/options/keys/packing.OptionKeys.gen.hh>

//C++ headers
//#include <set>

static core::util::Tracer TR("protocols.toolbox.PoseMetricCalculators.NeighborhoodByDistanceCalculator");
using core::util::Error;
using core::util::Warning;

namespace protocols{
namespace toolbox {
namespace PoseMetricCalculators {

///@details this constructor checks that these calculators exists and fills central_residues_ from them
NeighborhoodByDistanceCalculator::NeighborhoodByDistanceCalculator( std::set< std::string > calc_names )
	: parent(), num_neighbors_(0), calc_names_(calc_names) {

	//iterate through the set in each calculator
	for( std::set< std::string >::const_iterator it(calc_names_.begin()), end(calc_names_.end()); it != end; ++it){
		using namespace core::pose::metrics;
		if( CalculatorFactory::Instance().check_calculator_exists( *it ) ){
			//if calculator exists, get its pointer and cast into the expected type
			PoseMetricCalculatorOP maybe_subcalc(CalculatorFactory::Instance().retrieve_calculator(*it));
			NeighborsByDistanceCalculatorOP subcalc(dynamic_cast< NeighborsByDistanceCalculator * >( maybe_subcalc.get() ));
			if( subcalc ) {
				central_residues_.insert(subcalc->central_residue()); //if of the right type, pick out its data
			} else { //if the wrong type, abort
				Error() << "NeighborhoodByDistanceCalculator: calculator " << *it << " is not a NeighborsByDistanceCalculator object!";
				utility_exit();
			}
		} else { //if it does not exist at all, abort
			Error() << "NeighborhoodByDistanceCalculator: calculator " << *it << " does not exist, you must create and register NeighborsByDistanceCalculator objects yourself to use this constructor";
			utility_exit();
		}
	}
		runtime_assert(central_residues_.size() == calc_names_.size());
}

///@details this calculator takes central_residues and attempts to create calculators for them.  It is not the safer constructor to use but can be convenient.  It is possible to trip this constructor up by having pre-existing calculators with the precise names this calculator is looking for.
NeighborhoodByDistanceCalculator::NeighborhoodByDistanceCalculator( std::set< core::Size > central_residues )
	: parent(), central_residues_(central_residues), num_neighbors_(0)
{
	//iterate through central_residues, looking for pre-existing NeighborByDistanceCalculators and creating as needed
	for( std::set< core::Size >::const_iterator it(central_residues_.begin()), end(central_residues_.end());
			 it != end; ++it){
		//figure out what the string name would be
		std::string const calc_name(name_from_res(*it));

		using namespace core::pose::metrics;
		if( CalculatorFactory::Instance().check_calculator_exists( calc_name ) ){

			//get its pointer, attempt to cast
			PoseMetricCalculatorOP maybe_subcalc(CalculatorFactory::Instance().retrieve_calculator(calc_name));
			NeighborsByDistanceCalculatorOP subcalc(dynamic_cast< NeighborsByDistanceCalculator * >( maybe_subcalc.get() ));
			if( subcalc ) { //if cast succeeds, check its data
				core::Real dist_cutoff(subcalc->dist_cutoff());
				core::Size central_residue(subcalc->central_residue());

				Warning() << "In NeighborhoodByDistanceCalculator constructor, subcalculator " << calc_name
									<< " already exists with the parameters: cutoff " << dist_cutoff
									<< " and central_residue " << central_residue << std::endl;

				if( central_residue != *it ){ //if data does not match, complain
					utility_exit_with_message("central_residues don't match; pick calculator names more carefully; aborting");
				} else { //if data matches, use it
					Warning() << "This appears compatible with the requested calculator and will be used." << std::endl;
				}
			} else { //if the pointer cast fails, then we have a name conflict; abort
				Error() << "NeighborhoodByDistanceCalculator: calculator " << calc_name << " pre-exists but is not a NeighborsByDistanceCalculator object!";
				utility_exit();
			}
		} else { //else we make a new one
			core::pose::metrics::CalculatorFactory::Instance().register_calculator( calc_name, new NeighborsByDistanceCalculator( *it ) );
		}

		calc_names_.insert(calc_name);
	}//for the central_residues
	runtime_assert(central_residues_.size() == calc_names_.size());
}

NeighborhoodByDistanceCalculator::NeighborhoodByDistanceCalculator( NeighborhoodByDistanceCalculator const & calculator ) : parent(), central_residues_(calculator.central_residues()), calc_names_(calculator.calc_names())
{}

std::string NeighborhoodByDistanceCalculator::name_from_res( core::Size const res ){
	return ( "NeighborhoodByDistanceCalculator-associated_NeighborsByDistanceCalculator_centered_on_" + utility::to_string(res));
}

core::pose::metrics::PoseMetricCalculatorOP NeighborhoodByDistanceCalculator::clone() const
{ return new NeighborhoodByDistanceCalculator(*this); }

void
NeighborhoodByDistanceCalculator::lookup(
  std::string const & key,
  core::util::MetricValueBase * valptr
) const
{
	//emacs is having a hissy fit about the tab whitespace in this function (choking on the (<<< >>*>())->() combo), so don't freak out if there are spaces instead of tabs and/or it displays poorly...

	if ( key == "central_residues" ) {
		core::util::check_cast( valptr, &central_residues_, "central_residues expects to return a std::set< core::Size >" );
		(static_cast<core::util::MetricValue< std::set< core::Size > >* > (valptr))->set( central_residues_ );

  } else if ( key == "calc_names" ) {
	  core::util::check_cast( valptr, &calc_names_, "central_residues expects to return a std::set< std::string >" );
		(static_cast<core::util::MetricValue< std::set< std::string > > *>(valptr))->set( calc_names_ );

  } else if ( key == "num_neighbors" ) {
	  core::util::check_cast( valptr, &num_neighbors_, "num_neighbors expects to return a core::Size" );
		(static_cast<core::util::MetricValue<core::Size> *>(valptr))->set( num_neighbors_ );

  } else if ( key == "neighbors" ) {
	  core::util::check_cast( valptr, &neighbors_, "neighbors expects to return a std::set< core::Size >" );
		(static_cast<core::util::MetricValue< std::set< core::Size > > *>(valptr))->set( neighbors_ );

  } else {
	  core::util::Error() << "NeighborhoodByDistanceCalculator cannot compute metric " << key << std::endl;
		utility_exit();
 }

} //lookup

std::string
NeighborhoodByDistanceCalculator::print( std::string const & key ) const
{
	using namespace core::options;
  if ( key == "central_residues" ) {
		std::string const spacer( option[ OptionKeys::packing::print_pymol_selection].value() ? "+" : " ");
		std::string res_string("");
		for( std::set< core::Size >::const_iterator it(central_residues_.begin()), end(central_residues_.end()); it != end; ++it)
			res_string += utility::to_string(*it) + spacer;
		return res_string;

  } else if ( key == "calc_names" ) {
		std::string calc_string("");
		for( std::set< std::string >::const_iterator it(calc_names_.begin()), end(calc_names_.end()); it != end; ++it)
			calc_string += *it + " ";
		return calc_string;

  } else if ( key == "num_neighbors" ) {
    return utility::to_string( num_neighbors_ );

  } else if ( key == "neighbors" ) {
		using namespace core::options; //this lets you get + or (space) as spacer
		std::string const spacer( option[ OptionKeys::packing::print_pymol_selection].value() ? "+" : " ");
		std::string nbrs_string("");
		for( std::set< core::Size >::const_iterator it(neighbors_.begin()), end(neighbors_.end()); it != end; ++it)
			nbrs_string += utility::to_string(*it) + spacer;
    return nbrs_string;

  }//else
  core::util::Error() << "NeighborhoodByDistanceCalculator cannot compute metric " << key << std::endl;
  utility_exit();
  return "";
} //print

void
NeighborhoodByDistanceCalculator::recompute( core::pose::Pose const & pose )
{
	//clear old data
	neighbors_.clear();
	num_neighbors_ = 0;

	core::util::MetricValue< std::set<core::Size> > sub_neighbors;

	//iterate through the set in each calculator
	for( std::set< std::string >::const_iterator it(calc_names_.begin()), end(calc_names_.end()); it != end; ++it){
		pose.metric( *it, "neighbors", sub_neighbors);

		for( std::set< core::Size >::const_iterator nbr_it(sub_neighbors.value().begin()), nbr_end(sub_neighbors.value().end()); nbr_it != nbr_end; ++nbr_it){
			neighbors_.insert(*nbr_it); //insert does nothing if element is already present
		}//for each position
	}//for each subcalculator

	num_neighbors_ = neighbors_.size();

	return;
} //recompute

} //namespace PoseMetricCalculators
} //namespace toolbox
} //namespace protocols
