// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file /src/protocols/toolbox/PoseMetricCalculators/NeighborhoodByDistanceCalculator.hh
/// @brief NeighborhoodByDistanceCalculator can determine all the neighbors of group of residues within a certain distance.  It uses NeighborsByDistanceCalculator objects to do this.
/// @author Steven Lewis

#ifndef INCLUDED_protocols_toolbox_PoseMetricCalculators_NeighborhoodByDistanceCalculator_HH
#define INCLUDED_protocols_toolbox_PoseMetricCalculators_NeighborhoodByDistanceCalculator_HH

//Unit headers
#include <core/pose/metrics/PoseMetricCalculatorBase.hh>
#include <protocols/toolbox/PoseMetricCalculators/NeighborhoodByDistanceCalculator.fwd.hh>

//
#include <core/pose/Pose.fwd.hh>
#include <core/util/MetricValue.fwd.hh>

//Utility headers
#include <core/options/option.hh>
#include <core/types.hh>


//C++ headers
#include <set>

namespace protocols{
namespace toolbox {
namespace PoseMetricCalculators {

///@details this calculator determines the number and resids of residues within X angstroms of a group of given residues.  Its intended purpose is the backend for a TaskOperation that allows one to construct a PackerTask based on neighborhoods around a set of particular residues.  (It combines with NeighborsByDistanceCalculator for that purpose).  It can return the string names of its substituent NeighborsByDistanceCalculators, its set of central residues, the total count of their neighbors as determined by the sub-calculators (inclusive of the central residues), and the identities of those neighbors.
class NeighborhoodByDistanceCalculator : public core::pose::metrics::StructureDependentCalculator {

public:
	typedef core::pose::metrics::StructureDependentCalculator parent;

  ///@brief ctor for pre-existing NeighborsByDistanceCalculators
  NeighborhoodByDistanceCalculator( std::set< std::string > calc_names );

	///@brief ctor for positions - automatically creates underlying calculators
	NeighborhoodByDistanceCalculator( std::set< core::Size > central_residues );

	///@brief copy ctor
	NeighborhoodByDistanceCalculator( NeighborhoodByDistanceCalculator const & calculator );

	virtual core::pose::metrics::PoseMetricCalculatorOP clone() const;

	//accessors for constant/input data
	///@brief return central residues set
	std::set< core::Size > const & central_residues() const { return central_residues_; }

	///@brief return names of subcalculators
	std::set< std::string > const & calc_names() const { return calc_names_; }

protected:
  virtual void lookup( std::string const & key, core::util::MetricValueBase * valptr ) const;
  virtual std::string print( std::string const & key ) const;
  virtual void recompute( core::pose::Pose const & pose );

private:

	///@brief utility function: creates a (hopefully unique) name for a calc from which residue it is
	std::string name_from_res( core::Size const res );

  ///@brief whose neighbors are we finding?
	std::set< core::Size > central_residues_;
  ///@brief the number of neighbors, INCLUSIVE of central residues
  core::Size num_neighbors_;
  ///@brief the set of neighbors, INCLUSIVE of central_residues
  std::set< core::Size > neighbors_;
	///@brief underlying calculators
	std::set< std::string > calc_names_;
};


} // namespace PoseMetricCalculators
} // namespace toolbox
} // namespace protocols

#endif //INCLUDED_protocols_toolbox_PoseMetricCalculators_NeighborhoodByDistanceCalculator_HH

