// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file TopologyBroker
/// @brief  top-class (Organizer) of the TopologyBroker mechanism
/// @detailed responsibilities:
/// @author Oliver Lange

// Unit Headers
#include <protocols/topology_broker/ConstraintClaimer.hh>

// Package Headers
#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/util.hh>


//#include <protocols/topology_broker/DofClaim.hh>

// Project Headers
#include <core/pose/Pose.hh>
// #include <core/kinematics/MoveMap.hh>
// #include <core/fragment/FragSet.hh>
// #include <protocols/abinitio/FragmentMover.hh>


#include <core/id/Exceptions.hh>

// ObjexxFCL Headers

// Utility headers
#include <utility/excn/Exceptions.hh>

//#include <utility/io/izstream.hh>
//#include <utility/io/ozstream.hh>
//#include <utility/io/util.hh>
#include <core/util/Tracer.hh>
#include <core/options/option.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
//// C++ headers
#include <fstream>

// option key includes

static core::util::Tracer tr("protocols.topo_broker",core::util::t_info);
//static numeric::random::RandomGenerator RG(18828234);

namespace protocols {
namespace topology_broker {

using namespace core;
using namespace scoring::constraints;
ConstraintClaimer::ConstraintClaimer() :
	filename_( "NO_FILE"),
	tag_( "NO_TAG" ),
	constraints_( NULL ),
	bCentroid_( true ),
	bFullatom_( false ),
	bCmdFlag_( false )
{}

ConstraintClaimer::ConstraintClaimer( std::string filename, std::string tag ) :
	filename_( filename ),
	tag_( tag ),
	constraints_( NULL ),
	bCentroid_( true ),
	bFullatom_( false ),
	bCmdFlag_( false )
{}

ConstraintClaimer::ConstraintClaimer( bool CmdFlag )
	:	filename_( "" ),
		tag_( "" ),
		constraints_( NULL ),
		bCentroid_( true ),
		bFullatom_( false ),
		bCmdFlag_( CmdFlag )
{
	runtime_assert( CmdFlag );
}

void ConstraintClaimer::generate_claims( DofClaims& /*new_claims*/ ) {
}

void ConstraintClaimer::new_decoy() {
	using namespace options;
	using namespace OptionKeys;
	tr.Debug << "ConstraintClaimer::new_decoy: cst-modus: " << ( bFullatom_ ? " fullatom " : "no fullatom" )
					 <<  ( bCentroid_ ? " centroid " : " no centroid " )
					 << std::endl;
	if ( bCmdFlag_ && option[ constraints::cst_file ].user() ) {
		// reads and sets constraints -- this might be different each time we call this function
		std::string old_filename = filename_;
		filename_ = core::scoring::constraints::get_cst_file_option();
		if ( old_filename != filename_ ) constraints_ = NULL;
	}
}

void ConstraintClaimer::add_constraints( core::pose::Pose& pose ) {
	bool fullatom( pose.is_fullatom() );
	if ( fullatom && !bFullatom_ ) return;
	if ( !fullatom && !bCentroid_ ) return;
	tr.Debug << "add constraints "<< tag_ << std::endl;
	if ( constraints_ ) {
		tr.Debug << " constraint set is currently for a " <<( constraint_ref_pose_.is_fullatom() ? "fullatom" : "centroid") << " pose "
						 << "\n will now remap them to a " <<  (fullatom ? "fullatom" : "centroid") << " pose" << std::endl;
	}
	std::string const new_sequence ( pose.annotated_sequence( true ) );
	if ( !constraints_ || sequence_ != new_sequence ) {
		tr.Info << " read constraints from " << filename_ << "\n for pose " << new_sequence << "..." << std::endl;
		constraints_ = ConstraintIO::get_instance()->read_constraints( filename_, new ConstraintSet, pose );
		sequence_ = new_sequence;
	} else {
		ConstraintSetOP new_cst(NULL);
		try {
			new_cst = constraints_->remapped_clone( constraint_ref_pose_, pose );
		} catch( core::id::EXCN_AtomNotFound& excn ) {
			tr.Error << "[ERROR] failed attempt to add constraints to the "
							 << (fullatom ? "fullatom" : "centroid") << " pose" << std::endl;
			tr.Error << excn << std::endl;
			if ( tr.Debug.visible() ) {
				pose.dump_pdb("new_pose_failed_constraints.pdb");
				constraint_ref_pose_.dump_pdb("cst_ref_pose.pdb");
			}
			constraints_->show_definition( tr.Error, constraint_ref_pose_ );
			tr.Error << std::endl;
			tr.Error << " try to recover by reading in original constraints from " << filename_ << "\n for pose " << new_sequence << "..." << std::endl;
			new_cst = ConstraintIO::get_instance()->read_constraints( filename_, new ConstraintSet, pose );
		}

		constraints_ = new_cst;
	}
	constraint_ref_pose_ = pose;
	pose.add_constraints( constraints_->get_all_constraints() );
}

bool ConstraintClaimer::read_tag( std::string tag, std::istream& is ) {
	if ( tag == "file" || tag == "FILE" || tag == "CST_FILE" ) {
		is >> filename_;
	} else if ( tag == "NO_CENTROID" ) {
		bCentroid_ = false;
	} else if ( tag == "FULLATOM" ) {
		bFullatom_ = true;
	} else if ( tag == "CMD_FLAG" ) {
		bCmdFlag_ = true;
	} else return Parent::read_tag( tag, is );
	return true;
}


} //topology_broker
} //protocols
