// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file TopologyBroker
/// @brief  top-class (Organizer) of the TopologyBroker mechanism
/// @detailed responsibilities:
/// @author Oliver Lange

// Unit Headers
#include <protocols/topology_broker/RigidChunkClaimer.hh>

// Package Headers
#include <protocols/topology_broker/DofClaim.hh>
#include <protocols/topology_broker/Exceptions.hh>

// Project Headers
#include <core/conformation/Conformation.hh>
#include <core/pose/Pose.hh>
#include <core/fragment/BBTorsionSRFD.hh>
#include <core/fragment/Frame.hh>
#include <core/fragment/JumpingFrame.hh>
#include <core/conformation/util.hh>

#include <core/sequence/SequenceAlignment.hh>
#include <protocols/comparative_modeling/util.hh>

#include <core/chemical/util.hh>

#include <protocols/loops/LoopClass.hh>
#include <protocols/jumping/JumpSample.hh>
#include <protocols/jumping/util.hh>

#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/ThreadingJob.hh>

#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/util.hh>
#include <core/io/pdb/pose_io.hh>
// ObjexxFCL Headers

// Utility headers
//#include <utility/io/izstream.hh>
//#include <utility/io/ozstream.hh>
//#include <utility/io/util.hh>
#include <utility/excn/Exceptions.hh>
#include <core/util/Tracer.hh>
#include <numeric/random/random.hh>
#include <numeric/numeric.functions.hh>
#include <core/options/util.hh>
#include <core/options/after_opts.hh>
#include <core/options/keys/loops.OptionKeys.gen.hh>

//// C++ headers
#include <fstream>

// option key includes

static core::util::Tracer tr("protocols.topo_broker",core::util::t_info);
static numeric::random::RandomGenerator RG(188428234);

namespace protocols {
namespace topology_broker {

using namespace core;


//helper function
protocols::loops::Loops generate_rigid_from_alignment( pose::Pose query_pose, core::sequence::SequenceAlignment const& align, Size min_loop_size ) {
	using core::Size;
	using namespace core::options;

	loops::Loops loops = comparative_modeling::loops_from_alignment( query_pose.total_residue(), align, min_loop_size );

	// randomly grow loops by N residues (4 is a good amount)
	if( option[ OptionKeys::loops::random_grow_loops_by ]()  > 0.0 ){
		tr << "Growing loops: " << std::endl;
		tr << loops << std::endl;
		loops.grow_all_loops( query_pose.total_residue(), option[ OptionKeys::loops::random_grow_loops_by ]() );
		tr << "--------- " << std::endl;
		tr << loops << std::endl;

		// make sure loops are ok:
		loops.verify_against( query_pose );
	}


// 	//grow loops by 1 residue since rigid - chunk needs coords before and after rigid region to define torsions
// 	loops::Loops grown_loops;
// 	for ( loops::Loops::const_iterator it = loops.begin(), eit = loops.end();
// 				it != eit; ++it ) {
// 		grown_loops.add_loop( std::max( 1, (int) it->start()-1), it->stop()+1, 0, it->skip_rate() );
// 	}
	return loops.invert( query_pose.total_residue() );

}


RigidChunkClaimer::RigidChunkClaimer() : bExclusive_( true ), bUseInputPose_( true ) {}

RigidChunkClaimer::RigidChunkClaimer( pose::Pose const& input_pose, loops::Loops rigid ) :
	input_pose_( input_pose ),
	centroid_input_pose_( input_pose ),
	rigid_core_( rigid ),
	bExclusive_( true ),
	bUseInputPose_( true )
{
	if ( centroid_input_pose_.total_residue() && centroid_input_pose_.is_fullatom() ) chemical::switch_to_residue_type_set(	centroid_input_pose_, chemical::CENTROID );
	//coordinate-cst ---> bExclusive can be switched off.
}

void RigidChunkClaimer::set_defaults() {
	Parent::set_defaults();
	bUseInputPose_ = true;
	bUseThreadingJobLoops_ = false;
	min_loop_size_ = 0;
}

void RigidChunkClaimer::receive_message( ClaimerMessage& cm ) {
	if ( typeid( cm ) == typeid( CM_SuggestFixResidue ) ) {
		CM_SuggestFixResidue& msg = dynamic_cast< CM_SuggestFixResidue& >( cm );

		//find good residue
		if ( !current_rigid_core_.size() ) return;
		msg.good_fix_pos_ =	current_rigid_core_.begin()->start() + numeric::nint( 0.5 * current_rigid_core_.begin()->size() ) - 1;
		msg.received_by_me( this );
	}
}

bool RigidChunkClaimer::read_tag( std::string tag, std::istream& is ) {
	if ( tag == "pdb" || tag == "PDB" || tag == "pdb:" || tag == "PDB_FILE" ) {
		std::string file;
		is >> file;
		core::io::pdb::pose_from_pdb( input_pose_,
			*core::chemical::ChemicalManager::get_instance()->residue_type_set( core::chemical::FA_STANDARD ),
			file );
		runtime_assert( input_pose_.is_fullatom() );
	} else if ( tag == "REGION" ) {
		rigid_core_.read_stream_to_END( is, false /*no strict checking */, type(), "RIGID" );
	} else if ( tag == "region_file" || tag == "REGION_FILE" ) {
		std::string file;
		is >> file;
		rigid_core_.read_loop_file( file, false /*no strict looprlx checking*/, "RIGID" );  // <==
	} else if ( tag == "NO_USE_INPUT_POSE" ) {
		bUseInputPose_ = false;
	} else if ( tag == "USE_THREADING_LOOPS" ) {
		bUseThreadingJobLoops_ = true;
	} else if ( tag == "MIN_LOOP_SIZE" ) {
		is >> min_loop_size_;
	} else if ( tag == "KEEP_FLEXIBLE" ) {
		bExclusive_ = false;
	}
	else return Parent::read_tag( tag, is );
	return true;
}

void RigidChunkClaimer::init_after_reading() {
	tr.Debug << type() << " initialized with input_pdb: " << input_pose_.sequence() << " and regions " << rigid_core_ << std::endl;
	centroid_input_pose_=input_pose_;
	if ( centroid_input_pose_.total_residue() && centroid_input_pose_.is_fullatom() ) chemical::switch_to_residue_type_set(	centroid_input_pose_, chemical::CENTROID );
}

void RigidChunkClaimer::select_parts() {
  current_rigid_core_.clear();
  int attempts = 0;
	if ( rigid_core_.size() == 0 ) return;//nothing to select
  while ( current_rigid_core_.size() == 0 && attempts < 50 ) {
		++attempts;
    for ( loops::Loops::const_iterator it = rigid_core_.begin(), eit = rigid_core_.end();
					it != eit; ++it ) {
      if ( RG.uniform() >= it->skip_rate() )  {
				current_rigid_core_.push_back( *it );
      }
    }
  }
  if ( current_rigid_core_.size() == 0 ) {
    current_rigid_core_ = rigid_core_;
  }
}

///@detail generate exclusive backbone claims for each residue of the rigid-chunk
/// jumps are not exclusive and are added later in final_claims --- want to reuse other jumps if possible
void RigidChunkClaimer::generate_claims( DofClaims& new_claims ) {
	using namespace loops;
	tr.Trace << "rigid chunk -- generate claim " << std::endl;
	//stochastically select rigid_core ( if skip-rate is set >0, otherwise all loops are selected )
	tr.Trace << "region selected: " << current_rigid_core_ << std::endl;
	//	new_claims.push_back( new CutBiasClaim( *this ) ); we don't need this claim type --- always call manipulate_cut_bias
	for ( Loops::const_iterator loop_it = current_rigid_core_.begin(); loop_it != current_rigid_core_.end(); ++loop_it ) {
		for ( Size pos = loop_it->start(); pos <= loop_it->stop(); ++pos ) {
			new_claims.push_back( new BBClaim( this, pos, DofClaim::EXCLUSIVE ) );
		}
	}
}

void RigidChunkClaimer::new_decoy() {
	select_parts(); //set members current_XXX
	current_jump_calculator_ = new JumpCalculator( current_rigid_core_ );
	current_jumps_.clear();
}

void RigidChunkClaimer::new_decoy( core::pose::Pose const& pose ) {
	//should we read input structures dofs?
	tr.Debug << "New decoy:" << std::endl;
	if ( bUseInputPose_ ) {
		input_pose_ = pose;
		centroid_input_pose_=input_pose_;
		if ( centroid_input_pose_.total_residue() && centroid_input_pose_.is_fullatom() ) chemical::switch_to_residue_type_set(	centroid_input_pose_, chemical::CENTROID );

		// use loops from ThreadingJob ???
		if ( bUseThreadingJobLoops_ ) {
			using namespace protocols::jd2;
			ThreadingJobCOP job = dynamic_cast< ThreadingJob const*  >( JobDistributor::get_instance()->current_job()->inner_job().get() );
			if ( job ) {
				tr.Debug << "------------------found ThreadingJob ... get loops " << std::endl;
				rigid_core_ = generate_rigid_from_alignment( input_pose_, job->alignment(), min_loop_size_ );
			}
		} //bUseThreading
		tr.Debug << "RigidChunk defined for " << rigid_core_ << std::endl;
	}
	new_decoy();
	//
}

bool RigidChunkClaimer::allow_claim( DofClaim const& foreign_claim ) {
	if ( foreign_claim.owner() == this ) return true; // always allow your own claims!

	// check foreign claim
	if ( foreign_claim.type() == DofClaim::BB && current_rigid_core_.is_loop_residue( foreign_claim.pos( 1 ) ) ) {
		if ( bExclusive_ ) { 	// if we want exclusive claim this is not acceptable
			return false;
		} else {
			// allow only the weakest claim. We want to initialize ourselves... don't know if we need to be so restrictive!
			if ( !(foreign_claim.right() == DofClaim::CAN_INIT) ) return false;
		}
	} // DofClaim::BB

	if ( foreign_claim.type() == DofClaim::JUMP ) {
		runtime_assert( current_jump_calculator_ );
		if ( current_jump_calculator_->irrelevant_jump( foreign_claim.pos( 1 ), foreign_claim.pos( 2 ) ) ) return true;
		if ( !current_jump_calculator_->good_jump( foreign_claim.pos( 1 ), foreign_claim.pos( 2 ) ) ) {
			return false;
		} else if ( bExclusive_ ) { // but probably a good jump --- since it has a physical reason.
			//reclaim the claim
			current_jumps_.push_back( new JumpClaim( this, foreign_claim.pos( 1 ), foreign_claim.pos( 2 ), DofClaim::EXCLUSIVE ) );
			return false;
		} else {
			current_jumps_.push_back( foreign_claim.clone() ); //ok - remember this jump, since it connects rigid1 and rigid2
		}
	} // DofClaim::JUMP

	if ( foreign_claim.type() == DofClaim::CUT) {
		for ( loops::Loops::const_iterator region = current_rigid_core_.begin(); region != current_rigid_core_.end(); ++region ) {
			if (foreign_claim.pos( 1 ) >= region->start() && foreign_claim.pos( 1 ) < region->stop() )  // cut claim can be at the chunk end
			 return false; // no cuts within our rigid-core boundaries
		}
	}

	return true;
} // RigidChunkClaimer::allow_claim()

bool RigidChunkClaimer::accept_declined_claim( DofClaim const& was_declined ) {
	tr.Warning << "[WARNING] RigidChunkClaimer couldn't get " << was_declined << std::endl;
	return false; // no tolerance here --- don't accept any of these
}

void RigidChunkClaimer::finalize_claims( DofClaims& new_claims ) {
	DofClaims extra_jumps;
	current_jump_calculator_->generate_rigidity_jumps( this, extra_jumps );
	std::copy( extra_jumps.begin(), extra_jumps.end(), back_inserter( current_jumps_ ) );
	std::copy( extra_jumps.begin(), extra_jumps.end(), back_inserter( new_claims ) );
}

// 	// this could be made faster by getting the composite rotation and translation
// void superimpose_chain( core::pose::Pose& pose, core::pose::Pose& input_pose, loops::Loops region ) {
// 	if ( region.begin() != region.end() ) {
// 		void start_pos = region.begin()->start();
// 		kinematics::Stub src_stub, rot_stub;
// 		pose.residue( start_pos ).get_stubs_to_orient_onto_residue( input_pose.residue( start_pos ), src_stub, rot_stub );
// 		for ( loops::Loops::const_iterator region = rigid_core_.begin(); region != rigid_core_.end(); ++region ) {
// 			for ( Size pos = region->start(); pos<=region->stop(); ++pos ) {
// 				for ( Size j=1; j<= input_pose.residue_type( pos ).natoms(); ++j ) { // use residue_type to prevent internal coord update
// 					Vector const old_xyz( input_pose.residue( pos ).atoms()[j].xyz() );
// 					Vector const new_xyz( src_stub.local2global( rot_stub.global2local( old_xyz ) ) );
// 					pose.residue( pos ).atoms()[i].xyz( new_xyz );
// 					//or 					mod_pose.set_xyz( id::AtomID( j,i), x2 );
// 				}
// 			}
// 		}

// 	}
// }

void copy_internal_coords( pose::Pose& pose, pose::Pose const& ref_pose, loops::Loops core ) {
	//core::kinematics::AtomTree const & atomtree = pose.atom_tree();
	//
	///fpd if there are post modifications to pose (not in ref_pose), we can't just copy ref_pose->pose
	///fpd    instead ... make xyz copy in rigid regions
	for ( loops::Loops::const_iterator region = core.begin(); region != core.end(); ++region ) {
		for (Size i=region->start(); i<=region->stop(); ++i) {
			core::conformation::Residue const &rsd_i = ref_pose.residue(i);
			pose.replace_residue ( i , rsd_i , false );
		}
	}

	///fpd fix connections
	///fpd this requires that the input pose have one flanking residue on each side of each region
	for ( loops::Loops::const_iterator region = core.begin(); region != core.end(); ++region ) {
	  Size loop_start = region->start();
		Size loop_stop  = region->stop();

		bool lower_connect = ( loop_start > 1
		                       && !pose.residue(loop_start).is_lower_terminus()
		                       && !pose.fold_tree().is_cutpoint( loop_start-1 ) );
		bool upper_connect = ( loop_stop < pose.total_residue()
		                       && !pose.residue(loop_stop).is_upper_terminus()
		                       && !pose.fold_tree().is_cutpoint( loop_stop ) );

		if ( lower_connect ) {
			tr.Trace << "fixing lower connection for " << loop_start << std::endl;
			core::conformation::Residue const & prev_rsd( ref_pose.residue( loop_start-1 ) );
			core::conformation::Residue const &      rsd( ref_pose.residue( loop_start ) );
			core::Size const nbb_prev( prev_rsd.n_mainchain_atoms() );
			core::id::AtomID bbM1   ( prev_rsd.mainchain_atom( nbb_prev-2 ),  loop_start-1 );
			core::id::AtomID bb0    ( prev_rsd.mainchain_atom( nbb_prev-1 ),  loop_start-1 );
			core::id::AtomID bb1    ( prev_rsd.mainchain_atom( nbb_prev   ),  loop_start-1 );
			core::id::AtomID bb2    (      rsd.mainchain_atom(        1   ),  loop_start   );
			core::id::AtomID bb3    (      rsd.mainchain_atom(        2   ),  loop_start   );
			core::id::AtomID bb4    (      rsd.mainchain_atom(        3   ),  loop_start   );

			tr.Trace << "nbb_prev " << nbb_prev << std::endl;

 			core::conformation::Residue const & ref_resi = ref_pose.residue( loop_start );
  			tr.Trace << "mainchain torsion: ref: " << ref_resi.mainchain_torsion( 1 ) << " atom-tree: "
 								 << ref_pose.conformation().torsion_angle( bb1, bb2, bb3, bb4 ) << std::endl;

 			core::conformation::Residue const & resi = pose.residue( loop_start );
  			tr.Trace << "mainchain torsion (before): conf: " << resi.mainchain_torsion( 1 ) << " atom-tree: "
 								 << pose.conformation().torsion_angle( bb1, bb2, bb3, bb4 ) << std::endl;
			//olli these angles were not coming thru. now it seems to work, and even the underlying atom-tree angle is updated.
// 			pose.set_phi( loop_start, ref_pose.phi( loop_start ) );
// 			pose.set_psi( loop_start-1, ref_pose.psi( loop_start-1 ) );
// 			pose.set_omega( loop_start-1, ref_pose.omega( loop_start-1 ) );
						pose.conformation().set_bond_length( bb1, bb2, ref_pose.conformation().bond_length( bb1, bb2 ) );
						pose.conformation().set_bond_angle ( bb0, bb1, bb2, ref_pose.conformation().bond_angle( bb0, bb1, bb2 ) );
						pose.conformation().set_bond_angle ( bb1, bb2, bb3, ref_pose.conformation().bond_angle( bb1, bb2, bb3 ) );
						pose.conformation().set_torsion_angle( bbM1, bb0, bb1, bb2, ref_pose.conformation().torsion_angle( bbM1, bb0, bb1, bb2 ) );
						pose.conformation().set_torsion_angle( bb0, bb1, bb2, bb3, ref_pose.conformation().torsion_angle( bb0, bb1, bb2, bb3 ) );
						pose.conformation().set_torsion_angle( bb1, bb2, bb3, bb4, ref_pose.conformation().torsion_angle( bb1, bb2, bb3, bb4 ) );

			core::conformation::Residue const & new_resi = pose.residue( loop_start ); //this should trigger update of coords and torsions
 			tr.Trace << "mainchain torsion (after): conf: " << new_resi.mainchain_torsion( 1 ) << " atom-tree: "
								 << pose.conformation().torsion_angle( bb1, bb2, bb3, bb4 ) << std::endl;

			//lin Listen, we have to correct the O position after move the Nn+1, and correct the H pistion after move
			// core::id::NamedAtomID bbO( "O",  loop_start-1 );
// 			core::id::NamedAtomID bbH( "H",  loop_start );
// 			pose.conformation().set_torsion_angle( bb2, bb0, bb1, bbO, ref_pose.conformation().torsion_angle( bb2, bb0, bb1, bbO ) );  //Nn+1,Can,Cn,On
// 			pose.conformation().set_torsion_angle( bb1, bb3, bb2, bbH, ref_pose.conformation().torsion_angle( bb1, bb3, bb2, bbH ) );  //Cn,Can+1,Nn+1,Hn+1
		} else {
			tr.Trace << "NOT fixing lower connection for " << loop_start << std::endl;
		}

		if ( upper_connect ) {
			core::conformation::Residue const & next_rsd( ref_pose.residue( loop_stop+1 ) );
			core::conformation::Residue const &      rsd( ref_pose.residue( loop_stop ) );
			core::Size const nbb( rsd.n_mainchain_atoms() );
			core::id::AtomID bb0    (      rsd.mainchain_atom( nbb-2 ),  loop_stop   );
			core::id::AtomID bb1    (      rsd.mainchain_atom( nbb-1 ),  loop_stop   );
			core::id::AtomID bb2    (      rsd.mainchain_atom( nbb   ),  loop_stop   );
			core::id::AtomID bb3    ( next_rsd.mainchain_atom(     1 ),  loop_stop+1 );
			core::id::AtomID bb4    ( next_rsd.mainchain_atom(     2 ),  loop_stop+1 );
			core::id::AtomID bb5    ( next_rsd.mainchain_atom(     3 ),  loop_stop+1 );

			pose.conformation().set_bond_length( bb2, bb3, ref_pose.conformation().bond_length( bb2, bb3 ) );
			pose.conformation().set_bond_angle( bb1, bb2, bb3, ref_pose.conformation().bond_angle( bb1, bb2, bb3 ) );
			pose.conformation().set_bond_angle( bb2, bb3, bb4, ref_pose.conformation().bond_angle( bb2, bb3, bb4 ) );
			pose.conformation().set_torsion_angle( bb0, bb1, bb2, bb3, ref_pose.conformation().torsion_angle( bb0, bb1, bb2, bb3 ) );
			pose.conformation().set_torsion_angle( bb1, bb2, bb3, bb4, ref_pose.conformation().torsion_angle( bb1, bb2, bb3, bb4 ) );
			pose.conformation().set_torsion_angle( bb2, bb3, bb4, bb5, ref_pose.conformation().torsion_angle( bb2, bb3, bb4, bb5 ) );

// 			pose.set_phi( loop_stop+1, ref_pose.phi( loop_stop+1 ) );
// 			pose.set_psi( loop_stop, ref_pose.psi( loop_stop ) );
// 			pose.set_omega( loop_stop, ref_pose.omega( loop_stop ) );

 //lin we have to correct the O and H position after move the N,C,Ca
// 			core::id::AtomID bbO    (      rsd.atom_index("O"),  loop_stop );
// 			core::id::AtomID bbH    (      rsd.atom_index("H"),  loop_start );
// 			pose.conformation().set_torsion_angle( bb3, bb1, bb2, bbO, ref_pose.conformation().torsion_angle( bb3, bb1, bb2, bbO ) ); //Nn+1,Can,Cn,On
// 			pose.conformation().set_torsion_angle( bb2, bb4, bb3, bbH, ref_pose.conformation().torsion_angle( bb2, bb4, bb3, bbH ) );  //Cn,Can+1,Nn+1,Hn+1
		}

	}
}

void RigidChunkClaimer::initialize_dofs( core::pose::Pose& pose, DofClaims const& /* init_claims*/, DofClaims& /*failed_init_claims*/ ) {
	//need to copy coords and jumps --- if chunks were idealized no problem .... but non-idealized stuff ?
	//also take care of fullatom vs centroid...
	//	tr.Warning << "[WARNING] *** use input structure of RigidChunkClaimer --- NEEDS TO BE IDEALIZED !!! *** \n";
	// to do this without idealized :

	// get rigid-body reorientation for first residue... this must be applied to all residues in the chunk,
	// and then just set from the coordinates.

	//in a sense this is not necessary since we asked for exclusive rights for the stuff in RIGID ...
	/*	for ( DofClaims::const_iterator claim=init_claims.begin(); claim!=init_claims.end(); ++claim ) {
		if ( (*claim)->owner() != this ) continue;

		}*/
	//	superimpose_chain (pose, input_pose_, rigid_core_ );


	//need to have same number of residues for fold-tree transfer...
	// would be nice to drop this restriction but for now, fill up with missing density...

	//fpd runtime_assert( pose.total_residue() == centroid_input_pose_.total_residue() );
	//fpd   really, we just have to make sure that #residues in the input pose > index of last rigid chunk
	//fpd   (strictly greater-than since we have to have a flanking res on each side of each region)
	//fpd   we still need missing dens in the gaps (but not at c term now!)
	core::Size lastChunk=1;
	for ( loops::Loops::const_iterator it = current_rigid_core_.begin(); it!=current_rigid_core_.end(); ++it )
		lastChunk = std::max( lastChunk , it->stop() );
	runtime_assert ( lastChunk <= centroid_input_pose_.total_residue() );

	bool missing_density( false );
	//sanity check: no missing density in backbon in any of the rigid_core residues?
	for ( loops::Loops::const_iterator it = current_rigid_core_.begin(); it!=current_rigid_core_.end(); ++it ) {
		for ( Size pos = it->start(); pos <=it->stop(); ++pos ) {
			// Do we really have Sidechains ?
			// check this my making sure that no SC atom is more than 20A (?) away from CA
			numeric::xyzVector< core::Real> ca_pos = input_pose_.residue( pos ).atom("CA").xyz();
			numeric::xyzVector< core::Real> n_pos = input_pose_.residue( pos ).atom("N").xyz();
			numeric::xyzVector< core::Real> o_pos = input_pose_.residue( pos ).atom("O").xyz();
			if ( ( n_pos - ca_pos).length() > 20 || ( ( n_pos - o_pos ).length() > 20 ) ) {
				tr.Error << "missing backcone in rigid-chunk at" << pos << std::endl;
				missing_density = true;
			}
		}
	}
	if ( missing_density ) throw utility::excn::EXCN_BadInput( " missing density in backbone of rigid-chunk... this is not good check you LOOP definition ");
	//centroid_input_pose_.fold_tree( pose.fold_tree() );
	copy_internal_coords( pose, centroid_input_pose_, current_rigid_core_ );

// 	using namespace core::fragment;
// 	for ( loops::Loops::const_iterator region = rigid_core_.begin(); region != rigid_core_.end(); ++region ) {
// 		Frame steal_frame( region->start(), region->size(), new core::fragment::BBTorsionSRFD );
// 		steal_frame.steal( input_pose_ );
// 		steal_frame.apply( 1, pose );
// 	}

// 	for ( DofClaims::const_iterator jump_it = current_jumps_.begin(); jump_it!=current_jumps_.end(); ++jump_it ) {
// 		DofClaim& jump( **jump_it );
// 		Size jump_nr ( pose.fold_tree().jump_nr( jump.pos( 1 ), jump.pos( 2 ) ) );
// 		runtime_assert( jump_nr ); //XOR would be even better
// 		JumpingFrameOP steal_frame = jumping::generate_jump_frame( jump.pos( 1 ), jump.pos( 2 ), false );
// // 		JumpingFrameOP steal_frame = ( up_down ?
// // 			jumping::generate_jump_frame( jump.pos( 1 ), jump.pos( 2 ), false ) :
// // 			jumping::generate_jump_frame( jump.pos( 2 ), jump.pos( 1 ), false ) );
// 		steal_frame->steal( input_pose_ );
// 		steal_frame->apply( 1, pose );
// 	}
}

///@brief multiply your bias to this -- if its zero don't change that, i.e., multiply only
void RigidChunkClaimer::manipulate_cut_bias( utility::vector1< core::Real >& cut_bias ) {
	current_rigid_core_.transfer_to_residue_vector( cut_bias, 0.0 );
}


void RigidChunkClaimer::switch_to_fullatom( core::pose::Pose& pose , utility::vector1< bool > bNeedToRepack ) const {
	loops::Loops const& region( current_rigid_core_ );
	// copy sidechain torsions from input pose
	tr.Debug << "copy side chains for residues with * / missing density residues with - ";
	for ( loops::Loops::const_iterator it = region.begin(); it!=region.end(); ++it ) {
		for ( Size pos = it->start(); pos <=it->stop(); ++pos ) {
			bNeedToRepack[ pos ] = false; //in principle our residues don't need a repack since we have a side-chains for them.
			// Do we really have Sidechains ?
			// check this my making sure that no SC atom is more than 20A (?) away from CA
			numeric::xyzVector< core::Real> ca_pos = input_pose_.residue( pos ).atom("CA").xyz();
			for ( Size j = 1; j<=input_pose_.residue( pos ).natoms(); ++j ) {
				if ( ( ca_pos - input_pose_.residue( pos ).atom( j ).xyz()).length() > 20 ) {
					tr.Debug << "-" << pos << " ";
					bNeedToRepack[ pos ] = true;
					break; //one bad atom is enough
				}
			}
			//copy sidechains only for non-loop regions
			if ( !bNeedToRepack[ pos ] ) {
				tr.Debug <<  "*" << pos << " ";
				bool const lower_cut ( pose.residue( pos ).has_variant_type( chemical::CUTPOINT_LOWER ) );
				bool const upper_cut ( pose.residue( pos ).has_variant_type( chemical::CUTPOINT_UPPER ) );
				pose.replace_residue( pos, input_pose_.residue( pos ), true /*orient backbone*/ );
				if ( lower_cut ) chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_LOWER, pos );
				if ( upper_cut ) chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_UPPER, pos );
			}
		} //for all rigid residues
	}
	tr.Debug << " that have not moved from start-structure" << std::endl;
} //switch to fullatom
// ================================================================================
// ======================== JumpCalculator ========================================
// ================================================================================

RigidChunkClaimer::JumpCalculator::JumpCalculator( loops::Loops const& rigid ) :
	rigid_ ( rigid ), visited_( rigid.size(), 0 ), new_nr_( 1 )
{
}

bool RigidChunkClaimer::JumpCalculator::irrelevant_jump( Size pos1, Size pos2 ) {
	Size rigid1 = rigid_.loop_index_of_residue( pos1 );
	Size rigid2 = rigid_.loop_index_of_residue( pos2 );
	if ( !rigid1 || !rigid2 ) return true; //jump doesn't connect two rigid regions -- irrelevant
	return false;
}

///@brief check if this (relevant) jump is compatible with our rigid-structure
///   not on the same continuous stretch of rigid residues  ( we don't allow cuts within rigid stretches )
///   not connecting already conntected rigid stretches
/// if it connects two unconnected rigid stretches ---> its a good jump we'll keep it, update visited_
bool RigidChunkClaimer::JumpCalculator::good_jump( Size up, Size down ) {
	Size up_loop( rigid_.loop_index_of_residue( up ) );
	Size down_loop( rigid_.loop_index_of_residue( down ) );
	runtime_assert( up_loop && down_loop ); //since this would be irrelevant --- already checked.

	//don't allow jumps within same loop -- that will make a nasty cut within rigid region
	if ( up_loop == down_loop ) return false;

	// at this point rigid1 and rigid2 refer to rigid regions but not the same --- this might be useful if we need to connect rigid regions

  runtime_assert( visited_.size() >= up_loop );
  runtime_assert( visited_.size() >= down_loop );
	// jump touches unvisited regions or visited but yet disconnected regions
	if ( !visited_[ up_loop ] || !visited_[ down_loop ] || ( visited_[ up_loop ] != visited_[ down_loop ] ) ) {

		// decide upon 3 cases: both nodes unvisited, 1 node visited, both nodes visited
		// case0 : both new tag with new jump_nr
		Size visit_nr = new_nr_++;
		// case1 : both visited--> replace all higher visit_nr by lower visit_nr
		if ( visited_[ up_loop ] && visited_[ down_loop ] ) {
			Size old_visit_nr = visited_[ down_loop ]; //arbitrary choice
			visit_nr = visited_[ up_loop ];
			for ( Size i=1; i<=visited_.size(); i++ ) {
				if ( visited_[ i ] == old_visit_nr ) visited_[ i ] = visit_nr;
			}
		} else if ( visited_[ up_loop ] || visited_[ down_loop ]) {
			// case2: one already visited the other is still zero and thus neutral to addition
			visit_nr = visited_[ up_loop ] + visited_[ down_loop ];
		}
		visited_[ up_loop ] = visit_nr;
		visited_[ down_loop ] = visit_nr;
	} // jump between different regions
	return true;
}

///@detail generate a list of Jumps (Size tupels) that fix the remaining part of the chunk
void
RigidChunkClaimer::JumpCalculator::generate_rigidity_jumps( RigidChunkClaimer* THIS, DofClaims& extra_jumps ) {
	if( visited_.size() == 0 ){ // No rigid chunks ??
		return;
	}
  //now we have a connection pattern based on the jumps already present.
  //take a visited region and make it the root-reg
  Size root_reg = 0;
  for ( Size region = 1; region <= visited_.size(); region++ ) {
    if ( visited_[ region ] ) {
      root_reg = region;
      break;
    }
  }

  // if no rigid regions are yet connected, define one arbitrarily as the root-reg
  if ( root_reg == 0 ) {
    root_reg = 1;
		runtime_assert( visited_.size() > 0 );
		visited_[ root_reg ] = 1;
  }

	loops::Loops::LoopList rigid_loops = rigid_.loops(); // loops in sequence that correspond to the regions

	//	take middle of this loop piece. ... there might be better ways to make the extra jumps...
	Size const anchor( static_cast< Size >( 0.5*(rigid_loops[ root_reg ].stop()
				- rigid_loops[ root_reg ].start()) ) + rigid_loops[ root_reg ].start() );

  for ( Size region = 1; region <= visited_.size(); region++ ) {
    Size old_visited = visited_[ region ];
    if ( visited_[ region ] != visited_[ root_reg ] ) {
			Size target_pos (	rigid_loops[ region ].start()
				+ static_cast< Size >( 0.5*( rigid_loops[ region ].stop()-rigid_loops[ region ].start() ) ) );
			extra_jumps.push_back( new JumpClaim( THIS, anchor, target_pos, DofClaim::EXCLUSIVE ) );
      visited_[ region ] = visited_[ root_reg ];

      if ( old_visited ) { //if we connected a cluster make sure to update all its nodes
				for ( Size i=1; i<=visited_.size(); i++ ) {
					if ( visited_[ i ] == old_visited ) visited_[ i ] = visited_[ root_reg ];
				}
      }
    }
  } // for region
} //generate_rigidity_jumps




} //topology_broker
} //protocols
