# (c) Copyright Rosetta Commons Member Institutions.
# (c) This file is part of the Rosetta software suite and is made available under license.
# (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
# (c) For more information, see http://www.rosettacommons.org. Questions about this can be
# (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

# System, Installer-related, Distutils, and GUI-related imports
import sys, os, platform, math, imp, threading, itertools
import urllib2, socket, tempfile, tarfile
import distutils.sysconfig, distutils.version
import Pmw, Tkinter, tkFileDialog, webbrowser

class Installer:
    """This class, the most basic part of the installation framework,
    installs Rosetta Design Wizard for the user."""

    # These are the big exception cases in our installation procedure
    class CancellationException(Exception):
        pass
    class PackageUnavailableException(Exception):
        pass
    class CannotWriteException(Exception):
        pass
    class ConnectionException(Exception):
        pass
    class MalformedPackageException(Exception):
        pass

    # Note that this is an *in order* list.  It's here mainly so we can present the stages
    # of installation/updating to the user in the side panel.
    __stage_names = ['Introduction', 'Check for Updates', 'Download', 'Verify',
                    'Extract', 'Install', 'Conclusion' ]

    # Convenient Externalized Strings
    __approve_installation = 'Install'
    __cancel = 'Cancel'
    __url = "http://rosettadesign.med.unc.edu/rdwizard/dist"
    __homepage = "http://rosettadesign.med.unc.edu/rdwizard"
    __no_package = "You have an unsupported version of PyMOL.  If you'd like us to add packages\n" + \
                   "for your version of PyMOL, visit our site at\n" + __homepage
    __stage_list_inactive_color = '#AAA'
    __stage_list_active_color = '#000'

    def __init__( self, app ):
        self.app = app

        # This will ensure that if the download stalls, the user isn't stuck with a dialog
        # box stealing focus.  Note that timeouts throw socket.timeout Exceptions.
        socket.setdefaulttimeout( 10.0 )

        # Figure out the package identifier (pretty much the name of the package)
        self.__package_identifier = 'rdwizard_'
        self.__package_identifier += sys.platform + '_'
        self.__package_identifier += platform.uname()[4] + '_'
        libpath = distutils.sysconfig.get_config_var("LIBPL")
        truncated_libpath = "_".join( libpath.split('/')[1:-1] )
        self.__package_identifier += truncated_libpath

        # This is the Installer's main dialog.  This will be activated when install or update_online
        # are called and deactivated when installation/updates are complete or the user cancels.
        self.dialog = Tkinter.Toplevel()
        self.dialog.withdraw()
        self.dialog.title( "Rosetta Design Wizard Installer" )
        self.dialog.resizable( 0, 0 )

        # The side panel contains labels for each step of installation.  They will be added during either
        # prompt_to_install or check_for_updates as appropriate.
        self.frame_side_panel = Tkinter.Frame( self.dialog, bd=2, relief=Tkinter.GROOVE, height=250, width=100 )
        self.frame_side_panel.grid_propagate( False )
        self.frame_side_panel.labels = []
        self.frame_side_panel.grid( row=0, column=0, rowspan=2, sticky=Tkinter.N + Tkinter.S )

        # This panel will contain the "Cancel" button.  Each stage of installation will update this button's command
        # to work appropriately, and in the case that it is clicked, the stage should raise a CanceledException.
        # In addition, each stage's Frame should have a list of Buttons named "buttons", which will be added in when
        # we switch to that stage.
        self.frame_bottom_panel = Tkinter.Frame( self.dialog, height=25, width=450 )
        self.frame_bottom_panel.grid( row=1, column=1, padx=5, pady=5, sticky=Tkinter.E + Tkinter.W )
        self.cancel_button = Tkinter.Button( self.frame_bottom_panel, text="Cancel" )
        self.cancel_button.pack( side=Tkinter.RIGHT )

        # This is an internal mapping of each stage to its appropriate Frame
        self.__stages = {}
        for stage_name in self.__stage_names:
            self.__stages[ stage_name ] = Tkinter.Frame( self.dialog, height=225, width=450 )
            self.__stages[ stage_name ].buttons = []
            self.__stages[ stage_name ].pack_propagate( False )

        # This string, as an invariant, always points to the current stage.
        self.__current_stage = None

        # A bug in some versions of Tkinter causes the geometry manager for a Frame to screw lots of things up when
        # we add and remove it to the window.  As a workaround, we'll add instancemethods to each frame that are called
        # after adding the Frame to the Toplevel that destroy the and recreate the widgets.  To do this, we'll use the
        # instancemethod type and manually add the methods to the Frames.  This is easier and cleaner than creating a
        # bunch of subclasses.
        instancemethod = type( Installer.__init__ )

        # This is a convenience method that will be called to destroy the widgets stored in our Frames, and reset the Frame's
        # grid_propagate flag.
        def clean_frame( F, *members):
            for member in members:
                if member in dir(F):
                    getattr( F, member ).pack_forget()
                    getattr( F, member ).destroy()

            F.grid_propagate( False )

        ##################################################
        # INSTALL FRAME
        ##################################################
        def repopulate( F ):
            clean_frame( F, 'text' )
            F.text = Tkinter.Label( F,
                                    text = 'Please make sure you have sufficient permissions to install Python packages ' + \
                                           'before continuing with the installation of Rosetta Design Wizard.',
                                    padx = 10,
                                    wraplength = 430,
                                    justify = Tkinter.LEFT )

            # Setup buttons for this Frame
            F.buttons = [ Tkinter.Button( self.frame_bottom_panel, text="Install Online", command=self.__install_from_online_source ),
                          Tkinter.Button( self.frame_bottom_panel, text="Select Local Package...", command=self.__install_from_local_source ) ]
            self.cancel_button['command'] = self.__cope_with_rejection

            # Layout widgets in frame
            F.text.pack( anchor=Tkinter.CENTER )

        self.__stages['Introduction'].repopulate = instancemethod( repopulate, self.__stages['Introduction'], Tkinter.Frame )

        ##################################################
        # UPDATE FRAME
        ##################################################
        def repopulate( F ):
            # Because the text is filled in by check_for_updates, we must store the former value (if any)
            if 'text' in dir(F):
                old_text = F.text['text']
            else:
                old_text = None

            clean_frame( F, 'text' )
            # This text is filled in by check_for_updates depending on whether there are updates available.
            # The same method will also disable the Update button if no updates are available.
            F.text = Tkinter.Label( F,
                                    pady = 20,
                                    text = old_text )

            # Setup buttons for this Frame
            F.buttons = [ Tkinter.Button( self.frame_bottom_panel, text="Update", command=self.__install_from_online_source ) ]

            # Here, the cancel button shouldn't offer to remove the plugin.
            self.cancel_button['command'] = self.dialog.destroy

            # Layout widgets in frame
            F.text.pack()

        self.__stages['Check for Updates'].repopulate = instancemethod( repopulate, self.__stages['Check for Updates'], Tkinter.Frame )


        ##################################################
        # DOWNLOAD FRAME
        ##################################################
        def repopulate( F ):
            clean_frame( F, 'text', 'progress' )
            F.text = Tkinter.Label( F,
                                    text = 'Downloading Rosetta Design Wizard...',
                                    pady=20 )
            F.progress = ProgressBar( F )

            # Pack items in order.
            F.text.pack()
            F.progress.pack()

        self.__stages['Download'].repopulate = instancemethod( repopulate, self.__stages['Download'], Tkinter.Frame )

        ##################################################
        # VERIFY PACKAGE FRAME
        ##################################################
        def repopulate( F ):
            clean_frame( F, 'text', 'throbber' )
            F.text = Tkinter.Label( F,
                                    text = 'Verifying package contents.  Please wait...',
                                    pady = 20 )
            F.throbber = Throbber( F )

            F.text.pack()
            F.throbber.pack()

        self.__stages['Verify'].repopulate = instancemethod( repopulate, self.__stages['Verify'], Tkinter.Frame )

        ##################################################
        # EXTRACT PACKAGE FRAME
        ##################################################
        def repopulate( F ):
            clean_frame( F, 'text', 'progress' )
            F.text = Tkinter.Label( F,
                                    text = 'Extracting package...',
                                    pady=20 )
            F.progress = ProgressBar( F )

            # Pack items in order.
            F.text.pack()
            F.progress.pack()

        self.__stages['Extract'].repopulate = instancemethod( repopulate, self.__stages['Extract'], Tkinter.Frame )

        ##################################################
        # COPY PACKAGE FRAME
        ##################################################
        def repopulate( F ):
            clean_frame( F, 'text' )
            F.text = Tkinter.Label( F,
                                    text = 'Installing package.  Please wait...',
                                    pady = 20 )
            F.throbber = Throbber( F )

            F.text.pack()
            F.throbber.pack()

        self.__stages['Install'].repopulate = instancemethod( repopulate, self.__stages['Install'], Tkinter.Frame )

        ##################################################
        # INSTALLATION FINISHED FRAME
        ##################################################
        def repopulate( F ):
            clean_frame( F, 'text' )
            F.text = Tkinter.Label( F,
                                    text = 'Installation Complete!\n\n' + \
                                           'Please enjoy the Rosetta Design Wizard.' )
            F.buttons = [ Tkinter.Button( self.frame_bottom_panel, text="Finish", command=self.dialog.destroy ) ]

            F.text.pack()

        self.__stages['Conclusion'].repopulate = instancemethod( repopulate, self.__stages['Conclusion'], Tkinter.Frame )

    def switch_stage( self, stage ):
        # Update the sidebar to indicate the new stage.
        for label in self.frame_side_panel.labels:
            if label['text'] == stage:
                label['fg'] = self.__stage_list_active_color
            else:
                label['fg'] = self.__stage_list_inactive_color

        # Remove old stage from grid, and remove its buttons from bottom panel
        if self.__current_stage != None:
            self.__stages[ self.__current_stage ].grid_forget()
            for button in self.__stages[ self.__current_stage ].buttons:
                button.pack_forget()

        # Add the new stage and repopulate it.
        self.__current_stage = stage
        self.__stages[ self.__current_stage ].grid( row=0, column=1 )
        self.__stages[ self.__current_stage ].repopulate()

        # Add the buttons in the new stage to the bottom panel.
        for button in self.__stages[ self.__current_stage ].buttons:
            button.pack( side=Tkinter.LEFT )

        # Update the dialog.
        self.dialog.update()

    def prompt_to_install( self ):
        """Ask the user if they want to install rdwizard, and if so, do it. (Or at least try.)"""
        # Populate the sidebar with appropriate Labels (all except Check for Updates).
        for stage in self.__stage_names:
            if stage == 'Check for Updates':
                continue
            label = Tkinter.Label( self.frame_side_panel,
                                   text=stage,
                                   fg=self.__stage_list_inactive_color,
                                   justify=Tkinter.LEFT,
                                   pady=7,
                                   wraplength=95 )
            self.frame_side_panel.labels.append( label )
            label.grid(sticky=Tkinter.W + Tkinter.N)

        # Show the install dialog frame.
        self.switch_stage( "Introduction" )

        # Show the dialog
        self.dialog.deiconify()
        self.dialog.lift()

    def check_for_updates( self ):
        """One of two methods for launching the installer.  This method is the only one in the
        installer that assumes rdwizard is available.  Attempts to go online and check for
        updates.  If one is available, populates self.__stages['Check for Updates'] with contents
        necessary to begin the update.  If not, populates self.__stages['Check for Updates'] with
        a notification that no updates are available.

        If a ConnectionException occurs, notifies the user without displaying the dialog."""
        import rdwizard
        current_version = distutils.version.LooseVersion( str(rdwizard.__revision__) )

        try:
            latest_version = distutils.version.LooseVersion( self.latest_online_version()[1] )
        except:
            self.__notify_connection_problem()
            return

        for stage in self.__stage_names:
            if stage == 'Introduction':
                continue
            label = Tkinter.Label( self.frame_side_panel,
                                   text=stage,
                                   fg=self.__stage_list_inactive_color,
                                   justify=Tkinter.LEFT,
                                   pady=7,
                                   wraplength=95)
            self.frame_side_panel.labels.append( label )
            label.grid(sticky=Tkinter.W + Tkinter.N)

        self.switch_stage( "Check for Updates" )

        F = self.__stages['Check for Updates']

        # Fill the Check for Updates frame with approriate stuff.
        if current_version < latest_version:
            F.text['text'] = 'Updates to Rosetta Design Wizard are available.'
        else:
            F.text['text'] = 'No updates are available.'
            F.buttons[0]['state'] = Tkinter.DISABLED

        # Show the dialog
        self.dialog.deiconify()
        self.dialog.lift()

    def latest_online_version( self ):
        """Returns a tuple (URL, version) for the latest version.  Raises ConnectionException if unable to get the package
        list, or PackageUnavailableException if unable to find an appropriate package."""
        packages = []
        try:
            # Read packages from online list.  Result is a tuple of package names, similar to listing the directory
            url = urllib2.urlopen( self.__url + '/packages.list' )
            for line in url.readlines():
                packages.append( line.strip().split(' ') )

        except: # No internet connection, most likely.  Either way, we can't continue
            raise Installer.ConnectionException()

        # Find packages that fit our package identifier, or raise an exception
        potential_packages = filter( lambda p: p[0] == self.__package_identifier, packages )
        if len(potential_packages) == 1:
            package_filename = potential_packages[0][0] + '_' + potential_packages[0][1] + '.tar.gz'
            result = ( Installer.__url + '/' + package_filename, potential_packages[0][1] )
        else:
            raise Installer.PackageUnavailableException()

        return result

    def __install_from_online_source( self ):
        """Installs the package online, handling exceptions with informative dialogs.

        If an exception occurs, will either close the main dialog or restore current_frame to the main
        dialog as appropriate."""
        # This should point to either the update or the install frame.  We'll store it so we know where
        # to return the user to in case of failure.
        original_stage = self.__current_stage

        # Attempt installation.  Handle all exceptions.
        try:
            self.__fetch_and_install_package()

        except Installer.CancellationException:
            # User canceled somewhere along the way.  Return them to the main dialog.
            self.switch_stage( original_stage )

        except Installer.PackageUnavailableException:
            # Package was unavailable.  Notify the user, deactivate the dialog, and offer
            # to remove the plugin
            self.__notify_package_unavailable()
            self.__cope_with_rejection()

        except Installer.CannotWriteException:
            # Unable to write package to disk.  Notify the user and deactivate the dialog.
            self.__notify_unable_to_write()
            self.dialog.destroy()

        except Installer.MalformedPackageException:
            # Package was malformed.  Return user to main dialog and notify the user of the event.
            self.switch_stage( original_stage )
            self.__notify_package_malformed()

        except Installer.ConnectionException:
            # A connection problem occurred.  Return user to main dialog and notify.
            self.switch_stage( original_stage )
            self.__notify_connection_problem()

        else:
            # Installation completed successfully.
            self.switch_stage( "Conclusion" )

    def __install_from_local_source( self ):
        # This should point to either the update or the install frame.  We'll store it so we know where
        # to return the user to in case of failure.
        filename = tkFileDialog.askopenfilename( parent=self.dialog,
                                                 filetypes=[ ('Rosetta Design Wizard Packages', '.tar.gz'),
                                                             ('All Files', '.*') ] )
        original_stage = self.__current_stage

        # Attempt installation.  Handle all exceptions.
        try:
            self.__install_package( filename )

        except Installer.CancellationException:
            # User canceled somewhere along the way.  Return them to the main dialog.
            self.switch_stage( original_stage )

        except Installer.CannotWriteException:
            # Unable to write package to disk.  Notify the user and deactivate the dialog.
            self.switch_stage( original_stage )
            self.__notify_unable_to_write()

        except Installer.MalformedPackageException:
            # Package was malformed.  Return user to main dialog and notify the user of the event.
            self.switch_stage( original_stage )
            self.__notify_package_malformed()

        else:
            # Installation completed successfully.
            self.switch_stage( "Conclusion" )


    def __fetch_and_install_package( self ):
        """Attempts to locate an appropriate package online and, if one is available, downloads it (with a
        progress dialog) and installs it."""
        # Figure out the latest available package.  Potentially, either a ConnectionException or a
        # PackageUnavailableException will pass through and end this method early.
        package = self.latest_online_version()

        # Create a temporary file, where we will download the package file.
        _, filename = tempfile.mkstemp()

        # Download the package and install it.  If any errors occur, be sure to clean up the file
        try:
            # Download the package file
            self.switch_stage( "Download" )
            self.__download_package( filename, package[0] )

            # Verify, extract, and install the package
            self.__install_package( filename )

        finally:
            os.remove( filename )

    def __install_package( self, filename ):
        """Verifies the given package file, creates a temporary directory, and installs the package.

        Always cleans up the temporary directory it creates."""
        temp_dir = tempfile.mkdtemp()
        try:
             # Open and verify the package file
            self.switch_stage( "Verify" )
            tar = self.__open_archive( filename )

            # Extract the package file
            self.switch_stage( "Extract" )
            self.__extract_archive( tar, temp_dir )

            # Copy the package file into a python path
            self.switch_stage( "Install" )
            self.__setup_extracted_package( temp_dir )

        finally:
            self.__clean_up( temp_dir )

    def __download_package( self, filename, url ):
        """Downloads the item at URL to the given filename.  Displays a dialog informing the user of this,
        with the option of canceling the download.  Assumes file is writable.

        Raises CanceledDownloadException if the user cancels the download.  Raises ConnectionException
        if a connection error (such as a timeout) occurs."""
        # Open the file in wb mode.
        file = open( filename, mode='wb' )

        def download():
            """Thread method that downloads the contents at URL to the given filename.

            Threads with this, when started, will have three members: canceled, which can be set to stop
            the download; timed_out, which is set if the thread exited due to a timeout, and completed,
            a float between 0 and 1 indicating how far along the download is."""
            # Initialize thread globals
            threading.currentThread().canceled = False
            threading.currentThread().timed_out = False
            threading.currentThread().completed = 0.0

            # Do download; catch timeouts by exiting (this will raise a ConnectionException)
            try:
                downloader = urllib2.urlopen( url )
                increment=4096
                size = float(downloader.info().get('Content-Length'))
                for x in itertools.count():
                    data = downloader.read(increment)
                    if data == '' or threading.currentThread().canceled:
                        break
                    file.write( data )
                    threading.currentThread().completed = x*increment/size
            except socket.timeout:
                threading.currentThread().timed_out = True

        download_thread = threading.Thread( target=download )

        # Make the cancel button stop the thread
        def cancel_download():
            download_thread.canceled = True
        self.cancel_button['command'] = cancel_download

        # Start download thread
        download_thread.start()

        # Block, while updating the progress bar.
        while download_thread.isAlive():
            download_thread.join( 0.1 )
            self.__stages['Download'].progress.update_progress( download_thread.completed )

        # On returning to execution, close the download file.
        file.close()

        # Raise exceptions as necessary.
        if download_thread.canceled:
            raise Installer.CancellationException()
        elif download_thread.timed_out:
            raise Installer.ConnectionException()

    def __open_archive( self, filename ):
        """Open the archive at the given path and verify that:

        1) Is a properly-formed archive
        2) Has all components rooted relatively, e.g., nothing starts with / or ..
        3) Contains a 'setup.py' file.

        If any of these conditions are not met, raises a MalformedPackageException

        Otherwise, returns the tarfile.TarFile object pointing to the proper tar file.

        Displays a notification to the user that the package contents are being verified."""
        # Open the tarfile, raising an exception if a problem occurs
        try:
            tar = tarfile.open( filename )
        except:
            raise Installer.MalformedPackageException()

        def verify_archive():
            """Checks the archive to see that the archive is valid; e.g., all files are properly rooted, and there
            is a setup.py file preset."""
            threading.currentThread().canceled = False
            threading.currentThread().archive_is_valid = False
            for member in tar.getmembers():
                if threading.currentThread().canceled:
                    break
                if member.name.startswith('/') or member.name.startswith('..'):
                    threading.currentThread().archive_is_valid = False
                    return
                if member.name == 'setup.py':
                    threading.currentThread().archive_is_valid = True

        verify_thread = threading.Thread( target = verify_archive )

        def cancel_verification():
            verify_thread.canceled = True
        self.cancel_button['command'] = cancel_verification

        # Start verify thread and block until done (animating the throbber).
        verify_thread.start()
        verify_thread.canceled = False
        while verify_thread.isAlive() and not verify_thread.canceled:
            verify_thread.join( 0.03333333 )
            self.__stages['Verify'].throbber.update_with_interval( 0.1 )

        if verify_thread.canceled:
            raise Installer.CancellationException()
        if not verify_thread.archive_is_valid:
            raise Installer.MalformedPackageException()

        return tar

    def __extract_archive( self, tar, temp_dir ):
        """Extract the given tarfile.TarFile to the given temporary directory.

        Raises CanceledExtractionException if the user cancels."""
        def extract():
            """Lightweight thread method that extracts members, extracting a new member and yielding its
            current ratio on each call to next().

            Two methods of interaction are supported: the thread's 'canceled' boolean, which, if set to True,
            stops thread execution, and the thread's 'completed' float, between 0 and 1, which indicates how
            far along the extraction is."""
            threading.currentThread().canceled = False
            threading.currentThread().completed = 0.0
            members = tar.getmembers()
            total = len( members )
            for i, member in enumerate( members ):
                tar.extract(member, temp_dir)
                if threading.currentThread().canceled:
                    break
                threading.currentThread().completed = float(i)/total
            return

        extract_thread = threading.Thread( target=extract )

        # Make the cancel button stop the thread.
        def cancel_extraction():
            extract_thread.canceled = True
        self.cancel_button['command'] = cancel_extraction

        # Start thread, and block until done
        extract_thread.start()

        while extract_thread.isAlive():
            extract_thread.join( 0.1 )
            self.__stages['Extract'].progress.update_progress( extract_thread.completed )

        # If the user canceled, raise exception
        if extract_thread.canceled:
            raise Installer.CancellationException()

    def __setup_extracted_package( self, temp_dir ):
        """Assumes a setup.py file is present in temp_dir.  Creates a module from it and runs the module's install()
        method.  Displays a notification while doing all this."""
        # Disable the cancel button
        self.cancel_button['state'] = Tkinter.DISABLED

        # This method will be run in another thread
        def install():
            threading.currentThread().installation_failed = False
            try:
                setup_module = imp.load_source( 'setup_module', os.path.join(temp_dir,'setup.py') )
                setup_module.install()
            except:
                threading.currentThread().installation_failed = True
                raise Installer.CannotWriteException()

        install_thread = threading.Thread( target = install )

        # Start install thread, and block until done
        install_thread.start()
        while install_thread.isAlive():
            install_thread.join( 0.03333333 )
            self.__stages['Install'].throbber.update_with_interval( 0.1 )

        # Check if installation failed and raise the appropriate exception
        if install_thread.installation_failed:
            raise Installer.CannotWriteException()

        # Add the package to PyMOL
        add_rdwizard_package_to_pymol( self.app )

    def __clean_up( self, dir ):
        """Deletes the provided directory and all its contents."""
        for root, dirs, files in os.walk(dir, topdown=False):
            for name in files:
                os.remove(os.path.join(root, name))
            for name in dirs:
                os.rmdir(os.path.join(root, name))
        os.rmdir( dir )

    def __cope_with_rejection( self ):
        """Displays a dialog asking the user if they would like to remove this plugin completely."""
        ask_me_later = "Ask again on startup"
        remove_plugin = "Remove plugin"

        def dialog_handler( msg ):
            """Removes this file (rdwizard_launcher.py) if that's what the user opted for."""
            if msg == remove_plugin:
                errmsg="Unable to remove plugin. Perhaps you don't\n"+\
                       "have sufficient priveleges?"
                delete_python_file( __file__, self.dialog, errmsg )
            self.rejection.deactivate()
            self.dialog.destroy()

        # Create our dialog.  The function above will respond to button presses.
        self.rejection = Pmw.Dialog( parent=self.dialog,
                                     buttons = [ ask_me_later, remove_plugin ],
                                     defaultbutton = remove_plugin,
                                     title = 'Rosetta Design Wizard Launcher',
                                     command = dialog_handler )
        self.rejection.withdraw()
        w = Tkinter.Label( self.rejection.interior(),
                           text = "You have installed a launcher in PyMOL.  Would you like to remove it, or\n" + \
                                  "would you like to display the installer next time you launch PyMOL?",
                           pady = 20 )
        w.pack(expand = 1, fill = 'both', padx = 4, pady = 4)
        self.rejection.activate()

    def __notify_package_unavailable( self ):
        """Notify the user the package is unavailable and close the dialog."""
        go_to_site = "Go to site"

        def dialog_handler( msg ):
            """Sends the user to the project site if they press the button."""
            if msg == go_to_site:
                # For some reason, if we don't do this in a new thread, things screw up.
                # I believe this is a problem in webbrowser.
                t = threading.Thread( target=webbrowser.open, args=[Installer.__homepage])
                t.start()
            self.info.deactivate()

        self.info = Pmw.Dialog( parent=self.dialog,
                           buttons = [go_to_site, "OK"],
                           defaultbutton = go_to_site,
                           title = 'Package unavailable',
                           command = dialog_handler )
        self.info.withdraw()
        w = Tkinter.Label( self.info.interior(),
                           text = Installer.__no_package + "\n\n" + \
                                  "Your package identifier is:\n\n" + self.__package_identifier + \
                                  "\n\nProviding this information to the developers can help us make packages for your system.",
                           pady = 20 )
        w.pack(expand = 1, fill = 'both', padx = 4, pady = 4)

        self.info.activate()
        del self.info

    def __notify_unable_to_write( self, location="Python package" ):
        """Displays a dialog stating that the system is unable to write to the
        specified location, and blocks until the user clicks 'OK'."""
        d = Pmw.Dialog( parent=self.dialog,
                        title="Error" )
        w = Tkinter.label( d.interior(),
                           text="Unable to write " + location + ".\n" + \
                                "Please relaunch PyMOL with sufficient priveleges." )
        w.pack(expand = 1, fill = 'both', padx = 4, pady = 4)
        d.activate()
        del d

    def __notify_connection_problem( self ):
        """Displays a dialog stating that there was a connection problem."""
        d = Pmw.Dialog( parent=self.dialog,
                        title="Error" )
        w = Tkinter.Label( d.interior(),
                           text="There was a connection problem while attempting\n" +\
                                "to download." )
        w.pack(expand = 1, fill = 'both', padx = 4, pady = 4)
        d.activate()
        del d

    def __notify_package_malformed( self ):
        """Displays a dialog stating the fetched package was malformed."""
        d = Pmw.Dialog( parent=self.dialog,
                        title="Error" )
        w = Tkinter.Label( d.interior(),
                           text="The package was malformed." )
        w.pack(expand = 1, fill = 'both', padx = 4, pady = 4)
        d.activate()
        del d

class Throbber(Tkinter.Canvas):
    """A simple throbber that animates."""
    def __init__( self, parent = None, barcolor='blue', period=2*math.pi, bar_width=40, **options ):
        """Initialize a throbber with the given bar color, width, and period.

        For the period, note that, if you give a period of, say, 20, then do update_with_interval(10), the
        bar will be in the same position but moving the opposite direction."""
        # Some sane default options.
        if 'borderwidth' not in options:
            options['borderwidth'] = 2
        if 'height' not in options:
            options['height'] = 24
        if 'relief' not in options:
            options['relief'] = 'groove'

        # Initialize base class
        Tkinter.Canvas.__init__(self, parent, options)

        # Create the progress bar and text
        self.__bar = self.create_rectangle( 0,0, 0,0, fill=barcolor )

        # Initialize time to 0.
        self.__time = 0.0

        self.__period = period
        self.__bar_width = bar_width

    def update_with_interval( self, interval ):
        # Update the current time
        self.__time = ( self.__time + interval ) % self.__period

        # Update where the bar should be (0 to 1) based on the current time
        bar_position = ( math.sin( self.__time/self.__period * 2*math.pi ) + 1.0 ) / 2.0

        try:
            width = self.winfo_width() - self.__bar_width
            height = self.winfo_height()

            # Update bar position
            self.coords( self.__bar,
                         bar_position*width,                    0,
                         bar_position*width + self.__bar_width, height )

            # Update the canvas itself
            self.update()
        except:
            pass

class ProgressBar(Tkinter.Canvas):
    def __init__( self, parent = None, barcolor='blue', textcolors=['black', 'white'], **options ):

        # Some sane default options.
        if 'borderwidth' not in options:
            options['borderwidth'] = 2
        if 'height' not in options:
            options['height'] = 24
        if 'relief' not in options:
            options['relief'] = 'groove'

        self.__textcolors = textcolors

        # Initialize base class
        Tkinter.Canvas.__init__(self, parent, options)

        # Create the progress bar and text
        self.__bar = self.create_rectangle( 0,0, 0,0, fill=barcolor )
        self.__text = self.create_text( 0,0 )

        # Initialize progress to 0
        self.__progress = 0.0

    def update_progress( self, progress ):
        self.__progress = progress
        try:
            width = self.winfo_width()
            height = self.winfo_height()

            # Update bar width
            self.coords( self.__bar, 0,0, progress*width,height )

            # Switch text color if halfway done
            if progress <= .5:
                fill = self.__textcolors[0]
            else:
                fill = self.__textcolors[1]

            # Update the text
            text = str( round(progress*100,1) ) + '%'
            self.coords( self.__text, width/2 - 4, height/2 - 3 )
            self.itemconfigure(self.__text, text = text, fill = fill)

            # Update the canvas itself
            self.update()
        except:
            pass

def delete_python_file(path, root, message):
    """Attempt to delete the Python (.py) file at the given path, along with the
    corresponding byte-compiled (.pyc) file, if it exists.  If unable to do so,
    use the provided pointer to a Tk root to display a dialog notifying the user
    with the provided message."""
    try:
        os.remove( path )
        if os.path.exists( path + 'c' ):
            os.remove( path+'c' )
    except:
        d = Pmw.Dialog( parent=root,
                        title="Error" )
        w = Tkinter.Label( d.interior(),
                           text=message )
        w.pack(expand = 1, fill = 'both', padx = 4, pady = 4)
        d.activate()
        del d

def delete_duplicate_rdwizard_launchers(app):
    """Search files in this module's directory and attempts to locate duplicates
    of this module and remove them.  If a duplicate is found with a last-modified
    date earlier than or identical to this file's, delete it.  If a duplicate is
    found with a last-modified date later than this file's, delete this file.
    Returns true iff there were no duplicates of this launcher with a later last-
    modified date."""
    result = True
    # Add the current directory to the beginning of sys.path, so we can be sure
    # to import the other plugins properly

    dirname = os.path.dirname( __file__ )
    sys.path = [dirname] + sys.path
    my_last_modified = os.stat( __file__ ).st_mtime

    # Loop through each .py file in the current path and check if it contains
    # this method and its last-modified date is earlier than this file's.  If so,
    # delete it.  If not, delete self.
    for file in [ f for f in os.listdir( dirname ) if f.endswith( '.py' ) ]:
        mod = __import__( file[:-3] )

        # Files without this method certainly aren't duplicates.
        if 'delete_duplicate_rdwizard_launchers' not in dir(mod):
            continue

        # Skip *this* file (and its byte-compiled equivalent)
        if file == os.path.basename( __file__ ) or file+'c' == os.path.basename( __file__ ):
            continue

        other_last_modified = os.stat( os.path.join(dirname, file) ).st_mtime

        # Message to be used in case of errors removing files.
        errmsg="You appear to have duplicate Roestta Design Wizard plugins, but\n" +\
               "the extras couldn't be cleaned up.  Behavior may become erratic\n" +\
               "under these circumstances.  Please run PyMOL with higher priveleges\n" +\
               "so that these may be cleaned up."

        if other_last_modified <= my_last_modified:
            print file, os.path.basename( __file__ )
            print "Deleting older copy of rdwizard launcher."
            print "Old version modified:", other_last_modified
            print "New version modified:", my_last_modified
            delete_python_file( os.path.join( dirname, file ), app.root, errmsg )
        else:
            print file, os.path.basename( __file__ )
            print "Deleting self, found newer copy of rdwizard launcher."
            print "Old version modified:", my_last_modified
            print "New version modified:", other_last_modified
            result = False
            delete_python_file( __file__, app.root, errmsg )

    # Undo the previous change to sys.path
    sys.path = sys.path[1:]

    return result

def add_rdwizard_package_to_pymol( app ):
    import rdwizard as rdwizard_module

    """Adds this file to the Python module list with the name rdwizard_launcher, and
    adds the Rosetta Design Wizard launcher item to the PyMOL menu."""
    # Add this file as a module to Python, so Rosetta Design Wizard can access its
    # automatic update functionality.

    # Verify that we're adding a source file and use load_source to generate the module
    my_source_file = os.path.abspath(__file__)
    if my_source_file[-1] == 'c':
        my_source_file = my_source_file[:-1]
    launcher_module = imp.load_source( 'rdwizard_launcher', my_source_file )

    # Add the PyMOL application to it, so calls to the updater can update the menu bar items
    launcher_module.PyMOL_App = app

    # Add ourselves to the modules.
    sys.modules['rdwizard_launcher'] = launcher_module

    # If we already have a Rosetta Design Wizard item, remove it.
    try:
        rdw_menu_index = self.app.menuBar.component( "Wizard-menu" ).index('Rosetta Design Wizard')
        self.app.menuBar.deletemenuitems( "Wizard", rdw_menu_index )
    except:
        pass

    # Add the menu item to launch Rosetta Design Wizard
    app.menuBar.addmenuitem( 'Wizard', 'command', 'Rosetta Design Wizard',
                             label= "Rosetta Design Wizard",
                             command= lambda a=app: rdwizard_module.do_rdw( a ) )

# This method will be executed when PyMOL starts, if this file is in the pmg_tk/startup plugins directory.
def __init__( self ):
    # Check for more (and less) recent duplicates of this file.  If a more recent duplicate is found, this
    # method will delete this file, and we should use the more recent version and just return.
    if not delete_duplicate_rdwizard_launchers(app=self):
        return

    # Test to see if rdwizard is installed.  If not, install it and continue.
    # If we are successful in importing rdwizard, check for updates if the user has auto update enabled.
    try:
        import rdwizard
    except ImportError:
        inst = Installer( app=self )
        try:
            inst.prompt_to_install()
            import rdwizard
        except:
            pass
    else:
        add_rdwizard_package_to_pymol( self )

# If the user has tried to execute this as a script, display a notification.
if __name__ == "__main__":
    print """Welcome to the Rosetta Design Wizard.

To use please install this script as a plugin in PyMOL

See http://rosettadesign.med.unc.edu/rdwizard for more information."""
