// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   utility/string_util.hh
///
/// @brief  Some std::string helper functions.
/// @author Sergey Lyskov

#ifndef INCLUDED_utility_string_util_HH
#define INCLUDED_utility_string_util_HH

// C++ headers
#include <string>
#include <vector>
#include <list>
#include <sstream>
#include <utility/vector1.hh>

namespace utility {

/// @brief split given std::string using ' ' symbol.
std::vector< std::string > split(const std::string &s);

///@brief combine strings with anything
std::string join(utility::vector1<std::string> const & s, std::string const & connector);

///@brief combine strings with anything
std::string join(std::vector<std::string> const & s, std::string const & connector);

/// @brief join space separations in a string with a connector such as '_'
std::string join(std::string const & string_w_spaces, std::string const & connector);

/// @brief split given std::string using ' ' symbol.
std::list< std::string > split_to_list(const std::string &s);

/// @details split to vector< std::string > using arbitrary split character
std::vector< std::string >
string_split( std::string const & in, char splitchar = ' ' );

/// @brief convert a string to an int
int string2int( std::string st );

/// @brief True iff haystack starts with needle
bool startswith(std::string const & haystack, std::string const & needle);

/// @brief True iff haystack ends with needle
bool endswith(std::string const & haystack, std::string const & needle);

void slurp(std::istream & in, std::string & out);

void trim( std::string & s, const std::string & drop = " " );

/// @brief create a new string that drops all the unwanted substrings of
/// the original string.
std::string
trim( std::string const & s, std::string const & drop = " " );

/// @brief compares two strings ignoring leading and trailing spaces
bool trimmed_compare( std::string const & s1, std::string const & s2 );

/// @brief adds spaces to a left aligned string until a given length is reached
void add_spaces_left_align( std::string & st, std::size_t const newlen );

/// @brief adds spaces to a right aligned string until a given length is reached
void add_spaces_right_align( std::string & st, std::size_t const newlen );

/// @brief Compre two strings, but ignore spaces.  Useful for comparing atom
/// name strings which have pdb-alignment built into them.  Slightly dangerous
/// if you consider the fact that atom names in the PDB are different for
/// different indentation rules: ' CA ' is c-alpha.  'CA  ' is calcium.
inline
bool same_ignoring_spaces( std::string const & s1, std::string const & s2 )
{
	int const size1( s1.size()), size2( s2.size() );
	int c1( 0 ), c2( 0 );
	while ( c1 < size1 && c2 < size2 ) {
		if ( s1[ c1 ] == ' ' ) { ++c1; continue; }
		if ( s2[ c2 ] == ' ' ) { ++c2; continue; }

		if ( s1[ c1 ] != s2[ c2 ] ) return false;
		++c1; ++c2;
	}
	/// check if there are extra non-spaces in c1 that have been missed.
	while ( c1 < size1 ) {
		if ( s1[ c1 ] != ' ' ) return false;
		++c1;
	}

	/// check if there are extra non-spaces in c2 that have been missed.
	while ( c2 < size2 ) {
		if ( s2[ c2 ] != ' ' ) return false;
		++c2;
	}

	return true;
}

template <class T>
inline std::string to_string (const T & t)
{
	std::ostringstream ss;
	ss << t;
	return ss.str();
}

template <class T>
inline T const from_string (std::string const & s, T )
{
	T t;
	std::istringstream ss(s);
	ss >> t;
	return t;
}


} // namespace utility


#endif // INCLUDED_utility_string_util_HH
