// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   test/core/scoring/SurfacePotential.cxxtest.hh
/// @brief  test suite for core::scoring::SurfacePotential.cc
/// @author Ron Jacak

// Test headers
#include <cxxtest/TestSuite.h>

// Unit headers
#include <core/scoring/SurfacePotential.hh>
#include <core/chemical/AA.hh>

#include <core/pose/Pose.hh>
#include <core/io/pdb/pose_io.hh>

#include <core/scoring/Energies.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/TenANeighborGraph.hh>

#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/util.hh>

#include "platform/types.hh"

// Package Headers
#include <test/core/init_util.hh>


// --------------- Test Class --------------- //

// using declarations
using namespace core;

class SurfacePotentialTests : public CxxTest::TestSuite {

	public:

	scoring::SurfacePotential* sp;
	Real TOLERATED_ERROR;
	pose::Pose pose;
	scoring::ScoreFunctionOP sf;

	// --------------- Fixtures --------------- //

	// Define a test fixture (some initial state that several tests share)
	// In CxxTest, setUp()/tearDown() are executed around each test case. If you need a fixture on the test
	// suite level, i.e. something that gets constructed once before all the tests in the test suite are run,
	// suites have to be dynamically created. See CxxTest sample directory for example.

	// Shared initialization goes here.
	void setUp() {
		core_init();
		sp = scoring::SurfacePotential::get_instance();
		TOLERATED_ERROR = 0.001;
	}


	// Shared finalization goes here.
	void tearDown() {
		// sp is just a pointer, and the static class will deallocate when the program ends
	}


	// --------------- Test Cases --------------- //
	void test_average_residue_hASA() {

		// Values from the database file...
		// VAL 96.397 69.151 46.306
		// TRP 134.156 89.430 54.756
		// ASN 35.207 25.759 18.425
		// ARG 69.341 54.594 40.726

		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_val, 3 ), 96.397, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_val, 11 ), 69.151, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_val, 14 ), 46.306, TOLERATED_ERROR );

		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_trp, 1 ), 134.156, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_trp, 12 ), 89.430, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_trp, 15 ), 54.756, TOLERATED_ERROR );

		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_asn, 10 ), 35.207, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_asn, 13 ), 25.759, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_asn, 16 ), 18.425, TOLERATED_ERROR );

		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_arg, 7 ), 69.341, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_arg, 11 ), 54.594, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->average_residue_hASA( chemical::aa_arg, 14 ), 40.726, TOLERATED_ERROR );

	}

	void test_hASA_patch_energy() {

		// Values from the database file... 0-24 is 3.524; 25-50 is something else...
		// 0 3.524
		// 200 2.927
		// 500 6.864
		// 775 50.000

		TS_ASSERT_DELTA( sp->hASA_patch_energy( 5 ), 3.524, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->hASA_patch_energy( 212 ), 2.927, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->hASA_patch_energy( 501 ), 6.864, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->hASA_patch_energy( 800 ), 50.000, TOLERATED_ERROR );
		TS_ASSERT_DELTA( sp->hASA_patch_energy( 776 ), 50.000, TOLERATED_ERROR );

	}

	void test_compute_residue_surface_energy() {

		io::pdb::pose_from_pdb( pose, "core/pack/1l2y_renameH.pdb" );

		// create a score function using the standard packer weights
		sf = scoring::getScoreFunction();
		(*sf)( pose );

		scoring::EnergyMap emap;

		utility::vector1< Size > nbs;
		nbs.resize( pose.n_residue(), 0 );
		scoring::TenANeighborGraph const & tenA_neighbor_graph( pose.energies().tenA_neighbor_graph() );
		for ( Size ii=1; ii <= pose.n_residue(); ++ii ) {
			nbs[ ii ] = tenA_neighbor_graph.get_node( ii )->num_neighbors_counting_self();
		}

		emap.zero();
		sp->compute_residue_surface_energy( pose.residue(6), pose, emap, 6, nbs);
		TS_ASSERT_DELTA( emap[ scoring::surface ], 0.000, TOLERATED_ERROR );

		emap.zero();
		sp->compute_residue_surface_energy( pose.residue(7), pose, emap, 7, nbs);
		TS_ASSERT_DELTA( emap[ scoring::surface ], 4.509, TOLERATED_ERROR );

		emap.zero();
		sp->compute_residue_surface_energy( pose.residue(8), pose, emap, 8, nbs);
		TS_ASSERT_DELTA( emap[ scoring::surface ], 3.1760, TOLERATED_ERROR );

	}

	void test_compute_pose_surface_energy() {

		chemical::ResidueTypeSetCAP rsd_set = chemical::ChemicalManager::get_instance()->residue_type_set( "fa_standard" );
		chemical::make_pose_from_sequence( pose, "DFGLKANM", *rsd_set );

		for ( Size ii=1; ii <= pose.n_residue(); ii+=3 ) {
			pose.set_phi( ii, -150.0 );
			pose.set_psi( ii, 150.0 );
			pose.set_omega( ii, 180.0 );
		}

		Real total_surfaceE = -1.0;
		utility::vector1< Real > residue_surfaceE( pose.n_residue(), 0.0 );

		sp->compute_pose_surface_energy( pose, total_surfaceE, residue_surfaceE );
		TS_ASSERT_DELTA( total_surfaceE, 0.0000, TOLERATED_ERROR );  // tests unscored poses; return 0.0 for all residues

		sf = scoring::getScoreFunction();
		(*sf)( pose );

		sp->compute_pose_surface_energy( pose, total_surfaceE, residue_surfaceE );
		TS_ASSERT_DELTA( total_surfaceE, 22.2590, TOLERATED_ERROR );

		TS_ASSERT_DELTA( residue_surfaceE[1], 2.4360, TOLERATED_ERROR );
		TS_ASSERT_DELTA( residue_surfaceE[4], 4.7710, TOLERATED_ERROR );
		TS_ASSERT_DELTA( residue_surfaceE[5], 2.9270, TOLERATED_ERROR );
		TS_ASSERT_DELTA( residue_surfaceE[6], 2.6140, TOLERATED_ERROR );

	}

};


