// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   test/core/scoring/UnfoldedStatePotential.cxxtest.hh
/// @brief  test suite for core::scoring::UnfoldedStatePotential.cc
/// @author Ron Jacak

// Test headers
#include <cxxtest/TestSuite.h>

// Unit headers
#include <core/scoring/UnfoldedStatePotential.hh>

#include "platform/types.hh"

// Package Headers
#include <test/core/init_util.hh>
#include <core/pose/Pose.hh>

#include <core/chemical/util.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/ResidueTypeSet.hh>

#include <core/conformation/Residue.hh>

#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreType.hh>

#include <core/options/option.hh>


// --------------- Test Class --------------- //

// using declarations
using namespace core;
using namespace core::pose;
using namespace core::scoring;
using namespace core::chemical;

class UnfoldedStatePotentialTests : public CxxTest::TestSuite {

	public:

	PoseOP pose;
	UnfoldedStatePotentialOP unfE_potential;
	Real TOLERATED_ERROR;


	// --------------- Fixtures --------------- //

	// Define a test fixture (some initial state that several tests share)
	// In CxxTest, setUp()/tearDown() are executed around each test case. If you need a fixture on the test
	// suite level, i.e. something that gets constructed once before all the tests in the test suite are run,
	// suites have to be dynamically created. See CxxTest sample directory for example.

	// Shared initialization goes here.
	void setUp() {

		TOLERATED_ERROR = 0.001;

		core_init();

		pose = new Pose;
		core::chemical::ResidueTypeSetCAP rsd_set = chemical::ChemicalManager::get_instance()->residue_type_set( "fa_standard" );
		core::chemical::make_pose_from_sequence( *pose, "DFGLK", *rsd_set );

		unfE_potential = new UnfoldedStatePotential;

	}

	// Shared finalization goes here.
	void tearDown() {
		pose = 0;
		unfE_potential = 0;
	}


	// --------------- Test Cases --------------- //
	void test_unfolded_state_energy() {

		// atr  rep  sol  intra_rep  pro  pair  hb_sr_bb  hb_lr_bb  hb_bb_sc  hb_sc  rama  omega  dun  paapp
		// GLY -1.6933 0.48918 1.20216 0.01788 0.00784 0.00000 0.00000 0.00000 -0.0085 0.00000 0.61547 0.22644 0.00000 -1.0623
		// LEU -3.1780 0.71276 1.75359 2.00079 0.01865 0.00000 0.00000 0.00000 -0.0044 0.00000 -0.1056 0.20147 0.60841 -0.1721

		EnergyMap emap;
		emap.zero();
		unfE_potential->unfolded_state_energy( (pose->residue(3)).aa(), emap );
		TS_ASSERT_DELTA( emap[ fa_atr ], -1.6933, TOLERATED_ERROR );
		TS_ASSERT_DELTA( emap[ fa_sol ], 1.20216, TOLERATED_ERROR );
		TS_ASSERT_DELTA( emap[ rama ], 0.61547, TOLERATED_ERROR );
		TS_ASSERT_DELTA( emap[ omega ], 0.22644, TOLERATED_ERROR );

		emap.zero();
		unfE_potential->unfolded_state_energy( (pose->residue(4)).aa(), emap );
		TS_ASSERT_DELTA( emap[ fa_rep ], 0.71276, TOLERATED_ERROR );
		TS_ASSERT_DELTA( emap[ fa_dun ], 0.60841, TOLERATED_ERROR );
		TS_ASSERT_DELTA( emap[ p_aa_pp ], -0.1721, TOLERATED_ERROR );
		TS_ASSERT_DELTA( emap[ pro_close ], 0.01865, TOLERATED_ERROR );

	}

	void test_pose_unfolded_state_energy() {

		EnergyMap emap;
		emap.zero();
		unfE_potential->pose_unfolded_state_energy( *pose, emap );
		TS_ASSERT_DELTA( emap[ fa_atr ], -14.4040, TOLERATED_ERROR );
		TS_ASSERT_DELTA( emap[ fa_sol ], 9.0237, TOLERATED_ERROR );
		TS_ASSERT_DELTA( emap[ rama ], 0.0742, TOLERATED_ERROR );
		TS_ASSERT_DELTA( emap[ omega ], 1.1634, TOLERATED_ERROR );

	}

};


