// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/constaints/ConstraintSetRemapping.cxxtest.hh
/// @brief  test suite for remapping the constraintSet upon conformation length changes
/// @author Florian Richter, Jan 09, floric@u.washington.edu

// Test headers
#include <cxxtest/TestSuite.h>

#include <test/util/pose_funcs.hh>
#include <test/core/init_util.hh>

#include <core/conformation/Residue.hh>

#include <core/io/pdb/pose_io.hh>
#include <core/pose/Pose.hh>

#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/AtomPairConstraint.hh>
#include <core/scoring/constraints/AngleConstraint.hh>
#include <core/scoring/constraints/DihedralConstraint.hh>
#include <core/scoring/constraints/MultiConstraint.hh>
#include <core/scoring/constraints/AmbiguousConstraint.hh>
#include <core/scoring/constraints/ResidueTypeConstraint.hh>
#include <core/scoring/constraints/Func.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>
#include <core/scoring/EnergyMap.hh>

#include <core/types.hh>

#include <core/util/Tracer.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("core.scoring.constraints.ConstraintSetRemapping.cxxtest");

using namespace core;

class ConstraintSetRemappingTests : public CxxTest::TestSuite
{

public:
	ConstraintSetRemappingTests() {};

	// Shared initialization goes here.
	void setUp() {
		core_init();
	}

	// Shared finalization goes here.
	void tearDown() {
	}

	core::Real
	return_constraint_scores( 	core::pose::Pose const & pose, utility::vector1< core::Size > const & positions ){

		using namespace::scoring;
		core::Real cst_score(0);

		for( utility::vector1< core::Size >::const_iterator res_it = positions.begin(); res_it != positions.end(); ++res_it ){
			EnergyMap scores = pose.energies().residue_total_energies( *res_it );

			cst_score += ( scores[ coordinate_constraint ] + scores[ atom_pair_constraint ] + scores[ angle_constraint ] + scores[ dihedral_constraint ] + scores[ res_type_constraint ] );

		}

		return cst_score;
	}


///////////////////////////////////////////////////////////////////////////////
// ------------------------------------------ //
/// @brief strategy: put in a couple of random constraints, remember the score, mess around with the pose length a bit,
/// @brief then check if the score remains the same
void test_constraint_set_remapping()
{
	using namespace core::scoring::constraints;
	using core::chemical::ResidueType;
	using core::chemical::AtomIndices;
	using core::conformation::Residue;
	using core::id::AtomID;

	pose::Pose pose( create_test_in_pdb_pose() );
	//io::pdb::pose_from_pdb( pose, "core/scoring/constraints/test_in.pdb" );


	scoring::ScoreFunctionOP scorefxn = new scoring::ScoreFunction;
	scorefxn->reset();
	scorefxn->set_weight( scoring::fa_atr, 0.80 );
	scorefxn->set_weight( scoring::fa_rep, 0.44 );
	scorefxn->set_weight( scoring::fa_sol, 0.65 );
	scorefxn->set_weight( scoring::coordinate_constraint, 1.0 );
	scorefxn->set_weight( scoring::atom_pair_constraint, 1.0 );
	scorefxn->set_weight( scoring::angle_constraint, 1.0 );
	scorefxn->set_weight( scoring::dihedral_constraint, 1.0 );
	scorefxn->set_weight( scoring::res_type_constraint, 1.0 );

	core::Real const stddev_radians = numeric::conversions::radians( 5.0 );
	FuncOP some_func = new CircularHarmonicFunc( 1.05, stddev_radians );

	utility::vector1< core::Size > cst_positions;
	cst_positions.push_back( 3 );
	cst_positions.push_back( 12 );
	cst_positions.push_back( 15 );
	cst_positions.push_back( 17 );
	cst_positions.push_back( 19 );
	cst_positions.push_back( 20 );
	cst_positions.push_back( 30 );
	cst_positions.push_back( 33 );
	cst_positions.push_back( 35 );
	cst_positions.push_back( 44 );
	cst_positions.push_back( 45 );
	cst_positions.push_back( 48 );
	cst_positions.push_back( 57 );
	cst_positions.push_back( 65 );
	cst_positions.push_back( 69 );
	cst_positions.push_back( 70 );

	//some random constraints
	pose.add_constraint( new AtomPairConstraint( AtomID(1, 12), AtomID(3, 57), some_func ) );
	pose.add_constraint( new AtomPairConstraint( AtomID(6, 20), AtomID(1, 17), some_func ) );
	pose.add_constraint( new AtomPairConstraint( AtomID(3, 20), AtomID(2, 19), some_func ) );
	pose.add_constraint( new AngleConstraint( AtomID(6, 19), AtomID(1, 3), AtomID(10, 3 ), some_func ) );
	pose.add_constraint( new AngleConstraint( AtomID(4, 69), AtomID(1, 57), AtomID(3, 30 ), some_func ) );
	pose.add_constraint( new DihedralConstraint( AtomID(4, 65), AtomID(1, 65), AtomID(3, 65 ), AtomID(2, 65 ), some_func ) );
	pose.add_constraint( new DihedralConstraint( AtomID(8, 44), AtomID(1, 44), AtomID(2, 48 ), AtomID(1, 48 ), some_func ) );
	pose.add_constraint( new ResidueTypeConstraint( pose, 33, 1.0 ) );

	utility::vector1< ConstraintCOP > multi_csts;
	utility::vector1< ConstraintCOP > ambig_csts;

	multi_csts.push_back( new AngleConstraint( AtomID(6, 15), AtomID(1, 15), AtomID(3, 3 ), some_func ) );
	multi_csts.push_back( new AtomPairConstraint( AtomID(2, 15), AtomID(2, 3),  some_func ) );
	multi_csts.push_back( new DihedralConstraint( AtomID(2, 15), AtomID(7, 15), AtomID(1, 65), AtomID(3, 65 ),  some_func ) );

	ambig_csts.push_back( new AngleConstraint( AtomID(6, 33), AtomID(1, 33), AtomID(3, 30 ), some_func ) );
	ambig_csts.push_back( new AtomPairConstraint( AtomID(2, 35), AtomID(2, 13),  some_func ) );
	ambig_csts.push_back( new DihedralConstraint( AtomID(2, 35), AtomID(7, 35), AtomID(1, 45), AtomID(3, 65 ),  some_func ) );

	pose.add_constraint( new MultiConstraint( multi_csts ) );
	pose.add_constraint( new AmbiguousConstraint( ambig_csts ) );

	//score the pose and remeber the constraint score
	(*scorefxn)( pose );
	core::Real orig_score = return_constraint_scores( pose, cst_positions );

	TR << "orig_cst_score is  " << orig_score << std::endl;

	//now we go to town with the pose
	core::conformation::Residue dummy_ala( pose.residue( 3 ) );
	pose.append_polymer_residue_after_seqpos( dummy_ala, 19, true );
	for( utility::vector1< core::Size >::iterator res_it = cst_positions.begin(); res_it != cst_positions.end(); ++res_it ){
		if( *res_it > 19 ) (*res_it)++;
	}
	(*scorefxn)( pose );

	core::Real test1_score = return_constraint_scores( pose, cst_positions );
	TR << "test1_score (after appending at res19) is  " << test1_score << std::endl;
	TS_ASSERT_DELTA( orig_score, test1_score, 0.00001 );

	pose.delete_polymer_residue( 28 );
	pose.delete_polymer_residue( 28 );
	pose.delete_polymer_residue( 28 );
	for( utility::vector1< core::Size >::iterator res_it = cst_positions.begin(); res_it != cst_positions.end(); ++res_it ){
		if( *res_it > 35 ) (*res_it) -= 3;
	}
	(*scorefxn)( pose );

	core::Real test2_score = return_constraint_scores( pose, cst_positions );
	TR << "test2_score (after 3 deletions at pos 28) is  " << test2_score << std::endl;
	TS_ASSERT_DELTA( orig_score, test2_score, 0.00001 );

	pose.prepend_polymer_residue_before_seqpos( dummy_ala, 28, true );
	for( utility::vector1< core::Size >::iterator res_it = cst_positions.begin(); res_it != cst_positions.end(); ++res_it ){
		if( *res_it > 28 ) (*res_it)++;
	}
	(*scorefxn)( pose );

	core::Real test3_score = return_constraint_scores( pose, cst_positions );
	TR << "test3_score (after prepending before pos 28) is  " << test2_score << std::endl;
	TS_ASSERT_DELTA( orig_score, test3_score, 0.00001 );


}


};
