// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/Dunbrack.cxxtest.hh
/// @brief  Dunbrack Unit Test
/// @author Oliver Lange

// Test headers
#include <cxxtest/TestSuite.h>

// Unit headers

// Project Headers
#include <test/util/pose_funcs.hh>
#include <test/core/init_util.hh>

#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/ChemicalManager.hh>

#include <core/conformation/Residue.hh>

#include <core/graph/Graph.hh>

#include <core/pose/Pose.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
//#include <core/scoring/ScoringManager.hh>
#include <core/scoring/dunbrack/RotamerLibrary.hh>
#include <core/scoring/dunbrack/RotamerLibraryScratchSpace.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/PackerTask.hh>

#include <core/types.hh>

// Package headers
#include <core/io/pdb/pose_io.hh>
#include <utility/io/izstream.hh>

#include <core/util/Tracer.hh>
#include <test/UTracer.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("core.scoring.Dunbrack.cxxtest");

// using declarations
using namespace core;
using namespace scoring;
using namespace conformation;
using namespace chemical;
using namespace scoring;
using namespace pose;

///////////////////////////////////////////////////////////////////////////
/// @name DunbrackTest
/// @brief: unified tests for difference score functions/methods
///////////////////////////////////////////////////////////////////////////
class DunbrackTest : public CxxTest::TestSuite {

public:

	void setUp() {
		core_init();
	}

	void tearDown() {}

	void test_dunbrack_rotamers()
	{
		using namespace core;
		using namespace scoring;
		using namespace scoring::dunbrack;

		pose::Pose pose( create_test_in_pdb_pose() );
		//io::pdb::pose_from_pdb( pose, "core/scoring/test_in.pdb" );

		pack::task::PackerTaskOP task( pack::task::TaskFactory::create_packer_task( pose ));
		// just use standard task no command-line parsing to make test robust against commandline changes
		// bla

		utility::io::izstream data( "core/scoring/dunbrack/test_in_rotamers.dat" );
		std::string line;
		TS_ASSERT( data.good() );
		utility::vector1< utility::vector1< Real > > chi_gold;
		utility::vector1< Real > rot_ener_gold, pose_ener_gold;
		while ( getline( data, line ) ) {
			std::istringstream in( line );
			Size pos, nchi;
			std::string temp1, temp2;
			in >> temp1 >> temp2;

			assert( is_int(temp1) && is_int(temp2) );
			pos  = int_of(temp1);
			nchi = int_of(temp2);

			TS_ASSERT( pose.residue( pos ).nchi() == nchi );
			utility::vector1< Real > chis;
			chi_gold.push_back( chis );
			for ( Size ii = 1; ii <= nchi; ++ii ) {
				Real x;
				in >> x;
				chi_gold[ chi_gold.size() ].push_back( x );
			}
			Real rot_ener;
			Real pose_ener;
			in >> rot_ener >> pose_ener;
			rot_ener_gold.push_back( rot_ener );
			while ( pose_ener_gold.size() < pos ) pose_ener_gold.push_back( -1.0 );
			pose_ener_gold[ pos ] = pose_ener;
		}
		graph::GraphOP dummy_graph = new graph::Graph();
		scoring::ScoreFunction dummy_scorefxn;
		Size ct( 1 );
		for (Size pos = 1; pos <= pose.total_residue(); pos++ ) {
			Residue const & residue( pose.residue( pos ) );

			utility::vector1< ResidueOP > suggested_rotamers;

			// generate empty list of extra_chi_steps
			utility::vector1< utility::vector1< Real > > extra_chi_steps( residue.nchi() );

			SingleResidueRotamerLibraryCAP rotlib = residue.type().get_RotamerLibrary();
			if (rotlib) {
				rotlib->fill_rotamer_vector( pose, dummy_scorefxn, *task, dummy_graph, &( residue.type() ), residue, extra_chi_steps, false /*buried*/, suggested_rotamers);
			}

			bool bOut ( false  );//switch to true to produce a new test_input file
			for ( utility::vector1< ResidueOP >::const_iterator it = suggested_rotamers.begin(),
							eit = suggested_rotamers.end();
						it!=eit;
						++it ) {
				if ( bOut )	std::cout << pos << " " << residue.nchi() << " ";
				else TS_ASSERT( chi_gold[ ct ].size() == residue.nchi() );

				for ( Size n = 1; n <= residue.nchi(); ++n ) {
					if ( bOut ) {
						std::cout << (*it)->chi()[ n ] << " " ;
					}
					else {
						TS_ASSERT_DELTA( chi_gold[ ct ][ n ],  (*it)->chi()[ n ] , 0.001);
					}
				}
				RotamerLibraryScratchSpace scratch;
				if ( bOut ) std::cout << rotlib->rotamer_energy( (**it), scratch ) << " " << rotlib->rotamer_energy( residue, scratch);
				else {
					TS_ASSERT_DELTA( rot_ener_gold[ ct ], rotlib->rotamer_energy( **it, scratch ), 0.001);
					TS_ASSERT_DELTA( pose_ener_gold[ pos ], rotlib->rotamer_energy( residue, scratch ), 0.001);
				}
				if ( residue.nchi() ) {
					ct++;
				};
				if ( bOut ) std::cout << std::endl;
			} // read-out
		} // residues

	}// test

	void test_dunbrack_best_rotamer_energy()
	{
		using namespace core;
		using namespace scoring;
		using namespace scoring::dunbrack;
		test::UTracer UT("core/scoring/dunbrack/best_rotamer_energy.u"); // the name "UT" matters -- it goes with the UTRACE macro

		pose::Pose pose(create_test_in_pdb_pose() );
		//io::pdb::pose_from_pdb( pose, "core/scoring/test_in.pdb" );

		for (Size pos = 1; pos <= pose.total_residue(); pos++ ) {
			Residue const & residue( pose.residue( pos ) );
			SingleResidueRotamerLibraryCAP rotlib = residue.type().get_RotamerLibrary();
			if( rotlib() == NULL ) continue;

			RotamerLibraryScratchSpace scratch;
			Real const this_rotamerE = rotlib->best_rotamer_energy(residue, true /*current well only*/, scratch);
			Real const best_rotamerE = rotlib->best_rotamer_energy(residue, false /*global best*/, scratch);
			UTRACE << "this (ideal) = " << this_rotamerE << "; best (this phi,psi) = " << best_rotamerE << std::endl;
			TS_ASSERT( best_rotamerE <= this_rotamerE );
		}
	}

};
