// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/hbonds/HBonds.cxxtest.hh
/// @brief  Test evaluation of hbond potentials for each HBEvalType across parameter ranges.
/// @author Matthew O'Meara

// Test headers
#include <core/scoring/hbonds/constants.hh>
#include <core/scoring/hbonds/hbonds_geom.hh>
#include <core/scoring/hbonds/types.hh>
#include <core/types.hh>
#include <core/util/Tracer.hh>
#include <cxxtest/TestSuite.h>
#include <test/core/init_util.hh>
#include <test/UTracer.hh>
#include <utility/vector1.hh>

using namespace core;
using namespace scoring;
using namespace hbonds;

static core::util::Tracer TR("core.scoring.hbonds.HBonds.cxxtest");

class HBondsTest : public CxxTest::TestSuite {

public:
  void setUp() {
		core_init();

		// acceptor proton distance
		AHdis_step = .1;

		// -cos(180-theta), where theta is defined by Tanja K.
		xD_step = .1;

		// cos(180-phi, where phi is defined by Tanja K.
		xH_step = .1;

	}

	void tearDown(){}

	void test_hbond_compute_energy()
  {
		test::UTracer UT("core/scoring/hbonds/hbond_compute_energy.u");

		Real energy, dE_dr, dE_dxD, dE_dxH;

		UT << "# Computed energies and derivatives given HBEvalType and geometric parameters\n";
		UT << "# HBEvalType\tAHdis\txD\txH\tenergy\n";
		utility::vector1<HBEvalType> hbe_types;

		hbe_types.push_back(hbe_BB);
		hbe_types.push_back(hbe_BBTURN);
		hbe_types.push_back(hbe_BBHELIX);
		hbe_types.push_back(hbe_BBOTHER);
		hbe_types.push_back(hbe_SP2B);
		hbe_types.push_back(hbe_SP3B);
		hbe_types.push_back(hbe_RINGB);
		hbe_types.push_back(hbe_BSC);
		hbe_types.push_back(hbe_SP2SC);
		hbe_types.push_back(hbe_SP3SC);
		hbe_types.push_back(hbe_RINGSC);

    for( Size hbe = 1; hbe <= hbe_types.size(); hbe++){
			HBEvalType hbe_type = hbe_types[hbe];
			for (Real AHdis = MIN_R; AHdis <MAX_R; AHdis += AHdis_step){
				for (Real xD = MIN_xD; xD < MAX_xD; xD += xD_step){
					for (Real xH = MIN_xH; xH < MAX_xH; xH += xH_step){
						hbond_compute_energy(hbe_type, AHdis, xD, xH, energy, dE_dr, dE_dxD, dE_dxH);
						if (energy < 0){
							UT << hbe_type << "\t" << AHdis << "\t" << xD << "\t" << xH << "\t";
							UT << energy << "\n";
							//<< "\t" << dE_dr << "\t" << dE_dxD << "\t" << dE_dxH << "\n";
						}
					}
				}
			}
		}
	}

private:

	Real TOLERATED_ERROR;

	// acceptor proton distance
	Real AHdis_step;

	// -cos(180-theta), where theta is defined by Tanja K.
	Real xD_step;

	// cos(180-phi, where phi is defined by Tanja K.
	Real xH_step;

};
