// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   test/core/scoring/methods/ProClosureEnergy.cxxtest.hh
/// @brief  test suite for core::scoring::ProClosureEnergy.cc
/// @author Andrew Leaver-Fay

// Test headers
#include <cxxtest/TestSuite.h>

// Unit headers
#include <core/scoring/methods/ProClosureEnergy.hh>

#include <platform/types.hh>

// Package Headers
#include <test/util/pose_funcs.hh>
#include <test/core/init_util.hh>

#include <core/id/DOF_ID.hh>
#include <core/id/TorsionID.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/options/option.hh>

// --------------- Test Class --------------- //

// using declarations
using namespace core;
using namespace core::id;
using namespace core::kinematics;
using namespace core::pose;
using namespace core::scoring;
using namespace core::scoring::methods;

class ProClosureEnergyTests : public CxxTest::TestSuite {

	public:

	// --------------- Fixtures --------------- //

	// Define a test fixture (some initial state that several tests share)
	// In CxxTest, setUp()/tearDown() are executed around each test case. If you need a fixture on the test
	// suite level, i.e. something that gets constructed once before all the tests in the test suite are run,
	// suites have to be dynamically created. See CxxTest sample directory for example.

	// Shared initialization goes here.
	void setUp() {
		using namespace core;
		core_init();
	}

	// Shared finalization goes here.
	void tearDown() {}

	// --------------- Test Cases --------------- //
	void test_eval_energy()
	{
		Pose trpcage( create_trpcage_ideal_pose() );

		Size before_precision = std::cout.precision();
		std::cout.precision( 16 );

		ProClosureEnergy proclose_energy;
		ScoreFunction sfxn;
		sfxn.set_weight( pro_close, 0.5 );

		/*for ( Size ii = 1; ii <= trpcage.total_residue(); ++ii ) {

			EnergyMap emap;
			std::cout << "{\n"
			"EnergyMap emap;\n";

			proclose_energy.eval_intrares_energy( trpcage.residue( ii ), trpcage, sfxn, emap );
			std::cout << "proclose_energy.eval_intrares_energy( trpcage.residue( " << ii << " ), trpcage, sfxn, emap );\n";
			std::cout << "TS_ASSERT_DELTA( emap[ pro_close ], " << emap[ pro_close ] << ", 1e-12 );\n";
			std::cout << "}\n";

			for ( Size jj = std::max( Size(1), ii-1 ); jj <= std::min( trpcage.total_residue(), ii + 1 ); ++jj ) {

				if ( jj == ii ) continue;

				TwoBodyEnergyMap tbemap;
				std::cout << "{\n"
				"TwoBodyEnergyMap tbemap;\n";

				proclose_energy.residue_pair_energy( trpcage.residue( ii ), trpcage.residue( jj ), trpcage, sfxn, tbemap );
				std::cout << "proclose_energy.residue_pair_energy( trpcage.residue( " << ii << " ), trpcage.residue( " <<  jj <<" ), trpcage, sfxn, tbemap );\n";
				std::cout << "TS_ASSERT_DELTA( tbemap[ pro_close ], " << tbemap[ pro_close ] << ", 1e-12 );\n";
				std::cout << "}\n";

			}
		}*/


		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 1 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 1 ), trpcage.residue( 2 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 2 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 2 ), trpcage.residue( 1 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 2 ), trpcage.residue( 3 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 3 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 3 ), trpcage.residue( 2 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 3 ), trpcage.residue( 4 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 4 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 4 ), trpcage.residue( 3 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 4 ), trpcage.residue( 5 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 5 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 5 ), trpcage.residue( 4 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 5 ), trpcage.residue( 6 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 6 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 6 ), trpcage.residue( 5 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 6 ), trpcage.residue( 7 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 7 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 7 ), trpcage.residue( 6 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 7 ), trpcage.residue( 8 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 8 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 8 ), trpcage.residue( 7 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 8 ), trpcage.residue( 9 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 9 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 9 ), trpcage.residue( 8 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 9 ), trpcage.residue( 10 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 10 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 10 ), trpcage.residue( 9 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 10 ), trpcage.residue( 11 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 11 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 11 ), trpcage.residue( 10 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 11 ), trpcage.residue( 12 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0.004436792234172629, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 12 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0.01292460898481068, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 12 ), trpcage.residue( 11 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0.004436792234172629, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 12 ), trpcage.residue( 13 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 13 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 13 ), trpcage.residue( 12 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 13 ), trpcage.residue( 14 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 14 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 14 ), trpcage.residue( 13 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 14 ), trpcage.residue( 15 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 15 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 15 ), trpcage.residue( 14 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 15 ), trpcage.residue( 16 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 16 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 16 ), trpcage.residue( 15 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 16 ), trpcage.residue( 17 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 3.287050070076195e-05, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 17 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0.2078919252851279, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 17 ), trpcage.residue( 16 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 3.287050070076195e-05, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 17 ), trpcage.residue( 18 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0.008603170724419516, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 18 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0.1050023146212085, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 18 ), trpcage.residue( 17 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0.008603170724419516, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 18 ), trpcage.residue( 19 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0.07438490402267117, 1e-12 );
		}
		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 19 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0.2001501300597954, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 19 ), trpcage.residue( 18 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0.07438490402267117, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 19 ), trpcage.residue( 20 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		std::cout.precision( before_precision );

	}


	// --------------- Test Cases --------------- //
	void test_eval_deriv()
	{
		Pose trpcage( create_trpcage_ideal_pose() );

		Size before_precision = std::cout.precision();
		std::cout.precision( 16 );

		ProClosureEnergy proclose_energy;
		ScoreFunction sfxn;
		sfxn.set_weight( pro_close, 0.5 );
		DomainMap dm;

		/*for ( Size ii = 1; ii <= trpcage.total_residue(); ++ii ) {
			for ( Size jj = 1; jj <= trpcage.residue( ii ).nheavyatoms(); ++jj ) {

				AtomID atid( jj, ii );
				Vector f1(0.0), f2(0.0);
				std::cout << "{\n"
				"AtomID atid(" << jj << ", " <<  ii << ");\n"
				"Vector f1(0.0), f2(0.0);";


				proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
				std::cout << "proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );\n";

				std::cout << "TS_ASSERT_DELTA( f1.x(), " << f1.x() << ", 1e-12 );" << std::endl;
				std::cout << "TS_ASSERT_DELTA( f1.y(), " << f1.y() << ", 1e-12 );" << std::endl;
				std::cout << "TS_ASSERT_DELTA( f1.z(), " << f1.z() << ", 1e-12 );" << std::endl;
				std::cout << "TS_ASSERT_DELTA( f2.x(), " << f2.x() << ", 1e-12 );" << std::endl;
				std::cout << "TS_ASSERT_DELTA( f2.y(), " << f2.y() << ", 1e-12 );" << std::endl;
				std::cout << "TS_ASSERT_DELTA( f2.z(), " << f2.z() << ", 1e-12 );" << std::endl;

				std::cout << "}\n";

			}
		}*/


		{
		EnergyMap emap;
		proclose_energy.eval_intrares_energy( trpcage.residue( 20 ), trpcage, sfxn, emap );
		TS_ASSERT_DELTA( emap[ pro_close ], 0, 1e-12 );
		}
		{
		TwoBodyEnergyMap tbemap;
		proclose_energy.residue_pair_energy( trpcage.residue( 20 ), trpcage.residue( 19 ), trpcage, sfxn, tbemap );
		TS_ASSERT_DELTA( tbemap[ pro_close ], 0, 1e-12 );
		}
		{
		AtomID atid(1, 1);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 1);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 1);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 1);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 1);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 1);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 1);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 1);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 2);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 2);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 2);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 2);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 2);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 2);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 2);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 2);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(9, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(10, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(11, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(12, 3);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 4);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 4);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 4);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 4);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 4);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 4);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 4);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 4);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 5);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 5);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 5);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 5);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 5);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 5);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 5);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 5);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(9, 5);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(9, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(10, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(11, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(12, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(13, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(14, 6);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 7);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 7);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 7);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 7);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 7);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 7);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 7);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 7);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 8);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 8);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 8);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 8);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 8);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 8);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 8);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 8);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(9, 8);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 9);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 9);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 9);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 9);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 9);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 9);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 9);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 9);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 10);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 10);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 10);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 10);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 11);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 11);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 11);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -1.474631880094278, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -0.2155201857813231, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 2.129323662388725, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0.08527979845185839, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.5687979783684893, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0.00148820191644882, 1e-12 );
		}
		{
		AtomID atid(4, 11);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0.8612685797612546, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0.02792537865240938, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), -2.66937186546861, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -0.06849608745924113, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0.6098422088178803, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -0.01572034751268993, 1e-12 );
		}
		{
		AtomID atid(1, 12);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -2.138045361010974, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 5.981269846338597, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 7.194578787140178, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -0.9831349974827861, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.9243281303283368, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0.4762848326249335, 1e-12 );
		}
		{
		AtomID atid(2, 12);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 12);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 12);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 12);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 12);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 12);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 2.624790957800545, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -0.007696858205211322, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), -2.229330890898199, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -0.02725048728728082, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0.5341047784161067, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -0.03392845884041665, 1e-12 );
		}
		{
		AtomID atid(8, 12);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0.1266177035434523, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -5.78597818100448, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), -4.425199693162084, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0.9936017737774526, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0.3491791214628392, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -0.4281242281882757, 1e-12 );
		}
		{
		AtomID atid(1, 13);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 13);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 13);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 13);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 13);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 13);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 14);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 14);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 14);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 14);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 14);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 14);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 15);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 15);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 15);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 15);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0.01001105371125114, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -0.1619370001105013, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), -0.09142624391378709, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -0.04319035028972987, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0.008147622122878638, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -0.01916060778809877, 1e-12 );
		}
		{
		AtomID atid(4, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -0.00851768351328502, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0.1347698899648933, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0.06795891179447555, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0.04673953946886925, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.008756073745012014, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0.02322238037985436, 1e-12 );
		}
		{
		AtomID atid(5, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(8, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(9, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(10, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(11, 16);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(1, 17);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 1.413529548011816, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 5.39575732229966, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 27.78055673837027, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -1.187413035510852, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -4.307828959710055, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0.8971182002820073, 1e-12 );
		}
		{
		AtomID atid(2, 17);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 17);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 1.077448210624765, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -1.316333335233674, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), -3.474576147164164, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0.1170518913539495, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0.7429580545260719, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -0.2451703652138194, 1e-12 );
		}
		{
		AtomID atid(4, 17);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -0.2607467923542006, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 1.015942647364249, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 3.103708666635216, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -0.1630772376794198, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.8019460170105206, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0.2488021189730926, 1e-12 );
		}
		{
		AtomID atid(5, 17);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 17);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 17);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -0.052739578724423, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0.2485784621124829, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0.1698751392812644, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0.03904935869150208, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.007234184572057359, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0.02270908632499372, 1e-12 );
		}
		{
		AtomID atid(8, 17);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -1.362283339485359, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -5.617168674266535, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), -27.92696454553222, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 1.144814487640211, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 4.315671595904246, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -0.9238890591987567, 1e-12 );
		}
		{
		AtomID atid(1, 18);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 7.27290737840168, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -13.83592225782694, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 12.09716608376573, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 2.359927939656345, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.6728699299141235, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -2.188389675849113, 1e-12 );
		}
		{
		AtomID atid(2, 18);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 18);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -9.25777176428948, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 6.343964973594677, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), -7.590090203945331, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -0.9098607750731752, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0.934318494239786, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 1.890697845341337, 1e-12 );
		}
		{
		AtomID atid(4, 18);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 12.62553145999738, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -8.735935614420123, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 9.713703396534196, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 1.024153800493938, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.8823603550141677, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -2.12470243691984, 1e-12 );
		}
		{
		AtomID atid(5, 18);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 18);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 18);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -2.686707182244732, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 1.581981025322489, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 3.009438437874687, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -0.1916873707349972, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.6926746418713113, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0.1929895947209685, 1e-12 );
		}
		{
		AtomID atid(8, 18);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -5.402901614427512, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 12.55433192037385, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), -14.73573704111146, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -2.12221522259588, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 1.424532534269884, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 1.99176832736887, 1e-12 );
		}
		{
		AtomID atid(1, 19);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), -21.86514611329756, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 5.858723152760471, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 16.34192217752127, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 2.89412162692008, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.4635870468851155, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 4.038473546150337, 1e-12 );
		}
		{
		AtomID atid(2, 19);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 19);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 19);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 19);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 19);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 19);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 7.941456124025248, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -1.634213394323738, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 4.329161739869027, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0.9207691759949521, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), -0.5738467365623917, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -1.905689029522749, 1e-12 );
		}
		{
		AtomID atid(8, 19);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 10.55593029356438, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), -1.832539117611287, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), -22.79469710997918, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), -3.929183828335801, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0.9854756442218889, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), -1.898779925049082, 1e-12 );
		}
		{
		AtomID atid(1, 20);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(2, 20);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(3, 20);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(4, 20);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(5, 20);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(6, 20);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}
		{
		AtomID atid(7, 20);
		Vector f1(0.0), f2(0.0);proclose_energy.eval_atom_derivative( atid, trpcage, dm, sfxn, sfxn.weights(), f1, f2 );
		TS_ASSERT_DELTA( f1.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f1.z(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.x(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.y(), 0, 1e-12 );
		TS_ASSERT_DELTA( f2.z(), 0, 1e-12 );
		}

		std::cout.precision( before_precision );

	}



};


