// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   test/core/scoring/methods/P_AA_Energy.cxxtest.hh
/// @brief  test suite for core::scoring::P_AA_Energy.cc
/// @author Ron Jacak

// Test headers
#include <cxxtest/TestSuite.h>

// Unit headers
#include <core/scoring/methods/UnfoldedStateEnergy.hh>

#include "platform/types.hh"

// Package Headers
#include <test/core/init_util.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/EnergyMap.hh>

#include <core/chemical/util.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/scoring/ScoreType.hh>
#include <core/options/option.hh>
#include <core/chemical/ResidueTypeSet.hh>


// --------------- Test Class --------------- //

// using declarations
using namespace core;
using namespace core::pose;
using namespace core::scoring;
using namespace core::scoring::methods;

class UnfoldedStateEnergyTests : public CxxTest::TestSuite {

	public:

	PoseOP the_pose;
	Real TOLERATED_ERROR;

	// --------------- Fixtures --------------- //

	// Define a test fixture (some initial state that several tests share)
	// In CxxTest, setUp()/tearDown() are executed around each test case. If you need a fixture on the test
	// suite level, i.e. something that gets constructed once before all the tests in the test suite are run,
	// suites have to be dynamically created. See CxxTest sample directory for example.

	// Shared initialization goes here.
	void setUp() {

		TOLERATED_ERROR = 0.001;
		core_init();

		the_pose = new Pose;
		core::chemical::ResidueTypeSetCAP rsd_set = chemical::ChemicalManager::get_instance()->residue_type_set( "fa_standard" );
		core::chemical::make_pose_from_sequence( *the_pose, "DFGLK", *rsd_set );

	}

	// Shared finalization goes here.
	void tearDown() {
		the_pose = 0;
	}


	// --------------- Test Cases --------------- //

	// Test three different ways of using the UnfoldedStateEnergy energy method class. The difference lies in how the class
	// is constructed: with the empty constructor, with an EMap or with a vector.

	void test_residue_energy_no_weights() {

		UnfoldedStateEnergyOP unweighted_unf_energy = new UnfoldedStateEnergy;

		EnergyMap emap;
		emap.zero();

		unweighted_unf_energy->residue_energy( the_pose->residue(2), emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.0, TOLERATED_ERROR );

		emap.zero();
		unweighted_unf_energy->residue_energy( the_pose->residue(3), emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.0, TOLERATED_ERROR );

		unweighted_unf_energy = 0;
	}

	void test_residue_energy_emap_weights() {

		EnergyMap emap;
		emap[ fa_atr ] = 0.8;
		emap[ fa_rep ] = 0.454238;
		emap[ fa_sol ] = 1.04457;
		emap[ fa_intra_rep ] = 0.223082;
		emap[ pro_close ] = 0.374325;
		emap[ fa_pair ] = 0.743825;
		emap[ hbond_sr_bb ] = 0.585;
		emap[ hbond_lr_bb ] = 1.17;
		emap[ hbond_bb_sc ] = 2.58097;
		emap[ hbond_sc ] = 0.477475;
		emap[ dslf_ss_dst ] = 1;
		emap[ dslf_cs_ang ] = 1;
		emap[ dslf_ss_dih ] = 1;
		emap[ dslf_ca_dih ] = 1;
		emap[ rama ] = 0.2;
		emap[ omega ] = 0.151806;
		emap[ fa_dun ] = 0.358969;
		emap[ p_aa_pp ] = 1.01463;
		emap[ unfolded ] = 0.61846;
		emap[ ref ] = 0.0;

		UnfoldedStateEnergyOP weighted_unf_energy_emap = new UnfoldedStateEnergy( emap );

		// atr  rep  sol  intra_rep  pro  pair  hb_sr_bb  hb_lr_bb  hb_bb_sc  hb_sc  rama  omega  dun  paapp
		// GLY -1.6933 0.48918 1.20216 0.01788 0.00784 0.00000 0.00000 0.00000 -0.0085 0.00000 0.61547 0.22644 0.00000 -1.0623
		// LEU -3.1780 0.71276 1.75359 2.00079 0.01865 0.00000 0.00000 0.00000 -0.0044 0.00000 -0.1056 0.20147 0.60841 -0.1721
		// values below verified using Excel spreadsheet for correctness
		emap.zero();
		weighted_unf_energy_emap->residue_energy( the_pose->residue(4), emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.1083, TOLERATED_ERROR );

		emap.zero();
		weighted_unf_energy_emap->residue_energy( the_pose->residue(5), emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.5965, TOLERATED_ERROR );

		weighted_unf_energy_emap = 0;
	}

	// prefix'ing the name of a test with the letter 'x' disables that test (without having to comment it out)
	void xtest_residue_energy_vector_weights() {

		UnfoldedStateEnergyOP weighted_unf_energy_vector;

		Real weights[] = { 0.8, 0.454238, 1.04457, 0, 0.223082, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0.374325, 0, 0, 0, 0, 0, 0.743825, 0,
											 0, 0, 0, 0.585, 1.17, 2.58097, 0, 0, 0.477475, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
											 1, 0, 0, 0, 0, 0, 0, 0, 0.2, 0.151806,
											 0.358969,1.01463, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0.61846, 0 };

		const size_t size = sizeof(weights) / sizeof (weights[0]);
		utility::vector1< Real > v(weights, weights + size);

		weighted_unf_energy_vector = new UnfoldedStateEnergy( v );

		EnergyMap emap;
		emap.zero();
		weighted_unf_energy_vector->residue_energy( the_pose->residue(1), emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.8143, TOLERATED_ERROR );

		emap.zero();
		weighted_unf_energy_vector->residue_energy( the_pose->residue(4), emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.1083, TOLERATED_ERROR );

		emap.zero();
		weighted_unf_energy_vector->residue_energy( the_pose->residue(5), emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.5965, TOLERATED_ERROR );

		weighted_unf_energy_vector = 0;
	}

};


