// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/docking/DockingTest.cxxtest.hh
/// @brief  tests for container Docking Movers classes.
/// @author Sergey Lyskov

// Test headers
#include <test/UMoverTest.hh>
#include <test/UTracer.hh>


// Unit headers
#include <protocols/docking/DockingProtocol.hh>
#include <protocols/docking/DockingInitialPerturbation.hh>
#include <protocols/docking/DockingLowRes.hh>
#include <protocols/docking/DockingHighRes.hh>
#include <protocols/docking/RestrictTaskForDocking.hh>
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/SwitchResidueTypeSetMover.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/RigidBodyMover.hh>

// project headers
#include <core/types.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/kinematics/Edge.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

///////////////////////////////////////////////////////////////////////////
/// @name DockingTest
/// @brief: tests functions specific to the Docking Protocol
/// @author Sid Chaudhury
///////////////////////////////////////////////////////////////////////////
class DockingTest : public CxxTest::TestSuite {

		core::pose::Pose fullatom_pose;
		core::pose::Pose centroid_pose;
		core::Size rb_jump;

public:
	void setUp() {
		core_init();
    core::io::pdb::pose_from_pdb( fullatom_pose, "protocols/docking/DockingTest.pdb" );
    rb_jump = 1;

    //setting up the fold tree as is used in docking
    core::kinematics::FoldTree fold_tree;

    core::Size jump_pos1 = 197;
    core::Size jump_pos2 = 282;
    core::Size cutpoint = 245;

    fold_tree.clear();
    fold_tree.add_edge( jump_pos1, jump_pos2, rb_jump );
    fold_tree.add_edge( 1, jump_pos1, core::kinematics::Edge::PEPTIDE );
    fold_tree.add_edge( jump_pos1, cutpoint, core::kinematics::Edge::PEPTIDE );
    fold_tree.add_edge( jump_pos2, cutpoint+1, core::kinematics::Edge::PEPTIDE );
    fold_tree.add_edge( jump_pos2, fullatom_pose.total_residue(), core::kinematics::Edge::PEPTIDE );
    fold_tree.reorder( 1 );

    fullatom_pose.fold_tree(fold_tree);

		centroid_pose = fullatom_pose;

    protocols::moves::SwitchResidueTypeSetMover to_centroid( core::chemical::CENTROID );
		to_centroid.apply(centroid_pose);
	}

	void tearDown() {
		fullatom_pose.clear();
		centroid_pose.clear();
	}

	/// @brief test the docking protocol functions
	void test_DockingProtocolFunctions() {

		using protocols::docking::DockingProtocolOP;
		using protocols::docking::DockingProtocol;
		DockingProtocolOP docking_protocol = new DockingProtocol(rb_jump);

		test::UTracer UT("protocols/docking/DockingProtocolFunctions.u");

		core::scoring::ScoreFunctionOP scorefxn;
		core::pose::Pose decoy_pose, multichain_pose;

		UT << "Testing docking low-res scoring function..."<<std::endl;
		scorefxn = core::scoring::ScoreFunctionFactory::create_score_function( "interchain_cen" ) ;
		scorefxn->show(UT, centroid_pose);

		UT << "Testing docking high-res scoring function..."<<std::endl;
    scorefxn = core::scoring::ScoreFunctionFactory::create_score_function( "docking" ) ;
		scorefxn->show(UT, fullatom_pose);

		UT << "Testing DockingProtocol.setup_foldtree()..."<< std::endl;
		docking_protocol->setup_foldtree(fullatom_pose);
		UT << fullatom_pose.fold_tree() << std::endl;

		UT << "Testing DockingProtocol.setup_foldtree()for multichain..."<< std::endl;
    core::io::pdb::pose_from_pdb( multichain_pose, "protocols/docking/DockingMultiChain.pdb" );
		DockingProtocolOP docking_protocol2 = new DockingProtocol();
		docking_protocol2->setup_foldtree(multichain_pose, "AB_E");
		UT << multichain_pose.fold_tree() << std::endl;

		std::cout<<"interaction"<<std::endl;
		UT << "Testing DockingProtocol.calc_interaction_energy()..."<<std::endl;
		core::Real int_energy = docking_protocol->calc_interaction_energy(fullatom_pose);
		UT << int_energy << std::endl;

		UT << "Testing DockingProtocol.recover_sidechains()..."<<std::endl;
    core::io::pdb::pose_from_pdb( decoy_pose, "protocols/docking/DockingDecoy.pdb" );
		docking_protocol->setup_foldtree(decoy_pose);
		docking_protocol->recover_sidechains(decoy_pose, fullatom_pose);

		UT << "Testing DockingProtocol.calc_Lrmsd()..."<<std::endl;
		UT << docking_protocol->calc_Lrmsd(fullatom_pose, decoy_pose) << std::endl;

		UT << "Testing DockingProtocol.calc_Irmsd()..."<<std::endl;
		UT << docking_protocol->calc_Irmsd(fullatom_pose, decoy_pose) << std::endl;

		UT << "Testing DockingProtocol.calc_Fnat()..."<<std::endl;
		UT << docking_protocol->calc_Fnat(fullatom_pose, decoy_pose) << std::endl;

		UT << "Testing DockingProtocol.docking_lowres_filter()..."<<std::endl;
		UT << docking_protocol->docking_lowres_filter(centroid_pose) << std::endl;

		UT << "Testing DockingProtocol.docking_highres_filter()..."<<std::endl;
		UT << docking_protocol->docking_highres_filter(fullatom_pose) << std::endl;

				}

	void test_DockingPacking() {
		using protocols::docking::RestrictTaskForDocking;
		using protocols::docking::RestrictTaskForDockingOP;

		using protocols::moves::PackRotamersMover;
		using protocols::moves::PackRotamersMoverOP;

		core::scoring::ScoreFunctionOP scorefxn_pack = core::scoring::ScoreFunctionFactory::create_score_function("standard");
		core::scoring::ScoreFunctionOP scorefxn_dockmin = core::scoring::ScoreFunctionFactory::create_score_function("docking", "docking_min");
		(*scorefxn_pack)(fullatom_pose);

	  RestrictTaskForDockingOP rtfd = new RestrictTaskForDocking( scorefxn_pack, rb_jump, true );
	  core::pack::task::TaskFactory tf;
	  tf.push_back( rtfd );

		core::pack::task::PackerTaskOP task = tf.create_task_and_apply_taskoperations( fullatom_pose );

		core::Real temperature = 0.8;
		protocols::moves::MonteCarloOP mc = new protocols::moves::MonteCarlo(fullatom_pose, *scorefxn_dockmin, temperature);

	  PackRotamersMoverOP pack_interface_repack = new PackRotamersMover( scorefxn_pack, task );

		pack_interface_repack->apply(fullatom_pose);

		mc->reset(fullatom_pose);

		core::Real energy_cut = 0.01;
		protocols::moves::RotamerTrialsMoverOP rotamer_trials = new protocols::moves::EnergyCutRotamerTrialsMover( scorefxn_dockmin, *task, mc, energy_cut );
		rotamer_trials->apply(fullatom_pose);

    test::UTracer UT("protocols/docking/DockingPacking.pdb");
		UT.abs_tolerance(0.003);
		UT << std::endl;
		fullatom_pose.dump_pdb(UT);

	}

	void test_DockingSlideIntoContact() {
		using protocols::docking::DockingSlideIntoContact;

		DockingSlideIntoContact slide( rb_jump );


		protocols::moves::RigidBodyTransMoverOP trans_mover = new protocols::moves::RigidBodyTransMover(centroid_pose, rb_jump);
		trans_mover->step_size(10.26);
		trans_mover->apply(centroid_pose);
		slide.apply( centroid_pose );
    test::UTracer UT("protocols/docking/DockingSlideIntoContact.pdb");
		UT.abs_tolerance(0.003);
		UT << std::endl;
		centroid_pose.dump_pdb(UT);

	}

	void test_DockingRigidBodyMinimize() {
		using protocols::moves::MinMover;
		using protocols::moves::MinMoverOP;

    core::scoring::ScoreFunctionOP scorefxn = core::scoring::ScoreFunctionFactory::create_score_function( "docking", "docking_min" ) ;

		core::kinematics::MoveMapOP movemap = new core::kinematics::MoveMap();
		movemap->set_chi(false);
		movemap->set_bb(false);
		movemap->set_jump(rb_jump, true);

		core::Real tolerance = 0.01;
		std::string min_type = "dfpmin_armijo_nonmonotone";
		bool nb_list = true;

		MinMoverOP minmover = new MinMover(movemap, scorefxn, min_type, tolerance, nb_list);
		minmover->apply(fullatom_pose);
    test::UTracer UT("protocols/docking/DockingRigidBodyMinimize.pdb");
		UT.abs_tolerance(0.003);
		UT << std::endl;
		fullatom_pose.dump_pdb(UT);
		}

};


