// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   test/protocols/match/ProteinSCSampler.cxxtest.hh
/// @brief
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com)


// Test headers
#include <cxxtest/TestSuite.h>

#include <protocols/match/upstream/ProteinSCSampler.hh>
#include <protocols/match/upstream/ProteinUpstreamBuilder.hh>
#include <protocols/match/upstream/OriginalScaffoldBuildPoint.hh>
#include <protocols/match/upstream/UpstreamResTypeGeometry.hh>
#include <protocols/match/output/WriteUpstreamCoordinateKineamge.hh>

#include <core/chemical/ResidueTypeSet.hh>
#include <core/conformation/Residue.hh>

#include <test/util/pose_funcs.hh>
#include <test/core/init_util.hh>

// Utility headers
#include <utility/exit.hh>

/// Project headers
#include <core/types.hh>

// C++ headers
#include <string>
#include <iostream>
#include <sstream>

using namespace protocols::match;
using namespace protocols::match::upstream;
using namespace protocols::match::output;

// --------------- Test Class --------------- //

class ProteinUpstreamBuilderTests : public CxxTest::TestSuite {

	public:


	// --------------- Fixtures --------------- //

	// Define a test fixture (some initial state that several tests share)
	// In CxxTest, setUp()/tearDown() are executed around each test case. If you need a fixture on the test
	// suite level, i.e. something that gets constructed once before all the tests in the test suite are run,
	// suites have to be dynamically created. See CxxTest sample directory for example.


	// Shared initialization goes here.
	void setUp() {
		core_init();
	}

	// Shared finalization goes here.
	void tearDown() {
	}


	// --------------- Test Cases --------------- //
	void test_sc_upstream_geom_map_atoms_to_controlling_chi() {
		using namespace core;
		core::pose::Pose trpcage = create_trpcage_ideal_pose();

		UpstreamResTypeGeometry leu_geom( trpcage.residue_type( 2 ));

		/*
		std::cout << leu_geom.natoms() << std::endl;
		std::cout << leu_geom.nchi() << std::endl;
		for ( Size ii = 1; ii <= leu_geom.natoms(); ++ii ) {
			//std::cout << "Atom: " << ii << " " << trpcage.residue_type( 2 ).atom_name( ii );
			//std::cout << " " << leu_geom.atom_controlled_by_any_chi( ii );
			//std::cout << " " << leu_geom.atom_is_chitip( ii );
			//std::cout << " " << leu_geom.controlling_chi_for_atom()[ ii ];
			//std::cout << " " << leu_geom.which_point_for_atom()[ ii ] << std::endl;
			std::cout << "TS_ASSERT( leu_geom.atom_controlled_by_any_chi( " << ii << " ) = " << leu_geom.atom_controlled_by_any_chi( ii ) << "); " << std::endl;
			std::cout << "TS_ASSERT( leu_geom.atom_is_chitip( " << ii << " ) = " << leu_geom.atom_is_chitip( ii ) << "); " << std::endl;
			std::cout << "TS_ASSERT( leu_geom.controlling_chi_for_atom()[ " << ii << " ] = " << leu_geom.controlling_chi_for_atom()[ ii ] << "); " << std::endl;
			std::cout << "TS_ASSERT( leu_geom.which_point_for_atom()[ " << ii << " ] = " << leu_geom.which_point_for_atom()[ ii ] << "); " << std::endl;

		}

		for ( Size ii = 1; ii <= leu_geom.nchi(); ++ii ) {
			//std::cout << "Chi: " << ii;
			//std::cout << " " << leu_geom.chitip_atoms()[ ii ];
			//std::cout << " " << leu_geom.nonchitip_atoms()[ ii ].size() << std::endl;
			//for ( Size jj = 1; jj <= leu_geom.nonchitip_atoms()[ ii ].size(); ++jj ) {
			//	std::cout << "   " << jj << " " << leu_geom.nonchitip_atoms()[ ii ][ jj ] << std::endl;
			//}
			std::cout << "TS_ASSERT( leu_geom.chitip_atoms()[ " << ii << " ] = " << leu_geom.chitip_atoms()[ ii ] << "); " << std::endl;
			std::cout << "TS_ASSERT( leu_geom.nonchitip_atoms()[ " << ii << " ].size() = " << leu_geom.nonchitip_atoms()[ ii ].size() << "); " << std::endl;
			for ( Size jj = 1; jj <= leu_geom.nonchitip_atoms()[ ii ].size(); ++jj ) {
				std::cout << "TS_ASSERT( leu_geom.nonchitip_atoms()[ " << ii << " ][ " << jj << "] = " << leu_geom.nonchitip_atoms()[ ii ][ jj ] << "); " << std::endl;
			}

		}*/

		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 1 ) == 0);
		TS_ASSERT( leu_geom.atom_is_chitip( 1 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 1 ] == 0);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 1 ] == 0);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 2 ) == 0);
		TS_ASSERT( leu_geom.atom_is_chitip( 2 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 2 ] == 0);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 2 ] == 0);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 3 ) == 0);
		TS_ASSERT( leu_geom.atom_is_chitip( 3 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 3 ] == 0);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 3 ] == 0);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 4 ) == 0);
		TS_ASSERT( leu_geom.atom_is_chitip( 4 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 4 ] == 0);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 4 ] == 0);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 5 ) == 0);
		TS_ASSERT( leu_geom.atom_is_chitip( 5 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 5 ] == 0);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 5 ] == 0);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 6 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 6 ) == 1);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 6 ] == 1);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 6 ] == 0);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 7 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 7 ) == 1);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 7 ] == 2);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 7 ] == 0);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 8 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 8 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 8 ] == 2);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 8 ] == 1);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 9 ) == 0);
		TS_ASSERT( leu_geom.atom_is_chitip( 9 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 9 ] == 0);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 9 ] == 0);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 10 ) == 0);
		TS_ASSERT( leu_geom.atom_is_chitip( 10 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 10 ] == 0);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 10 ] == 0);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 11 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 11 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 11 ] == 1);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 11 ] == 1);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 12 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 12 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 12 ] == 1);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 12 ] == 2);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 13 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 13 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 13 ] == 2);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 13 ] == 2);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 14 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 14 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 14 ] == 2);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 14 ] == 3);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 15 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 15 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 15 ] == 2);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 15 ] == 4);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 16 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 16 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 16 ] == 2);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 16 ] == 5);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 17 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 17 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 17 ] == 2);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 17 ] == 6);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 18 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 18 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 18 ] == 2);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 18 ] == 7);
		TS_ASSERT( leu_geom.atom_controlled_by_any_chi( 19 ) == 1);
		TS_ASSERT( leu_geom.atom_is_chitip( 19 ) == 0);
		TS_ASSERT( leu_geom.controlling_chi_for_atom()[ 19 ] == 2);
		TS_ASSERT( leu_geom.which_point_for_atom()[ 19 ] == 8);
		TS_ASSERT( leu_geom.chitip_atoms()[ 1 ] == 6);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 1 ].size() == 2);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 1 ][ 1] == 11);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 1 ][ 2] == 12);
		TS_ASSERT( leu_geom.chitip_atoms()[ 2 ] == 7);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 2 ].size() == 8);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 2 ][ 1] == 8);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 2 ][ 2] == 13);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 2 ][ 3] == 14);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 2 ][ 4] == 15);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 2 ][ 5] == 16);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 2 ][ 6] == 17);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 2 ][ 7] == 18);
		TS_ASSERT( leu_geom.nonchitip_atoms()[ 2 ][ 8] == 19);
	}

	void test_sc_upstream_geom_coords() {
		using namespace core;
		core::pose::Pose trpcage = create_trpcage_ideal_pose();

		UpstreamResTypeGeometry leu_geom( trpcage.residue_type( 2 ));

		utility::vector1< UpstreamResTypeGeometry::HTReal > chitip_start_frames, chitip_frames, restype_icoor_chi;
		chitip_start_frames.reserve( leu_geom.nchi() );
		restype_icoor_chi.reserve( leu_geom.nchi() );

		core::chemical::ResidueType const & res( trpcage.residue_type( 2 ) );

		for ( Size ii = 1; ii <= leu_geom.nchi(); ++ii ) {
			//std::cout << "Chi: " << ii;
			//std::cout << " " << leu_geom.chitip_atoms()[ ii ];
			//std::cout << " " << leu_geom.nonchitip_atoms()[ ii ].size() << std::endl;
			//for ( Size jj = 1; jj <= leu_geom.nonchitip_atoms()[ ii ].size(); ++jj ) {
			//	std::cout << "   " << jj << " " << leu_geom.nonchitip_atoms()[ ii ][ jj ] << std::endl;
			//}
			UpstreamResTypeGeometry::HTReal ht = leu_geom.ht_for_chitip_atoms()[ ii ];
			//std::cout << "ht:" << ii << std::endl;
			//std::cout << "  " << ht.xx() << " " << ht.yx() << " " << ht.zx() << " " << ht.px() << std::endl;
			//std::cout << "  " << ht.xy() << " " << ht.yy() << " " << ht.zy() << " " << ht.py() << std::endl;
			//std::cout << "  " << ht.xz() << " " << ht.yz() << " " << ht.zz() << " " << ht.pz() << std::endl;

			Size const
				chiat1( res.chi_atoms( ii )[ 1 ] ),
				chiat2( res.chi_atoms( ii )[ 2 ] ),
				chiat3( res.chi_atoms( ii )[ 3 ] ),
				chiat4( res.chi_atoms( ii )[ 4 ] );

			chitip_start_frames.push_back( UpstreamResTypeGeometry::HTReal(
				res.xyz( chiat1 ),
				res.xyz( chiat2 ),
				res.xyz( chiat3 ) ));

			chitip_frames.push_back( UpstreamResTypeGeometry::HTReal(
				res.xyz( chiat2 ),
				res.xyz( chiat3 ),
				res.xyz( chiat4 ) ));

			//ht = chitip_start_frames[ ii ];
			//std::cout << "chitip_start_frames:" << ii << std::endl;
			//std::cout << "  " << ht.xx() << " " << ht.yx() << " " << ht.zx() << " " << ht.px() << std::endl;
			//std::cout << "  " << ht.xy() << " " << ht.yy() << " " << ht.zy() << " " << ht.py() << std::endl;
			//std::cout << "  " << ht.xz() << " " << ht.yz() << " " << ht.zz() << " " << ht.pz() << std::endl;


			Real const dihedral = numeric::dihedral_degrees(
				res.xyz( chiat1 ),
				res.xyz( chiat2 ),
				res.xyz( chiat3 ),
				res.xyz( chiat4 )
			);
			//std::cout << "Ideal dihedral " << ii << " " << dihedral << std::endl;

			//std::cout << "Ideal improper bond angle: "<< ii << " " << res.icoor( chiat4 ).theta() << std::endl;

			UpstreamResTypeGeometry::HTReal chi_transform;
			chi_transform.set_zaxis_rotation_deg( dihedral );
			restype_icoor_chi.push_back( chi_transform );

			//ht = restype_icoor_chi[ ii ];
			//std::cout << "restype_icoor_chi:" << ii << std::endl;
			//std::cout << "  " << ht.xx() << " " << ht.yx() << " " << ht.zx() << " " << ht.px() << std::endl;
			//std::cout << "  " << ht.xy() << " " << ht.yy() << " " << ht.zy() << " " << ht.py() << std::endl;
			//std::cout << "  " << ht.xz() << " " << ht.yz() << " " << ht.zz() << " " << ht.pz() << std::endl;

		}

		for ( Size ii = 1; ii <= leu_geom.natoms(); ++ii ) {
			//std::cout << "Atom " << ii << " Icoor: ";
			//std::cout << trpcage.residue_type( 2 ).xyz( ii ).x() << " " <<trpcage.residue_type( 2 ).xyz( ii ).y() << " " <<trpcage.residue_type( 2 ).xyz( ii ).z() << std::endl;
			Vector ideal = trpcage.residue_type( 2 ).xyz( ii );

			Size controlling_chi = leu_geom.controlling_chi_for_atom()[ ii ];
			if ( controlling_chi == 0 ) continue;
			Size which_vector    = leu_geom.which_point_for_atom()[ ii ];
			if ( which_vector == 0 ) continue;

			UpstreamResTypeGeometry::HTReal ht = leu_geom.ht_for_chitip_atoms()[ controlling_chi ];
			UpstreamResTypeGeometry::Vector v  = leu_geom.points_for_nonchitip_atoms()[ controlling_chi ][ which_vector ];
			//std::cout << "Atom " << ii << " v: ";
			//std::cout << v.x() << " " <<v.y() << " " <<v.z() << std::endl;

			//Vector launch_point = ( chitip_start_frames[ controlling_chi ] * restype_icoor_chi[ controlling_chi ] * ht ).point();
			//std::cout << "Atom " << ii << " launch_point: ";
			//std::cout << launch_point.x() << " " <<launch_point.y() << " " <<launch_point.z() << std::endl;

			Vector recreated = chitip_start_frames[ controlling_chi ] * restype_icoor_chi[ controlling_chi ] * ht * v;

			//std::cout << "Atom " << ii << " recreated: ";
			//std::cout << recreated.x() << " " <<recreated.y() << " " <<recreated.z() << std::endl;


			Vector recreated2 = chitip_frames[ controlling_chi ] * v;
			//std::cout << "Atom " << ii << " recreated2: ";
			//std::cout << recreated2.x() << " " <<recreated2.y() << " " <<recreated2.z() << std::endl;
			TS_ASSERT( recreated.distance_squared( ideal ) < 1e-6 );
			TS_ASSERT( recreated.distance_squared( recreated2 ) < 1e-6 );
		}
	}

	void test_build_set_ctor() {
		using namespace core::chemical;

		core::pose::Pose trpcage = create_trpcage_ideal_pose();

		ResidueTypeSet const & res2_set( trpcage.residue( 2 ).residue_type_set() );
		ResidueTypeCAPs const & aas( res2_set.aa_map( aa_phe ));

		BuildSet build_set;

		/// Find the matching phe residue type for residue 2.
		for ( ResidueTypeCAPs::const_iterator
				aas_iter = aas.begin(),
				aas_end = aas.end(); aas_iter != aas_end; ++aas_iter ) {
			if ( trpcage.residue(2).type().variants_match(**aas_iter )) {
				build_set.set_residue_type( *aas_iter );
				break;
			}
		}

		TS_ASSERT( build_set.restype_geometry().N_atom_id()  == 1 );
		TS_ASSERT( build_set.restype_geometry().CA_atom_id() == 2 );
		TS_ASSERT( build_set.restype_geometry().C_atom_id()  == 3 );
		TS_ASSERT( build_set.restype_geometry().O_atom_id()  == 4 );
		TS_ASSERT( build_set.restype_geometry().CB_atom_id() == 5 );

		TS_ASSERT( build_set.restype_geometry().nchi() == 2 );
		TS_ASSERT( build_set.restype_geometry().natoms() == 20 );
	}

	void test_full_sample_set_ctor()
	{
		using namespace core;
		using namespace core::chemical;

		core::pose::Pose trpcage = create_trpcage_ideal_pose();

		OriginalBackboneBuildPointOP res2bp = new OriginalBackboneBuildPoint( trpcage.residue( 2 ), 1 );

		ResidueTypeSet const & res2_set( trpcage.residue( 2 ).residue_type_set() );
		ResidueTypeCAPs const & aas( res2_set.aa_map( aa_phe ));

		BuildSet build_set;

		/// Find the matching phe residue type for residue 2.
		for ( ResidueTypeCAPs::const_iterator
				aas_iter = aas.begin(),
				aas_end = aas.end(); aas_iter != aas_end; ++aas_iter ) {
			if ( trpcage.residue(2).type().variants_match(**aas_iter )) {
				build_set.set_residue_type( *aas_iter );
				break;
			}
		}
		SampleStrategyData strat; strat.set_strategy( rotameric_chi_mimic_EX_flags );
		strat.set_sample_level( core::pack::task::EX_ONE_STDDEV );

		build_set.set_sample_strategy_for_chi( 1, strat );
		build_set.set_sample_strategy_for_chi( 2, strat );

		DunbrackSCSampler dunsampler;
		DunbrackSCSampler::DunbrackRotamerSampleDataVector dun_samps(
			dunsampler.samples( *res2bp, build_set.restype() ));
		FullChiSampleSet expanded_samples( build_set, dun_samps[ 1 ], false );


		/*
		for ( Size ii = 1; ii <= 2; ++ii ) {
			//std::cout << "Chi " << ii << " with " << expanded_samples.n_samples_per_chi()[ ii ] << ":" << std::endl;
			std::cout << "TS_ASSERT( expanded_samples.n_samples_per_chi()[ " << ii << " ] = " << expanded_samples.n_samples_per_chi()[ ii ] << " );" << std::endl;
			for ( Size jj = 1; jj <= expanded_samples.n_samples_per_chi()[ ii ]; ++jj ) {
				//std::cout << jj << ": " << expanded_samples.chi_sample( ii, jj ) << std::endl;
				std::cout << "TS_ASSERT_DELTA( expanded_samples.chi_sample( " << ii << ", " << jj << " ), " << expanded_samples.chi_sample( ii, jj ) << ", 1e-3 );" << std::endl;
			}
		}*/

		TS_ASSERT( expanded_samples.n_samples_per_chi()[ 1 ] = 3 );
		TS_ASSERT_DELTA( expanded_samples.chi_sample( 1, 1 ), 168.735, 3e-3 );
		TS_ASSERT_DELTA( expanded_samples.chi_sample( 1, 2 ), 178.019, 3e-3 );
		TS_ASSERT_DELTA( expanded_samples.chi_sample( 1, 3 ), 187.303, 3e-3 );

		TS_ASSERT( expanded_samples.n_samples_per_chi()[ 2 ] = 3 );
		TS_ASSERT_DELTA( expanded_samples.chi_sample( 2, 1 ), 68.3786, 3e-3 );
		TS_ASSERT_DELTA( expanded_samples.chi_sample( 2, 2 ), 80.3798, 3e-3 );
		TS_ASSERT_DELTA( expanded_samples.chi_sample( 2, 3 ), 92.381,  3e-3 );
	}

	void test_ProteinUpstreamBuilder_build()
	{
		using namespace core;
		using namespace core::chemical;

		core::pose::Pose trpcage = create_trpcage_ideal_pose();

		OriginalBackboneBuildPointOP res2bp = new OriginalBackboneBuildPoint( trpcage.residue( 2 ), 1 );

		ResidueTypeSet const & res2_set( trpcage.residue( 2 ).residue_type_set() );
		ResidueTypeCAPs const & aas( res2_set.aa_map( aa_phe ));

		BuildSet build_set;

		/// Find the matching phe residue type for residue 2.
		for ( ResidueTypeCAPs::const_iterator
				aas_iter = aas.begin(),
				aas_end = aas.end(); aas_iter != aas_end; ++aas_iter ) {
			if ( trpcage.residue(2).type().variants_match(**aas_iter )) {
				build_set.set_residue_type( *aas_iter );
				break;
			}
		}
		SampleStrategyData strat; strat.set_strategy( rotameric_chi_mimic_EX_flags );
		strat.set_sample_level( core::pack::task::EX_ONE_STDDEV );

		build_set.set_sample_strategy_for_chi( 1, strat );
		build_set.set_sample_strategy_for_chi( 2, strat );

		std::ostringstream sout;

		WriteUpstreamCoordinateKinemageOP dsalgorithm = new WriteUpstreamCoordinateKinemage( sout );

		build_set.set_downstream_algorithm( dsalgorithm );

		ProteinUpstreamBuilder scbuilder;
		scbuilder.add_build_set( build_set );
		scbuilder.set_sampler( new DunbrackSCSampler );

		BumpGridOP bbgrid = new BumpGrid;
		scbuilder.set_bb_grid( bbgrid );

		scbuilder.build( *res2bp );

		std::string correct_kinemage =
			"@kinemage {1}\n"
			"@title { matcher }\n"
			"@1center -4.3903 5.36323 -2.52079\n"
			"@1span 25\n"
			"@group { rot1 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -2.88935 5.37612 -2.58147\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.74274 5.87776 -3.41379\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -4.67091 5.93644 -1.63817\n"
			"{PHE 2  CG  (aroC)}P -2.88935 5.37612 -2.58147\n"
			"{PHE 2  CD1 (aroC)} -2.22223 4.90542 -3.70266\n"
			"{PHE 2  CG  (aroC)}P -2.88935 5.37612 -2.58147\n"
			"{PHE 2  CD2 (aroC)} -2.1415 5.85796 -1.51789\n"
			"{PHE 2  CD1 (aroC)}P -2.22223 4.90542 -3.70266\n"
			"{PHE 2  CE1 (aroC)} -0.841269 4.91666 -3.75952\n"
			"{PHE 2  CD1 (aroC)}P 'h' -2.22223 4.90542 -3.70266\n"
			"{PHE 2  HD1 (Haro)} 'h' -2.79998 4.52436 -4.54516\n"
			"{PHE 2  CD2 (aroC)}P -2.1415 5.85796 -1.51789\n"
			"{PHE 2  CE2 (aroC)} -0.761363 5.87156 -1.57235\n"
			"{PHE 2  CD2 (aroC)}P 'h' -2.1415 5.85796 -1.51789\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.65554 6.23023 -0.630985\n"
			"{PHE 2  CE1 (aroC)}P -0.841269 4.91666 -3.75952\n"
			"{PHE 2  CZ  (aroC)} -0.110832 5.39981 -2.69485\n"
			"{PHE 2  CE1 (aroC)}P 'h' -0.841269 4.91666 -3.75952\n"
			"{PHE 2  HE1 (Haro)} 'h' -0.329952 4.5436 -4.64657\n"
			"{PHE 2  CE2 (aroC)}P -0.761363 5.87156 -1.57235\n"
			"{PHE 2  CZ  (aroC)} -0.110832 5.39981 -2.69485\n"
			"{PHE 2  CE2 (aroC)}P 'h' -0.761363 5.87156 -1.57235\n"
			"{PHE 2  HE2 (Haro)} 'h' -0.185372 6.25357 -0.729709\n"
			"{PHE 2  CZ  (aroC)}P 'h' -0.110832 5.39981 -2.69485\n"
			"{PHE 2  HZ  (Haro)} 'h' 0.97735 5.40812 -2.73835\n"
			"@group { rot2 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -2.88935 5.37612 -2.58147\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.74274 5.87776 -3.41379\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -4.67091 5.93644 -1.63817\n"
			"{PHE 2  CG  (aroC)}P -2.88935 5.37612 -2.58147\n"
			"{PHE 2  CD1 (aroC)} -2.22731 5.14317 -3.77776\n"
			"{PHE 2  CG  (aroC)}P -2.88935 5.37612 -2.58147\n"
			"{PHE 2  CD2 (aroC)} -2.13643 5.62032 -1.44299\n"
			"{PHE 2  CD1 (aroC)}P -2.22731 5.14317 -3.77776\n"
			"{PHE 2  CE1 (aroC)} -0.846352 5.15464 -3.83472\n"
			"{PHE 2  CD1 (aroC)}P 'h' -2.22731 5.14317 -3.77776\n"
			"{PHE 2  HD1 (Haro)} 'h' -2.80907 4.95022 -4.6795\n"
			"{PHE 2  CD2 (aroC)}P -2.13643 5.62032 -1.44299\n"
			"{PHE 2  CE2 (aroC)} -0.756278 5.6336 -1.49711\n"
			"{PHE 2  CD2 (aroC)}P 'h' -2.13643 5.62032 -1.44299\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.64645 5.80441 -0.496731\n"
			"{PHE 2  CE1 (aroC)}P -0.846352 5.15464 -3.83472\n"
			"{PHE 2  CZ  (aroC)} -0.110835 5.40004 -2.69486\n"
			"{PHE 2  CE1 (aroC)}P 'h' -0.846352 5.15464 -3.83472\n"
			"{PHE 2  HE1 (Haro)} 'h' -0.339058 4.96983 -4.78128\n"
			"{PHE 2  CE2 (aroC)}P -0.756278 5.6336 -1.49711\n"
			"{PHE 2  CZ  (aroC)} -0.110835 5.40004 -2.69486\n"
			"{PHE 2  CE2 (aroC)}P 'h' -0.756278 5.6336 -1.49711\n"
			"{PHE 2  HE2 (Haro)} 'h' -0.176269 5.82746 -0.595038\n"
			"{PHE 2  CZ  (aroC)}P 'h' -0.110835 5.40004 -2.69486\n"
			"{PHE 2  HZ  (Haro)} 'h' 0.977339 5.40826 -2.73859\n"
			"@group { rot3 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -2.88935 5.37612 -2.58147\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.74274 5.87776 -3.41379\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -4.67091 5.93644 -1.63817\n"
			"{PHE 2  CG  (aroC)}P -2.88935 5.37612 -2.58147\n"
			"{PHE 2  CD1 (aroC)} -2.23041 5.39137 -3.80182\n"
			"{PHE 2  CG  (aroC)}P -2.88935 5.37612 -2.58147\n"
			"{PHE 2  CD2 (aroC)} -2.13334 5.37227 -1.41911\n"
			"{PHE 2  CD1 (aroC)}P -2.23041 5.39137 -3.80182\n"
			"{PHE 2  CE1 (aroC)} -0.84946 5.40308 -3.85883\n"
			"{PHE 2  CD1 (aroC)}P 'h' -2.23041 5.39137 -3.80182\n"
			"{PHE 2  HD1 (Haro)} 'h' -2.81462 5.39476 -4.72241\n"
			"{PHE 2  CD2 (aroC)}P -2.13334 5.37227 -1.41911\n"
			"{PHE 2  CE2 (aroC)} -0.753169 5.38517 -1.47297\n"
			"{PHE 2  CD2 (aroC)}P 'h' -2.13334 5.37227 -1.41911\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.6409 5.35992 -0.453895\n"
			"{PHE 2  CE1 (aroC)}P -0.84946 5.40308 -3.85883\n"
			"{PHE 2  CZ  (aroC)} -0.110835 5.40025 -2.69483\n"
			"{PHE 2  CE1 (aroC)}P 'h' -0.84946 5.40308 -3.85883\n"
			"{PHE 2  HE1 (Haro)} 'h' -0.344625 5.4148 -4.82449\n"
			"{PHE 2  CE2 (aroC)}P -0.753169 5.38517 -1.47297\n"
			"{PHE 2  CZ  (aroC)} -0.110835 5.40025 -2.69483\n"
			"{PHE 2  CE2 (aroC)}P 'h' -0.753169 5.38517 -1.47297\n"
			"{PHE 2  HE2 (Haro)} 'h' -0.170703 5.38261 -0.551853\n"
			"{PHE 2  CZ  (aroC)}P 'h' -0.110835 5.40025 -2.69483\n"
			"{PHE 2  HZ  (Haro)} 'h' 0.977329 5.40845 -2.73879\n"
			"@group { rot4 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -2.91477 5.37573 -2.8025\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.86453 5.93496 -3.31743\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -4.53026 5.87946 -1.57198\n"
			"{PHE 2  CG  (aroC)}P -2.91477 5.37573 -2.8025\n"
			"{PHE 2  CD1 (aroC)} -2.42659 4.97601 -4.03764\n"
			"{PHE 2  CG  (aroC)}P -2.91477 5.37573 -2.8025\n"
			"{PHE 2  CD2 (aroC)} -2.01202 5.78626 -1.83339\n"
			"{PHE 2  CD1 (aroC)}P -2.42659 4.97601 -4.03764\n"
			"{PHE 2  CE1 (aroC)} -1.06918 4.98696 -4.29788\n"
			"{PHE 2  CD1 (aroC)}P 'h' -2.42659 4.97601 -4.03764\n"
			"{PHE 2  HD1 (Haro)} 'h' -3.1272 4.6514 -4.80744\n"
			"{PHE 2  CD2 (aroC)}P -2.01202 5.78626 -1.83339\n"
			"{PHE 2  CE2 (aroC)} -0.655041 5.79941 -2.09099\n"
			"{PHE 2  CD2 (aroC)}P 'h' -2.01202 5.78626 -1.83339\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.38466 6.10234 -0.858383\n"
			"{PHE 2  CE1 (aroC)}P -1.06918 4.98696 -4.29788\n"
			"{PHE 2  CZ  (aroC)} -0.183472 5.39877 -3.32504\n"
			"{PHE 2  CE1 (aroC)}P 'h' -1.06918 4.98696 -4.29788\n"
			"{PHE 2  HE1 (Haro)} 'h' -0.699255 4.6701 -5.27268\n"
			"{PHE 2  CE2 (aroC)}P -0.655041 5.79941 -2.09099\n"
			"{PHE 2  CZ  (aroC)} -0.183472 5.39877 -3.32504\n"
			"{PHE 2  CE2 (aroC)}P 'h' -0.655041 5.79941 -2.09099\n"
			"{PHE 2  HE2 (Haro)} 'h' 0.0438573 6.12496 -1.32074\n"
			"{PHE 2  CZ  (aroC)}P 'h' -0.183472 5.39877 -3.32504\n"
			"{PHE 2  HZ  (Haro)} 'h' 0.88634 5.40677 -3.52886\n"
			"@group { rot5 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -2.91477 5.37573 -2.8025\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.86453 5.93496 -3.31743\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -4.53026 5.87946 -1.57198\n"
			"{PHE 2  CG  (aroC)}P -2.91477 5.37573 -2.8025\n"
			"{PHE 2  CD1 (aroC)} -2.43979 5.2181 -4.09603\n"
			"{PHE 2  CG  (aroC)}P -2.91477 5.37573 -2.8025\n"
			"{PHE 2  CD2 (aroC)} -1.99886 5.54429 -1.77519\n"
			"{PHE 2  CD1 (aroC)}P -2.43979 5.2181 -4.09603\n"
			"{PHE 2  CE1 (aroC)} -1.08239 5.22928 -4.35636\n"
			"{PHE 2  CD1 (aroC)}P 'h' -2.43979 5.2181 -4.09603\n"
			"{PHE 2  HD1 (Haro)} 'h' -3.1508 5.08502 -4.91184\n"
			"{PHE 2  CD2 (aroC)}P -1.99886 5.54429 -1.77519\n"
			"{PHE 2  CE2 (aroC)} -0.641819 5.5571 -2.03249\n"
			"{PHE 2  CD2 (aroC)}P 'h' -1.99886 5.54429 -1.77519\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.36107 5.66876 -0.754057\n"
			"{PHE 2  CE1 (aroC)}P -1.08239 5.22928 -4.35636\n"
			"{PHE 2  CZ  (aroC)} -0.183473 5.39899 -3.32504\n"
			"{PHE 2  CE1 (aroC)}P 'h' -1.08239 5.22928 -4.35636\n"
			"{PHE 2  HE1 (Haro)} 'h' -0.722931 5.10411 -5.37744\n"
			"{PHE 2  CE2 (aroC)}P -0.641819 5.5571 -2.03249\n"
			"{PHE 2  CZ  (aroC)} -0.183473 5.39899 -3.32504\n"
			"{PHE 2  CE2 (aroC)}P 'h' -0.641819 5.5571 -2.03249\n"
			"{PHE 2  HE2 (Haro)} 'h' 0.0675251 5.69107 -1.21603\n"
			"{PHE 2  CZ  (aroC)}P 'h' -0.183473 5.39899 -3.32504\n"
			"{PHE 2  HZ  (Haro)} 'h' 0.886296 5.40692 -3.52908\n"
			"@group { rot6 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -2.91477 5.37573 -2.8025\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.86453 5.93496 -3.31743\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -4.53026 5.87946 -1.57198\n"
			"{PHE 2  CG  (aroC)}P -2.91477 5.37573 -2.8025\n"
			"{PHE 2  CD1 (aroC)} -2.44336 5.46734 -4.10367\n"
			"{PHE 2  CG  (aroC)}P -2.91477 5.37573 -2.8025\n"
			"{PHE 2  CD2 (aroC)} -1.99532 5.29521 -1.76771\n"
			"{PHE 2  CD1 (aroC)}P -2.44336 5.46734 -4.10367\n"
			"{PHE 2  CE1 (aroC)} -1.08597 5.47876 -4.36404\n"
			"{PHE 2  CD1 (aroC)}P 'h' -2.44336 5.46734 -4.10367\n"
			"{PHE 2  HD1 (Haro)} 'h' -3.15717 5.53141 -4.92537\n"
			"{PHE 2  CD2 (aroC)}P -1.99532 5.29521 -1.76771\n"
			"{PHE 2  CE2 (aroC)} -0.638233 5.30763 -2.02477\n"
			"{PHE 2  CD2 (aroC)}P 'h' -1.99532 5.29521 -1.76771\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.35472 5.22243 -0.740604\n"
			"{PHE 2  CE1 (aroC)}P -1.08597 5.47876 -4.36404\n"
			"{PHE 2  CZ  (aroC)} -0.183465 5.39921 -3.32499\n"
			"{PHE 2  CE1 (aroC)}P 'h' -1.08597 5.47876 -4.36404\n"
			"{PHE 2  HE1 (Haro)} 'h' -0.729347 5.55095 -5.39122\n"
			"{PHE 2  CE2 (aroC)}P -0.638233 5.30763 -2.02477\n"
			"{PHE 2  CZ  (aroC)} -0.183465 5.39921 -3.32499\n"
			"{PHE 2  CE2 (aroC)}P 'h' -0.638233 5.30763 -2.02477\n"
			"{PHE 2  HE2 (Haro)} 'h' 0.0739382 5.24435 -1.20225\n"
			"{PHE 2  CZ  (aroC)}P 'h' -0.183465 5.39921 -3.32499\n"
			"{PHE 2  HZ  (Haro)} 'h' 0.886259 5.40712 -3.52927\n"
			"@group { rot7 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -2.97259 5.38959 -3.0169\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.97002 5.9862 -3.20066\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -4.38208 5.81795 -1.5311\n"
			"{PHE 2  CG  (aroC)}P -2.97259 5.38959 -3.0169\n"
			"{PHE 2  CD1 (aroC)} -2.67734 5.06851 -4.33348\n"
			"{PHE 2  CG  (aroC)}P -2.97259 5.38959 -3.0169\n"
			"{PHE 2  CD2 (aroC)} -1.93151 5.73457 -2.16849\n"
			"{PHE 2  CD1 (aroC)}P -2.67734 5.06851 -4.33348\n"
			"{PHE 2  CE1 (aroC)} -1.37329 5.09227 -4.79097\n"
			"{PHE 2  CD1 (aroC)}P 'h' -2.67734 5.06851 -4.33348\n"
			"{PHE 2  HD1 (Haro)} 'h' -3.488 4.79591 -5.00975\n"
			"{PHE 2  CD2 (aroC)}P -1.93151 5.73457 -2.16849\n"
			"{PHE 2  CE2 (aroC)} -0.627463 5.76037 -2.62318\n"
			"{PHE 2  CD2 (aroC)}P 'h' -1.93151 5.73457 -2.16849\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.15198 5.98854 -1.13104\n"
			"{PHE 2  CE1 (aroC)}P -1.37329 5.09227 -4.79097\n"
			"{PHE 2  CZ  (aroC)} -0.34849 5.43835 -3.93628\n"
			"{PHE 2  CE1 (aroC)}P 'h' -1.37329 5.09227 -4.79097\n"
			"{PHE 2  HE1 (Haro)} 'h' -1.15548 4.83753 -5.82787\n"
			"{PHE 2  CE2 (aroC)}P -0.627463 5.76037 -2.62318\n"
			"{PHE 2  CZ  (aroC)} -0.34849 5.43835 -3.93628\n"
			"{PHE 2  CE2 (aroC)}P 'h' -0.627463 5.76037 -2.62318\n"
			"{PHE 2  HE2 (Haro)} 'h' 0.181586 6.03387 -1.94616\n"
			"{PHE 2  CZ  (aroC)}P 'h' -0.34849 5.43835 -3.93628\n"
			"{PHE 2  HZ  (Haro)} 'h' 0.679449 5.45636 -4.29561\n"
			"@group { rot8 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -2.97259 5.38959 -3.0169\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.97002 5.9862 -3.20066\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -4.38208 5.81795 -1.5311\n"
			"{PHE 2  CG  (aroC)}P -2.97259 5.38959 -3.0169\n"
			"{PHE 2  CD1 (aroC)} -2.69605 5.31388 -4.3739\n"
			"{PHE 2  CG  (aroC)}P -2.97259 5.38959 -3.0169\n"
			"{PHE 2  CD2 (aroC)} -1.91287 5.48933 -2.12825\n"
			"{PHE 2  CD1 (aroC)}P -2.69605 5.31388 -4.3739\n"
			"{PHE 2  CE1 (aroC)} -1.39202 5.33788 -4.83146\n"
			"{PHE 2  CD1 (aroC)}P 'h' -2.69605 5.31388 -4.3739\n"
			"{PHE 2  HD1 (Haro)} 'h' -3.52145 5.2354 -5.08197\n"
			"{PHE 2  CD2 (aroC)}P -1.91287 5.48933 -2.12825\n"
			"{PHE 2  CE2 (aroC)} -0.60872 5.51477 -2.58267\n"
			"{PHE 2  CD2 (aroC)}P 'h' -1.91287 5.48933 -2.12825\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.11857 5.54909 -1.05889\n"
			"{PHE 2  CE1 (aroC)}P -1.39202 5.33788 -4.83146\n"
			"{PHE 2  CZ  (aroC)} -0.348488 5.43857 -3.93626\n"
			"{PHE 2  CE1 (aroC)}P 'h' -1.39202 5.33788 -4.83146\n"
			"{PHE 2  HE1 (Haro)} 'h' -1.18904 5.27743 -5.9004\n"
			"{PHE 2  CE2 (aroC)}P -0.60872 5.51477 -2.58267\n"
			"{PHE 2  CZ  (aroC)} -0.348488 5.43857 -3.93626\n"
			"{PHE 2  CE2 (aroC)}P 'h' -0.60872 5.51477 -2.58267\n"
			"{PHE 2  HE2 (Haro)} 'h' 0.215133 5.59409 -1.87365\n"
			"{PHE 2  CZ  (aroC)}P 'h' -0.348488 5.43857 -3.93626\n"
			"{PHE 2  HZ  (Haro)} 'h' 0.679374 5.45653 -4.29582\n"
			"@group { rot9 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -2.97259 5.38959 -3.0169\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.97002 5.9862 -3.20066\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -4.38208 5.81795 -1.5311\n"
			"{PHE 2  CG  (aroC)}P -2.97259 5.38959 -3.0169\n"
			"{PHE 2  CD1 (aroC)} -2.69764 5.5631 -4.36521\n"
			"{PHE 2  CG  (aroC)}P -2.97259 5.38959 -3.0169\n"
			"{PHE 2  CD2 (aroC)} -1.91133 5.24027 -2.13708\n"
			"{PHE 2  CD1 (aroC)}P -2.69764 5.5631 -4.36521\n"
			"{PHE 2  CE1 (aroC)} -1.39363 5.58736 -4.8228\n"
			"{PHE 2  CD1 (aroC)}P 'h' -2.69764 5.5631 -4.36521\n"
			"{PHE 2  HD1 (Haro)} 'h' -3.52425 5.68176 -5.06625\n"
			"{PHE 2  CD2 (aroC)}P -1.91133 5.24027 -2.13708\n"
			"{PHE 2  CE2 (aroC)} -0.607104 5.2653 -2.5913\n"
			"{PHE 2  CD2 (aroC)}P 'h' -1.91133 5.24027 -2.13708\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.11579 5.1028 -1.07467\n"
			"{PHE 2  CE1 (aroC)}P -1.39363 5.58736 -4.8228\n"
			"{PHE 2  CZ  (aroC)} -0.34847 5.43878 -3.9362\n"
			"{PHE 2  CE1 (aroC)}P 'h' -1.39363 5.58736 -4.8228\n"
			"{PHE 2  HE1 (Haro)} 'h' -1.19192 5.72425 -5.8849\n"
			"{PHE 2  CE2 (aroC)}P -0.607104 5.2653 -2.5913\n"
			"{PHE 2  CZ  (aroC)} -0.34847 5.43878 -3.9362\n"
			"{PHE 2  CE2 (aroC)}P 'h' -0.607104 5.2653 -2.5913\n"
			"{PHE 2  HE2 (Haro)} 'h' 0.218013 5.1474 -1.88915\n"
			"{PHE 2  CZ  (aroC)}P 'h' -0.34847 5.43878 -3.9362\n"
			"{PHE 2  HZ  (Haro)} 'h' 0.679312 5.45675 -4.29598\n"
			"@group { rot10 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -4.44241 5.97572 -3.8915\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.93001 6.03771 -1.85733\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -3.34037 5.33498 -2.23237\n"
			"{PHE 2  CG  (aroC)}P -4.44241 5.97572 -3.8915\n"
			"{PHE 2  CD1 (aroC)} -5.60721 6.56569 -4.35931\n"
			"{PHE 2  CG  (aroC)}P -4.44241 5.97572 -3.8915\n"
			"{PHE 2  CD2 (aroC)} -3.32769 5.96269 -4.71606\n"
			"{PHE 2  CD1 (aroC)}P -5.60721 6.56569 -4.35931\n"
			"{PHE 2  CE1 (aroC)} -5.65633 7.1294 -5.62035\n"
			"{PHE 2  CD1 (aroC)}P 'h' -5.60721 6.56569 -4.35931\n"
			"{PHE 2  HD1 (Haro)} 'h' -6.48995 6.58199 -3.71953\n"
			"{PHE 2  CD2 (aroC)}P -3.32769 5.96269 -4.71606\n"
			"{PHE 2  CE2 (aroC)} -3.37345 6.52632 -5.97628\n"
			"{PHE 2  CD2 (aroC)}P 'h' -3.32769 5.96269 -4.71606\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.40647 5.50112 -4.35869\n"
			"{PHE 2  CE1 (aroC)}P -5.65633 7.1294 -5.62035\n"
			"{PHE 2  CZ  (aroC)} -4.53978 7.11006 -6.42869\n"
			"{PHE 2  CE1 (aroC)}P 'h' -5.65633 7.1294 -5.62035\n"
			"{PHE 2  HE1 (Haro)} 'h' -6.57816 7.58947 -5.97539\n"
			"{PHE 2  CE2 (aroC)}P -3.37345 6.52632 -5.97628\n"
			"{PHE 2  CZ  (aroC)} -4.53978 7.11006 -6.42869\n"
			"{PHE 2  CE2 (aroC)}P 'h' -3.37345 6.52632 -5.97628\n"
			"{PHE 2  HE2 (Haro)} 'h' -2.48998 6.50994 -6.6142\n"
			"{PHE 2  CZ  (aroC)}P 'h' -4.53978 7.11006 -6.42869\n"
			"{PHE 2  HZ  (Haro)} 'h' -4.57786 7.55309 -7.42287\n"
			"@group { rot11 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -4.44241 5.97572 -3.8915\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.93001 6.03771 -1.85733\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -3.34037 5.33498 -2.23237\n"
			"{PHE 2  CG  (aroC)}P -4.44241 5.97572 -3.8915\n"
			"{PHE 2  CD1 (aroC)} -5.42106 6.90363 -4.21538\n"
			"{PHE 2  CG  (aroC)}P -4.44241 5.97572 -3.8915\n"
			"{PHE 2  CD2 (aroC)} -3.51395 5.62507 -4.85984\n"
			"{PHE 2  CD1 (aroC)}P -5.42106 6.90363 -4.21538\n"
			"{PHE 2  CE1 (aroC)} -5.47005 7.46769 -5.47627\n"
			"{PHE 2  CD1 (aroC)}P 'h' -5.42106 6.90363 -4.21538\n"
			"{PHE 2  HD1 (Haro)} 'h' -6.1563 7.18713 -3.46181\n"
			"{PHE 2  CD2 (aroC)}P -3.51395 5.62507 -4.85984\n"
			"{PHE 2  CE2 (aroC)} -3.55969 6.18801 -6.12037\n"
			"{PHE 2  CD2 (aroC)}P 'h' -3.51395 5.62507 -4.85984\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.74017 4.89611 -4.61635\n"
			"{PHE 2  CE1 (aroC)}P -5.47005 7.46769 -5.47627\n"
			"{PHE 2  CZ  (aroC)} -4.53954 7.11032 -6.42859\n"
			"{PHE 2  CE1 (aroC)}P 'h' -5.47005 7.46769 -5.47627\n"
			"{PHE 2  HE1 (Haro)} 'h' -6.24454 8.19539 -5.71732\n"
			"{PHE 2  CE2 (aroC)}P -3.55969 6.18801 -6.12037\n"
			"{PHE 2  CZ  (aroC)} -4.53954 7.11032 -6.42859\n"
			"{PHE 2  CE2 (aroC)}P 'h' -3.55969 6.18801 -6.12037\n"
			"{PHE 2  HE2 (Haro)} 'h' -2.82352 5.90419 -6.87219\n"
			"{PHE 2  CZ  (aroC)}P 'h' -4.53954 7.11032 -6.42859\n"
			"{PHE 2  HZ  (Haro)} 'h' -4.57797 7.55349 -7.42268\n"
			"@group { rot12 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -4.44241 5.97572 -3.8915\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.93001 6.03771 -1.85733\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -3.34037 5.33498 -2.23237\n"
			"{PHE 2  CG  (aroC)}P -4.44241 5.97572 -3.8915\n"
			"{PHE 2  CD1 (aroC)} -5.12119 7.16537 -4.10982\n"
			"{PHE 2  CG  (aroC)}P -4.44241 5.97572 -3.8915\n"
			"{PHE 2  CD2 (aroC)} -3.81381 5.36368 -4.96524\n"
			"{PHE 2  CD1 (aroC)}P -5.12119 7.16537 -4.10982\n"
			"{PHE 2  CE1 (aroC)} -5.16992 7.72973 -5.37059\n"
			"{PHE 2  CD1 (aroC)}P 'h' -5.12119 7.16537 -4.10982\n"
			"{PHE 2  HD1 (Haro)} 'h' -5.61904 7.65571 -3.27285\n"
			"{PHE 2  CD2 (aroC)}P -3.81381 5.36368 -4.96524\n"
			"{PHE 2  CE2 (aroC)} -3.85978 5.92595 -6.22606\n"
			"{PHE 2  CD2 (aroC)}P 'h' -3.81381 5.36368 -4.96524\n"
			"{PHE 2  HD2 (Haro)} 'h' -3.27743 4.42767 -4.80524\n"
			"{PHE 2  CE1 (aroC)}P -5.16992 7.72973 -5.37059\n"
			"{PHE 2  CZ  (aroC)} -4.53922 7.11048 -6.42853\n"
			"{PHE 2  CE1 (aroC)}P 'h' -5.16992 7.72973 -5.37059\n"
			"{PHE 2  HE1 (Haro)} 'h' -5.707 8.66473 -5.52803\n"
			"{PHE 2  CE2 (aroC)}P -3.85978 5.92595 -6.22606\n"
			"{PHE 2  CZ  (aroC)} -4.53922 7.11048 -6.42853\n"
			"{PHE 2  CE2 (aroC)}P 'h' -3.85978 5.92595 -6.22606\n"
			"{PHE 2  HE2 (Haro)} 'h' -3.36091 5.43499 -7.06142\n"
			"{PHE 2  CZ  (aroC)}P 'h' -4.53922 7.11048 -6.42853\n"
			"{PHE 2  HZ  (Haro)} 'h' -4.57793 7.55391 -7.4225\n"
			"@group { rot13 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -4.68824 6.08572 -3.80373\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.78408 5.98232 -1.71577\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -3.30554 5.30948 -2.4387\n"
			"{PHE 2  CG  (aroC)}P -4.68824 6.08572 -3.80373\n"
			"{PHE 2  CD1 (aroC)} -5.90978 6.71419 -3.9949\n"
			"{PHE 2  CG  (aroC)}P -4.68824 6.08572 -3.80373\n"
			"{PHE 2  CD2 (aroC)} -3.74857 6.13791 -4.82203\n"
			"{PHE 2  CD1 (aroC)}P -5.90978 6.71419 -3.9949\n"
			"{PHE 2  CE1 (aroC)} -6.18505 7.3791 -5.17496\n"
			"{PHE 2  CD1 (aroC)}P 'h' -5.90978 6.71419 -3.9949\n"
			"{PHE 2  HD1 (Haro)} 'h' -6.65628 6.6799 -3.20094\n"
			"{PHE 2  CD2 (aroC)}P -3.74857 6.13791 -4.82203\n"
			"{PHE 2  CE2 (aroC)} -4.02038 6.80267 -6.00193\n"
			"{PHE 2  CD2 (aroC)}P 'h' -3.74857 6.13791 -4.82203\n"
			"{PHE 2  HD2 (Haro)} 'h' -2.78475 5.64691 -4.68279\n"
			"{PHE 2  CE1 (aroC)}P -6.18505 7.3791 -5.17496\n"
			"{PHE 2  CZ  (aroC)} -5.24063 7.42368 -6.17824\n"
			"{PHE 2  CE1 (aroC)}P 'h' -6.18505 7.3791 -5.17496\n"
			"{PHE 2  HE1 (Haro)} 'h' -7.14904 7.86842 -5.31194\n"
			"{PHE 2  CE2 (aroC)}P -4.02038 6.80267 -6.00193\n"
			"{PHE 2  CZ  (aroC)} -5.24063 7.42368 -6.17824\n"
			"{PHE 2  CE2 (aroC)}P 'h' -4.02038 6.80267 -6.00193\n"
			"{PHE 2  HE2 (Haro)} 'h' -3.27282 6.83673 -6.79421\n"
			"{PHE 2  CZ  (aroC)}P 'h' -5.24063 7.42368 -6.17824\n"
			"{PHE 2  HZ  (Haro)} 'h' -5.45703 7.94649 -7.1088\n"
			"@group { rot14 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -4.68824 6.08572 -3.80373\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.78408 5.98232 -1.71577\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -3.30554 5.30948 -2.4387\n"
			"{PHE 2  CG  (aroC)}P -4.68824 6.08572 -3.80373\n"
			"{PHE 2  CD1 (aroC)} -5.6941 7.03888 -3.86214\n"
			"{PHE 2  CG  (aroC)}P -4.68824 6.08572 -3.80373\n"
			"{PHE 2  CD2 (aroC)} -3.96432 5.81353 -4.9546\n"
			"{PHE 2  CD1 (aroC)}P -5.6941 7.03888 -3.86214\n"
			"{PHE 2  CE1 (aroC)} -5.9692 7.70413 -5.04205\n"
			"{PHE 2  CD1 (aroC)}P 'h' -5.6941 7.03888 -3.86214\n"
			"{PHE 2  HD1 (Haro)} 'h' -6.26976 7.26131 -2.96328\n"
			"{PHE 2  CD2 (aroC)}P -3.96432 5.81353 -4.9546\n"
			"{PHE 2  CE2 (aroC)} -4.23618 6.47762 -6.13486\n"
			"{PHE 2  CD2 (aroC)}P 'h' -3.96432 5.81353 -4.9546\n"
			"{PHE 2  HD2 (Haro)} 'h' -3.17131 5.06562 -4.92037\n"
			"{PHE 2  CE1 (aroC)}P -5.9692 7.70413 -5.04205\n"
			"{PHE 2  CZ  (aroC)} -5.24037 7.42392 -6.17817\n"
			"{PHE 2  CE1 (aroC)}P 'h' -5.9692 7.70413 -5.04205\n"
			"{PHE 2  HE1 (Haro)} 'h' -6.76247 8.45059 -5.07386\n"
			"{PHE 2  CE2 (aroC)}P -4.23618 6.47762 -6.13486\n"
			"{PHE 2  CZ  (aroC)} -5.24037 7.42392 -6.17817\n"
			"{PHE 2  CE2 (aroC)}P 'h' -4.23618 6.47762 -6.13486\n"
			"{PHE 2  HE2 (Haro)} 'h' -3.65929 6.25472 -7.03222\n"
			"{PHE 2  CZ  (aroC)}P 'h' -5.24037 7.42392 -6.17817\n"
			"{PHE 2  HZ  (Haro)} 'h' -5.4571 7.94688 -7.10857\n"
			"@group { rot15 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -4.68824 6.08572 -3.80373\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.78408 5.98232 -1.71577\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -3.30554 5.30948 -2.4387\n"
			"{PHE 2  CG  (aroC)}P -4.68824 6.08572 -3.80373\n"
			"{PHE 2  CD1 (aroC)} -5.37526 7.29054 -3.79446\n"
			"{PHE 2  CG  (aroC)}P -4.68824 6.08572 -3.80373\n"
			"{PHE 2  CD2 (aroC)} -4.28311 5.5622 -5.02211\n"
			"{PHE 2  CD1 (aroC)}P -5.37526 7.29054 -3.79446\n"
			"{PHE 2  CE1 (aroC)} -5.65008 7.95608 -4.97427\n"
			"{PHE 2  CD1 (aroC)}P 'h' -5.37526 7.29054 -3.79446\n"
			"{PHE 2  HD1 (Haro)} 'h' -5.69855 7.71186 -2.84221\n"
			"{PHE 2  CD2 (aroC)}P -4.28311 5.5622 -5.02211\n"
			"{PHE 2  CE2 (aroC)} -4.55526 6.22565 -6.20266\n"
			"{PHE 2  CD2 (aroC)}P 'h' -4.28311 5.5622 -5.02211\n"
			"{PHE 2  HD2 (Haro)} 'h' -3.74251 4.61521 -5.04137\n"
			"{PHE 2  CE1 (aroC)}P -5.65008 7.95608 -4.97427\n"
			"{PHE 2  CZ  (aroC)} -5.24004 7.42408 -6.17816\n"
			"{PHE 2  CE1 (aroC)}P 'h' -5.65008 7.95608 -4.97427\n"
			"{PHE 2  HE1 (Haro)} 'h' -6.19092 8.90186 -4.95246\n"
			"{PHE 2  CE2 (aroC)}P -4.55526 6.22565 -6.20266\n"
			"{PHE 2  CZ  (aroC)} -5.24004 7.42408 -6.17816\n"
			"{PHE 2  CE2 (aroC)}P 'h' -4.55526 6.22565 -6.20266\n"
			"{PHE 2  HE2 (Haro)} 'h' -4.23068 5.80359 -7.15358\n"
			"{PHE 2  CZ  (aroC)}P 'h' -5.24004 7.42408 -6.17816\n"
			"{PHE 2  HZ  (Haro)} 'h' -5.45702 7.94728 -7.10836\n"
			"@group { rot16 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -4.91129 6.18867 -3.66269\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.61516 5.91526 -1.60915\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -3.31051 5.30085 -2.64925\n"
			"{PHE 2  CG  (aroC)}P -4.91129 6.18867 -3.66269\n"
			"{PHE 2  CD1 (aroC)} -6.13594 6.83302 -3.56932\n"
			"{PHE 2  CG  (aroC)}P -4.91129 6.18867 -3.66269\n"
			"{PHE 2  CD2 (aroC)} -4.17876 6.32202 -4.83242\n"
			"{PHE 2  CD1 (aroC)}P -6.13594 6.83302 -3.56932\n"
			"{PHE 2  CE1 (aroC)} -6.61633 7.59261 -4.61939\n"
			"{PHE 2  CD1 (aroC)}P 'h' -6.13594 6.83302 -3.56932\n"
			"{PHE 2  HD1 (Haro)} 'h' -6.72056 6.73544 -2.65416\n"
			"{PHE 2  CD2 (aroC)}P -4.17876 6.32202 -4.83242\n"
			"{PHE 2  CE2 (aroC)} -4.65574 7.08145 -5.88299\n"
			"{PHE 2  CD2 (aroC)}P 'h' -4.17876 6.32202 -4.83242\n"
			"{PHE 2  HD2 (Haro)} 'h' -3.21458 5.81942 -4.91705\n"
			"{PHE 2  CE1 (aroC)}P -6.61633 7.59261 -4.61939\n"
			"{PHE 2  CZ  (aroC)} -5.87646 7.71715 -5.77592\n"
			"{PHE 2  CE1 (aroC)}P 'h' -6.61633 7.59261 -4.61939\n"
			"{PHE 2  HE1 (Haro)} 'h' -7.58029 8.09335 -4.53263\n"
			"{PHE 2  CE2 (aroC)}P -4.65574 7.08145 -5.88299\n"
			"{PHE 2  CZ  (aroC)} -5.87646 7.71715 -5.77592\n"
			"{PHE 2  CE2 (aroC)}P 'h' -4.65574 7.08145 -5.88299\n"
			"{PHE 2  HE2 (Haro)} 'h' -4.06975 7.17865 -6.79673\n"
			"{PHE 2  CZ  (aroC)}P 'h' -5.87646 7.71715 -5.77592\n"
			"{PHE 2  HZ  (Haro)} 'h' -6.25466 8.31464 -6.60422\n"
			"@group { rot17 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -4.91129 6.18867 -3.66269\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.61516 5.91526 -1.60915\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -3.31051 5.30085 -2.64925\n"
			"{PHE 2  CG  (aroC)}P -4.91129 6.18867 -3.66269\n"
			"{PHE 2  CD1 (aroC)} -5.89358 7.14536 -3.45411\n"
			"{PHE 2  CG  (aroC)}P -4.91129 6.18867 -3.66269\n"
			"{PHE 2  CD2 (aroC)} -4.42115 6.00997 -4.9474\n"
			"{PHE 2  CD1 (aroC)}P -5.89358 7.14536 -3.45411\n"
			"{PHE 2  CE1 (aroC)} -6.37378 7.90529 -4.50403\n"
			"{PHE 2  CD1 (aroC)}P 'h' -5.89358 7.14536 -3.45411\n"
			"{PHE 2  HD1 (Haro)} 'h' -6.28629 7.29475 -2.44799\n"
			"{PHE 2  CD2 (aroC)}P -4.42115 6.00997 -4.9474\n"
			"{PHE 2  CE2 (aroC)} -4.89825 6.76876 -5.99838\n"
			"{PHE 2  CD2 (aroC)}P 'h' -4.42115 6.00997 -4.9474\n"
			"{PHE 2  HD2 (Haro)} 'h' -3.64887 5.26022 -5.12313\n"
			"{PHE 2  CE1 (aroC)}P -6.37378 7.90529 -4.50403\n"
			"{PHE 2  CZ  (aroC)} -5.87619 7.71738 -5.77587\n"
			"{PHE 2  CE1 (aroC)}P 'h' -6.37378 7.90529 -4.50403\n"
			"{PHE 2  HE1 (Haro)} 'h' -7.14587 8.65339 -4.32601\n"
			"{PHE 2  CE2 (aroC)}P -4.89825 6.76876 -5.99838\n"
			"{PHE 2  CZ  (aroC)} -5.87619 7.71738 -5.77587\n"
			"{PHE 2  CE2 (aroC)}P 'h' -4.89825 6.76876 -5.99838\n"
			"{PHE 2  HE2 (Haro)} 'h' -4.50405 6.61878 -7.00329\n"
			"{PHE 2  CZ  (aroC)}P 'h' -5.87619 7.71738 -5.77587\n"
			"{PHE 2  HZ  (Haro)} 'h' -6.25468 8.315 -6.60394\n"
			"@group { rot18 } animate dominant\n"
			"@vectorlist {PHE 1} color= bluetint master= {rotamers}\n"
			"{PHE 2  N   (Nbb)}P -6.414 4.03 -2.127\n"
			"{PHE 2  CA  (CAbb)} -4.993 3.959 -2.449\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  C   (CObb)} -4.238 3.117 -1.429\n"
			"{PHE 2  CA  (CAbb)}P -4.993 3.959 -2.449\n"
			"{PHE 2  CB  (CH2)} -4.3903 5.36323 -2.52079\n"
			"{PHE 2  C   (CObb)}P -4.238 3.117 -1.429\n"
			"{PHE 2  O   (OCbb)} -3.492 2.207 -1.791\n"
			"{PHE 2  CB  (CH2)}P -4.3903 5.36323 -2.52079\n"
			"{PHE 2  CG  (aroC)} -4.91129 6.18867 -3.66269\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 1HB  (Hapo)} 'h' -4.61516 5.91526 -1.60915\n"
			"{PHE 2  CB  (CH2)}P 'h' -4.3903 5.36323 -2.52079\n"
			"{PHE 2 2HB  (Hapo)} 'h' -3.31051 5.30085 -2.64925\n"
			"{PHE 2  CG  (aroC)}P -4.91129 6.18867 -3.66269\n"
			"{PHE 2  CD1 (aroC)} -5.5634 7.39003 -3.42789\n"
			"{PHE 2  CG  (aroC)}P -4.91129 6.18867 -3.66269\n"
			"{PHE 2  CD2 (aroC)} -4.75124 5.76561 -4.97344\n"
			"{PHE 2  CD1 (aroC)}P -5.5634 7.39003 -3.42789\n"
			"{PHE 2  CE1 (aroC)} -6.0433 8.15023 -4.47775\n"
			"{PHE 2  CD1 (aroC)}P 'h' -5.5634 7.39003 -3.42789\n"
			"{PHE 2  HD1 (Haro)} 'h' -5.69481 7.73279 -2.40121\n"
			"{PHE 2  CD2 (aroC)}P -4.75124 5.76561 -4.97344\n"
			"{PHE 2  CE2 (aroC)} -5.2287 6.52379 -6.02469\n"
			"{PHE 2  CD2 (aroC)}P 'h' -4.75124 5.76561 -4.97344\n"
			"{PHE 2  HD2 (Haro)} 'h' -4.24033 4.82231 -5.16983\n"
			"{PHE 2  CE1 (aroC)}P -6.0433 8.15023 -4.47775\n"
			"{PHE 2  CZ  (aroC)} -5.87586 7.71754 -5.77591\n"
			"{PHE 2  CE1 (aroC)}P 'h' -6.0433 8.15023 -4.47775\n"
			"{PHE 2  HE1 (Haro)} 'h' -6.55398 9.09212 -4.27891\n"
			"{PHE 2  CE2 (aroC)}P -5.2287 6.52379 -6.02469\n"
			"{PHE 2  CZ  (aroC)} -5.87586 7.71754 -5.77591\n"
			"{PHE 2  CE2 (aroC)}P 'h' -5.2287 6.52379 -6.02469\n"
			"{PHE 2  HE2 (Haro)} 'h' -5.09579 6.18017 -7.05036\n"
			"{PHE 2  CZ  (aroC)}P 'h' -5.87586 7.71754 -5.77591\n"
			"{PHE 2  HZ  (Haro)} 'h' -6.25455 8.31538 -6.60373\n";
		TS_ASSERT( sout.str() == correct_kinemage );
		if ( sout.str() != correct_kinemage ) {
			std::cout << sout.str() << std::endl;
		}

	}

};


