// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/toolbox/PoseMetricCalculators/Neighbor_Calculators.cxxtest.hh
/// @brief  test for NeighborsByDistanceCalculator and NeighborhoodByDistanceCalculator
/// @author Steven Lewis

// Test headers
#include <cxxtest/TestSuite.h>
#include <test/core/init_util.hh>
//#include <test/UTracer.hh>

// Unit header
#include <protocols/toolbox/PoseMetricCalculators/NeighborsByDistanceCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/NeighborhoodByDistanceCalculator.hh>

// project headers
#include <core/types.hh>

#include <core/io/pdb/pose_io.hh>
#include <core/pose/Pose.hh>

#include <core/pose/metrics/CalculatorFactory.hh>
#include <core/util/MetricValue.hh>

// option key includes

#include <core/options/keys/pose_metrics.OptionKeys.gen.hh>




// --------------- Test Class --------------- //

class Neighbor_CalculatorsTests : public CxxTest::TestSuite {

	core::pose::Pose pose; //core/conformation/dock_in.pdb

public:

	// --------------- Fixtures --------------- //
	Neighbor_CalculatorsTests() {
		core_init();

		core::io::pdb::pose_from_pdb( pose, "core/conformation/dock_in.pdb" );
		//213, 204, and 265 are appropriate residues from the dock_in pdb

		//set up the Neighbors calculators
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( "nbdc_buried", new protocols::toolbox::PoseMetricCalculators::NeighborsByDistanceCalculator(213) );

		core::pose::metrics::CalculatorFactory::Instance().register_calculator( "nbdc_surface", new protocols::toolbox::PoseMetricCalculators::NeighborsByDistanceCalculator(204) );

		core::pose::metrics::CalculatorFactory::Instance().register_calculator( "nbdc_interface", new protocols::toolbox::PoseMetricCalculators::NeighborsByDistanceCalculator(265) );

		//set up the Neighborhood calculators (testing each constructor type)
		std::set< core::Size > crset;
		crset.insert(213); crset.insert(204), crset.insert(265);
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( "nh_crset_calc", new protocols::toolbox::PoseMetricCalculators::NeighborhoodByDistanceCalculator(crset) );

		std::set< std::string > nameset;
		nameset.insert("nbdc_buried"); nameset.insert("nbdc_surface"); nameset.insert("nbdc_interface");
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( "nh_nameset_calc", new protocols::toolbox::PoseMetricCalculators::NeighborhoodByDistanceCalculator(nameset) );
	}

	virtual ~Neighbor_CalculatorsTests() {}

	static Neighbor_CalculatorsTests* createSuite() {
		return new Neighbor_CalculatorsTests();
	}

	static void destroySuite( Neighbor_CalculatorsTests *suite ) {
		delete suite;
	}

	void setUp() {}

	void tearDown() {}

	// ------------- Helper Functions ------------- //
	//runs through the list of legal metrics from a Neighbor calculator
	void check_nbrs(
									std::string const & calc,
									core::Size resid,
									std::string const & neighbors,
									core::Size num_neighbors,
									core::Real cutoff = core::options::option[core::options::OptionKeys::pose_metrics::neighbor_by_distance_cutoff]){

		core::util::MetricValue< core::Size > central_residue_, num_neighbors_;
		core::util::MetricValue< core::Real > cutoff_;

		pose.metric( calc, "central_residue", central_residue_);
		pose.metric( calc, "dist_cutoff",     cutoff_);
		pose.metric( calc, "num_neighbors",   num_neighbors_);

		TS_ASSERT( resid == central_residue_.value() );
		TS_ASSERT( cutoff == cutoff_.value() );
		TS_ASSERT( num_neighbors == num_neighbors_.value() );
		TS_ASSERT( neighbors == pose.print_metric(calc, "neighbors") );
		//std::cout << "X" << pose.print_metric(calc, "neighbors") << "X" << std::endl;
		return;
	}

	//runs through the list of legal metrics from a Neighborhood calculator
	void check_hood(
									std::string const & calc,
									std::string const & calcs,
									std::string const & resids,
									std::string const & neighbors,
									core::Size num_neighbors) {

		core::util::MetricValue< core::Size > num_neighbors_;

		pose.metric( calc, "num_neighbors", num_neighbors_);

		TS_ASSERT( calcs == pose.print_metric(calc, "calc_names") );
		TS_ASSERT( resids == pose.print_metric(calc, "central_residues") );
		TS_ASSERT( num_neighbors == num_neighbors_.value() );
		TS_ASSERT( neighbors == pose.print_metric(calc, "neighbors") );
		return;
	}

	// --------------- Test Cases --------------- //

	///@brief
	void test_NeighborsByDistanceCalculators() {
		//the strings and magic numbers are the return values from calling the calculator
		std::string const bur_nbr("42 43 44 54 55 58 102 138 139 140 142 183 190 191 193 194 195 196 197 198 199 211 212 213 214 215 216 226 227 228 229 262 263 264 ");
		std::string const sur_nbr("122 202 203 204 205 206 208 ");
		std::string const int_nbr("33 39 40 41 42 43 58 141 142 143 149 150 151 191 192 193 194 195 263 264 265 266 267 277 278 299 ");

		check_nbrs("nbdc_buried", 213, bur_nbr, 34);
		check_nbrs("nbdc_surface", 204, sur_nbr, 7);
		check_nbrs("nbdc_interface", 265, int_nbr, 26);
		return;
	}

	void test_NeighborHoodDistanceCalculators() {
		//the strings and magic numbers are the return values from calling the calculator
		std::string const resids("204 213 265 ");
		std::string const calcs_nameset("nbdc_buried nbdc_interface nbdc_surface ");
		std::string const calcs_crset("NeighborhoodByDistanceCalculator-associated_NeighborsByDistanceCalculator_centered_on_204 NeighborhoodByDistanceCalculator-associated_NeighborsByDistanceCalculator_centered_on_213 NeighborhoodByDistanceCalculator-associated_NeighborsByDistanceCalculator_centered_on_265 ");
		core::Size const n_nbrs(57);
		std::string const neighborhood("33 39 40 41 42 43 44 54 55 58 102 122 138 139 140 141 142 143 149 150 151 183 190 191 192 193 194 195 196 197 198 199 202 203 204 205 206 208 211 212 213 214 215 216 226 227 228 229 262 263 264 265 266 267 277 278 299 ");

		check_hood("nh_nameset_calc", calcs_nameset, resids, neighborhood, n_nbrs);
		check_hood("nh_crset_calc", calcs_crset, resids, neighborhood, n_nbrs);

	}//end test_NeighborsByDistanceCalculator

};//end class
