// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/toolbox/TaskOperations/RestrictToInterfaceOperation.cxxtest.hh
/// @brief  test for RestrictToInterfaceOperation
/// @author Steven Lewis

// Test headers
#include <cxxtest/TestSuite.h>
#include <test/core/init_util.hh>
#include <test/UTracer.hh>

// Unit header
#include <protocols/toolbox/TaskOperations/RestrictToInterfaceOperation.hh>
#include <protocols/toolbox/TaskOperations/RestrictToNeighborhoodOperation.hh>
#include <protocols/toolbox/TaskOperations/RestrictByCalculatorsOperation.hh>

// project headers
#include <core/types.hh>

#include <core/io/pdb/pose_io.hh>
#include <core/pose/Pose.hh>

#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/PackerTask.hh>

#include <protocols/toolbox/PoseMetricCalculators/NeighborhoodByDistanceCalculator.hh>
#include <protocols/toolbox/PoseMetricCalculators/InterfaceNeighborDefinitionCalculator.hh>

// C++ headers
//#include <set>

// --------------- Test Class --------------- //

class RestrictOperationsTests : public CxxTest::TestSuite {

	core::pose::Pose pose;

public:

	// --------------- Fixtures --------------- //

	//ctor sets up the pose once
	RestrictOperationsTests(){
		core_init();
		//reuse for comparison with Interface class
		core::io::pdb::centroid_pose_from_pdb( pose, "core/conformation/dock_in.pdb" );
	}

	virtual ~RestrictOperationsTests() {}

	static RestrictOperationsTests* createSuite() {
		return new RestrictOperationsTests();
	}

	static void destroySuite( RestrictOperationsTests *suite ) {
		delete suite;
	}

	void setUp() {}

	void tearDown() {}

	// ------------- Helper Functions ------------- //


	// --------------- Test Cases --------------- //

	///@brief
	void test_RestrictToInterfaceOperation() {

		/* if you are debugging this test, I suggest passing the flag -print_pymol_selection for ease of use.
			 The flag -pose_metrics::interface_cutoff 8 will cause the results to match core/conformation/Interface.cxxtest
			 although the flag actually defaults to 10 (thus a larger interface)
		*/

		//set up test
		using namespace core::pack::task;
		using protocols::toolbox::TaskOperations::RestrictToInterfaceOperation;
		TaskFactory RTIO_factory;
		RTIO_factory.push_back( new RestrictToInterfaceOperation() ); //defaults to interface between chains 1 and 2

		//run
		test::UTracer UT_RTIO("protocols/toolbox/TaskOperations/RestrictToInterfaceOperation.u");
		//this call returns PackerTaskOP; we are dumping the ptask to utracer
		UT_RTIO << *(RTIO_factory.create_task_and_apply_taskoperations( pose )) << std::endl;

	}

	void test_RestrictToNeighborhoodOperation() {

		std::set< core::Size > crset;
		crset.insert(77); crset.insert(215), crset.insert(45); //surface, interface, buried residues to test

		using namespace core::pack::task;
		using protocols::toolbox::TaskOperations::RestrictToNeighborhoodOperation;
		TaskFactory RTNO_factory;
		RTNO_factory.push_back( new RestrictToNeighborhoodOperation( crset ) );

		test::UTracer UT_RTNO("protocols/toolbox/TaskOperations/RestrictToNeighborhoodOperation.u");
		UT_RTNO << *(RTNO_factory.create_task_and_apply_taskoperations( pose )) << std::endl;
	}

	void test_RestrictByCalculatorsOperation() {

		//first we set up the calculators that the Operation will use
		std::string const interface_calc("interface"), neighborhood_calc("neighborhood");
		std::set< core::Size > crset_RBC;
		crset_RBC.insert(127); crset_RBC.insert(170), crset_RBC.insert(46);
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( interface_calc, new protocols::toolbox::PoseMetricCalculators::InterfaceNeighborDefinitionCalculator( core::Size(1), core::Size(2) ) );
		core::pose::metrics::CalculatorFactory::Instance().register_calculator( neighborhood_calc, new protocols::toolbox::PoseMetricCalculators::NeighborhoodByDistanceCalculator( crset_RBC ) );

		//this is the constructor parameter for the calculator - pairs of calculators and calculations to perform
		utility::vector1< std::pair< std::string, std::string> > calcs_and_calcns;
		calcs_and_calcns.push_back(std::make_pair(interface_calc, "interface_residues"));
		calcs_and_calcns.push_back(std::make_pair(neighborhood_calc, "neighbors"));

		using protocols::toolbox::TaskOperations::RestrictByCalculatorsOperation;
		core::pack::task::TaskFactory RBC_factory;
		RBC_factory.push_back( new RestrictByCalculatorsOperation( calcs_and_calcns ) );

		test::UTracer UT_RBC("protocols/toolbox/TaskOperations/RestrictByCalculatorsOperation.u");
		UT_RBC << *(RBC_factory.create_task_and_apply_taskoperations( pose )) << std::endl;

	}//end test_RestrictByCalculatorsOperation

};//end class
