#!/usr/bin/env python
# :noTabs=true:
# (c) Copyright Rosetta Commons Member Institutions.
# (c) This file is part of the Rosetta software suite and is made available under license.
# (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
# (c) For more information, see http://www.rosettacommons.org. Questions about this can be
# (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

import sys
if not hasattr(sys, "version_info") or sys.version_info < (2,4):
    raise ValueError("Script requires Python 2.4 or higher!")

import os, commands, re, subprocess


UnitTestExecutable = ["ObjexxFCL.test", "core.test", "numeric.test", "protocols.test", "utility.test", "apps.test"]
#UnitTestExecutable = ["numeric.test", "utility.test"]

# Output info class, simple wrapper for named collection of fields.
class OI:
    def __init__(self, **entries): self.__dict__.update(entries)


class Tester:
    def __init__(self, db_path=None):
        self.db_path = db_path
        self.systemLog = ""  # System log - we store all information here
        self.results = {}


    # print and log given mesage, return message
    def log(self, s):
        self.systemLog += s
        print s
        return s


    # Try to identity plaform by using scons compiliation feature.
    def getPlatformID(self):
        self.log( "Identifying platform...\n")
        pl = commands.getoutput("""scons Abracadabra log=platform""")
        lines = pl.split('\n')
        for s in lines:
            if  len( s.split() ) > 1 and s.split()[0] == 'Platform:':
                platform = s.split()[1]
                self.log( "Platform found: " + platform )
                return platform
        return "PlatformWasNotFound!!!"  # <-- That should not reall happend.

    # extract information regarding how good unit tests run.
    def extractInfo(self, output):
        # default init, in case we can't extract information
        oi = OI(testCount=0, testFailed=0, failedTestsList=[])

        # extracting number of test
        s = re.search(r'Running (\d\d*) test', output)
        if s:
            g = s.groups()
            oi.testCount = int( g[0] )

        # extracting number of test failed
        s = re.search(r'Failed (\d\d*) of \d\d* test', output)
        if s:
            g = s.groups()
            oi.testFailed = int( g[0] )

            # extracting names of the failed tests
            s = re.findall(r'CXXTEST_ERROR: (\w*) Failed!', output)
            for t in s: oi.failedTestsList.append(t)

            #s = re.search(r'CXXTEST_ERROR: (\w*) Failed!', output)
            #if s:
            #    g = s.groups()
            #    for t in g: oi.failedTestsList.append(t)

        else: # all test pussed?
            s = re.search(r'All tests passed!', output)
            if not s: # Something wrong then, count all tests as failed.
                oi.testFailed = oi.testCount
                oi.failedTestsList = ['All']

        #print oi.__dict__
        return oi


    # Run unit test.
    def runUnitTests(self):
        platform = self.getPlatformID()
        #self.log( "Run unit tests...\n")
        self.unitTestLog = "================================ UnitTest Results ================================\n"
        all_results_yaml_file = '.unit_test_results.yaml'
        uf = file(all_results_yaml_file, 'w')
        for E in UnitTestExecutable:
            self.unitTestLog += self.log("-------- %s --------\n" % E)
            path = "cd build/test/" + platform + " && "
            exe = "./" + E
            if self.db_path: exe += " " + self.db_path
            print "Paths:", path

            #output = commands.getoutput(path + exe) + "\n\n"
            #print output

            yaml_file = 'build/test/'+ platform + '/' + E + '.yaml'
            if os.path.isfile(yaml_file):
                os.remove(yaml_file)

            output = ""

            timelimit = sys.executable + ' ' +os.path.abspath('test/timelimit.py') + ' 30 '
            #print "timelimit:", timelimit

            command_line = path + ' ' + timelimit + exe + " 1>&2"
            #print 'Executing:', command_line
            f = subprocess.Popen(command_line, bufsize=0, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE).stderr
            for line in f:
                print line,
                output += line
                sys.stdout.flush()
            f.close()

            info = self.extractInfo(output)
            info.name = E
            self.results[E] = info

            if not os.path.isfile(yaml_file):
                print "Unable to read yaml file with test results - unit test run aborted!"
                uf.close()
                os.remove(all_results_yaml_file)
                sys.exit(1)

            # generating summary yaml file for all tests
            uf.write(E + ':\n')
            f = file(yaml_file)
            for line in f: uf.write("    "+line)
            f.close()
            uf.write('\n')

        uf.close()
        #self.log( "Run unit tests... Done.\n")


    def printSummary(self):
        total = 0
        failed = 0
        failedTestsList = []

        for t in self.results:
            total += self.results[t].testCount
            failed += self.results[t].testFailed
            failedTestsList.extend( [ self.results[t].name + ": " + r  for r in self.results[t].failedTestsList] )
            #print "--> ", self.results[t].failedTestsList

        print "-------- Unit test summary --------"
        print "Total number of tests:", total
        print "  number tests passed:", total-failed
        print "  number tests failed:", failed
        if failedTestsList:
            print "  failed tests:"
            for t in failedTestsList:
                print "   ", t
        print "Success rate: %s%%" % ((total-failed)*100/total)
        print "---------- End of Unit test summary"


def main(args):
    db_path = None
    if len(args) <= 1: print "Warning: no database path was specified!"
    else: db_path = " ".join(args[1:])


    T = Tester(db_path)
    T.runUnitTests()
    T.printSummary()

    print "Done!"



if __name__ == "__main__": main(sys.argv)
