#!/usr/bin/env python
# :noTabs=true:
# (c) Copyright Rosetta Commons Member Institutions.
# (c) This file is part of the Rosetta software suite and is made available under license.
# (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
# (c) For more information, see http://www.rosettacommons.org. Questions about this can be
# (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

import sys, commands

from os import path
from optparse import OptionParser, IndentedHelpFormatter


def main(argv):
    '''
A script to help debug/run cluster scientic tests

Each test has its own subdirectory in tests/, which serves as its name.
Each test has a files named "submit" and "analyze", which should have the command to be executed.
Variable substitution is possible using Python printf format; see examples.
To create a new test, just make a new directory with a "command" file and other needed files.
See tests/HOW_TO_MAKE_TESTS for more information.

Intended usage is to run the tests once to submit jobs to cluster: './cluster submit docking'
and when condor run is finished to run again to analyze results: './cluster analyze docking'

Please note that this script is for debuging/testing purposes only, it is not used in ScientificCluster daemon.
    '''
    parser = OptionParser(usage="usage: %prog [OPTIONS] [TESTS]")
    parser.set_description(main.__doc__)
    parser.add_option("-d", "--database",
      default=path.join( path.expanduser("~"), "rosetta_database"),
      help="Directory where Rosetta database is found (default: ~/rosetta_database)",
    )

    parser.add_option("-r", "--rosetta_home",
      default = path.dirname( path.dirname( path.dirname( path.dirname(path.abspath(sys.argv[0])) ) ) ),
      help="Directory where Rosetta is found (default: ../)",
    )

    parser.add_option("--mode",
      default="release",
      help="In selecting binaries, which mode was used? (default: release)",
    )

    parser.add_option("-c", "--compiler",
      default="gcc",
      help="In selecting binaries, which compiler was used? (default: gcc)",
    )

    (options, args) = parser.parse_args(args=argv)

    if len(args) != 2:
        print 'You must supplies action and test name in command line! For example: "./cluster.py submit docking" or ""./cluster.py analyze docking""'
        return

    action, test = args
    if action != 'submit' and action != 'analyze' :
        print "Action must be eather 'submit' or 'analyze'..."
        return

    print 'Perform %s on test %s...' % (action, test)


    workdir = path.abspath( test )
    rosettadir = options.rosetta_home

    #print 'rosettadir:', rosettadir
    #return

    database = options.database
    print 'Database:', database
    bin = path.join(rosettadir, "bin")
    pyapps = path.join(rosettadir, "src", "python", "apps")
    if sys.platform.startswith("linux"): platform = "linux" # can be linux1, linux2, etc
    elif sys.platform == "darwin": platform = "macos"
    else: platform = "_unknown_"
    compiler = options.compiler
    mode = options.mode
    binext = platform+compiler+mode
    # Read the command from the file "command"

    cmd = file(path.join(workdir, action)).read().strip()
    # cmd = cmd % vars() # variable substitution using Python printf style
    mvars = dict(rosettadir=rosettadir, database=database, workdir=workdir, platform=platform, bin=bin, compiler=compiler, mode=mode, binext=binext)
    cmd = cmd % mvars

    # Writing result to .sh file for future refference.
    f = file(path.join(workdir, action)+'.sh', 'w');  f.write(cmd);  f.close()

    # Creating python style file with arguments in case test want to use Python as script language
    f = file(path.join(workdir, '_arguments.py'), 'w');  f.write( str(mvars) );  f.close()

    # Executing created bash script...
    print commands.getoutput('cd %s && sh %s.sh' % (test, action) )



if __name__ == "__main__":
    sys.exit(main(sys.argv[1:]))
