/// @file
/// @brief


#include <protocols/viewer/viewers.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/Mover.fwd.hh>
#include <protocols/moves/MoverContainer.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/RigidBodyMover.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/symmetry/SetupForSymmetryMover.hh>
#include <protocols/moves/SwitchResidueTypeSetMover.hh>

#include <protocols/relax/util.hh>
#include <utility/excn/Exceptions.hh>

#include <protocols/init.hh>
#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/ThreadingJob.hh>
#include <protocols/viewer/viewers.hh>
#include <protocols/relax/FastRelax.hh>
#include <protocols/electron_density/util.hh>
#include <protocols/loops/loops_main.hh>
#include <protocols/loops/Loops.hh>
#include <protocols/loops/LoopMover.fwd.hh>
#include <protocols/loops/LoopMover.hh>
#include <protocols/loops/LoopMover_QuickCCD.hh>
#include <protocols/loops/LoopMover_CCD.hh>
#include <protocols/loops/LoopMover_KIC.hh>
#include <protocols/loops/LoopMoverFactory.hh>
#include <protocols/loops/LoopRelaxMover.hh>

#include <protocols/relax/RelaxProtocolBase.hh>
#include <protocols/relax/util.hh>

#include <protocols/moves/BBGaussianMover.hh>
#include <protocols/moves/CanonicalSamplingMover.hh>
#include <protocols/moves/CanonicalSamplingMover.fwd.hh>
#include <protocols/moves/SidechainMover.hh>
#include <protocols/moves/SidechainMCMover.hh>
#include <protocols/RBSegmentMoves/AutoRBRelaxMover.hh>

#include <protocols/comparative_modeling/util.hh>
#include <protocols/comparative_modeling/ThreadingMover.hh>

#include <core/kinematics/MoveMap.hh>
#include <core/pose/Pose.hh>
#include <core/conformation/Conformation.hh>
#include <core/pose/util.hh>
#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/symmetry/SymAtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/electron_density/ElectronDensity.hh>
#include <core/scoring/electron_density/util.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/BoundConstraint.hh>
#include <core/scoring/constraints/util.hh>

#include <core/fragment/Frame.hh>
#include <core/fragment/FrameIterator.hh>
#include <core/fragment/FrameIteratorWorker_.hh>
#include <core/fragment/FragSet.hh>
#include <core/fragment/FragmentIO.hh>

#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/pack_rotamers.hh>
#include <core/pack/task/operation/TaskOperations.hh>


#include <core/sequence/Sequence.hh>
#include <core/sequence/SWAligner.hh>
#include <core/sequence/ScoringScheme.fwd.hh>
#include <core/sequence/SimpleScoringScheme.hh>
#include <core/sequence/SequenceMapping.hh>
#include <core/sequence/SequenceAlignment.hh>
#include <core/sequence/util.hh>

#include <core/conformation/symmetry/util.hh>

#include <protocols/moves/symmetry/SymPackRotamersMover.hh>
#include <protocols/moves/MinMover.hh>

#include <core/conformation/Residue.hh>
#include <core/conformation/Residue.functions.hh>

// #include <protocols/Gaussian/BBGaussianMover.hh>

//options
#include <core/options/option.hh>
#include <core/options/option_macros.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/cm.OptionKeys.gen.hh>
#include <core/options/keys/loops.OptionKeys.gen.hh>
#include <core/options/keys/edensity.OptionKeys.gen.hh>
#include <core/options/keys/symmetry.OptionKeys.gen.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/packing.OptionKeys.gen.hh>
#include <core/util/Tracer.hh>

//
#include <iostream>
#include <string>
#include <fstream>
#include <sstream>

OPT_1GRP_KEY(Integer, MR, max_gaplength_to_model)
OPT_1GRP_KEY(Integer, MR, nrebuildcycles)
OPT_1GRP_KEY(Boolean, MR, debug)
OPT_1GRP_KEY(String, MR, mode)

static core::util::Tracer TR("rosetta_MR");

///////////////////////////////////////////////////////////////////////////////

class MRMover : public protocols::moves::Mover {
private:
	core::scoring::ScoreFunctionOP pattonly_scorefxn_;
	core::scoring::ScoreFunctionOP fapatt_scorefxn_;
	core::scoring::ScoreFunctionOP cen_scorefxn_;
	core::scoring::ScoreFunctionOP cenpatt_scorefxn_;
	core::scoring::ScoreFunctionOP fa_scorefxn_;

	utility::vector1< core::fragment::FragSetOP > frag_libs_;
	core::Size nrebuildcycles;

public:
	MRMover(){
		using namespace core::options;
		using namespace core::options::OptionKeys;

		pattonly_scorefxn_ = new core::scoring::ScoreFunction();
		fapatt_scorefxn_ = core::scoring::getScoreFunction();
		cen_scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function("cen_std","score4L");
		cenpatt_scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function("cen_std","score4L");
		fa_scorefxn_ = core::scoring::getScoreFunction();

		// if not specified should set patterson wt to some value
		core::scoring::electron_density::add_dens_scores_from_cmdline_to_scorefxn( *fapatt_scorefxn_ );
		core::scoring::electron_density::add_dens_scores_from_cmdline_to_scorefxn( *pattonly_scorefxn_ );
		core::scoring::electron_density::add_dens_scores_from_cmdline_to_scorefxn( *cenpatt_scorefxn_ );

		// set cst weight (fa only)
		core::scoring::constraints::add_constraints_from_cmdline_to_scorefxn( *cen_scorefxn_  );
		core::scoring::constraints::add_fa_constraints_from_cmdline_to_scorefxn( *fapatt_scorefxn_  );

		//nrebuildcycles =  option[ OptionKeys::MR::nrebuildcycles ]();
		if ( option[ OptionKeys::loops::frag_files ].user() )
			protocols::loops::read_loop_fragments( frag_libs_ );
	}

	virtual std::string get_name() const {return "MRMover";}

	void pack_missing_sidechains( Pose & pose ) {
		utility::vector1< bool > needToRepack( pose.total_residue() , false );
		bool needToRepackAny = false;
		for ( core::Size i = 1; i <= pose.total_residue(); ++i ) {
			// if there is missing density in the sidechain, then we need to repack
			// check:
			//    (a) CA-CB distance > 3A
			//    (b) CB-any distance > 10A
			if ( pose.residue_type(i).is_protein() && pose.residue_type(i).has("CA") ) {
				numeric::xyzVector< core::Real> ca_pos = pose.residue(i).atom("CA").xyz();
				numeric::xyzVector< core::Real> cb_pos = ca_pos;

				if ( !pose.residue_type(i).has("CB") ) continue;

				cb_pos = pose.residue(i).atom("CB").xyz();
				if ((ca_pos - cb_pos).length() > 3) {
					needToRepack[i] = true;
					needToRepackAny = true;
				} else {
					for (int j=(int)pose.residue(i).first_sidechain_atom()+1; j<=(int)pose.residue(i).natoms(); ++j) {
						if ( (cb_pos - pose.residue(i).atom(j).xyz()).length() > 10 ) {
							needToRepack[i] = true;
							needToRepackAny = true;
							break;
						}
					}
				}
			}
		}

		core::pack::task::PackerTaskOP taskstd = core::pack::task::TaskFactory::create_packer_task( pose );
		taskstd->restrict_to_repacking();
		taskstd->or_include_current(false);
		core::conformation::symmetry::make_residue_mask_symmetric( pose, needToRepack );
		taskstd->restrict_to_residues(needToRepack);

		if ( core::conformation::symmetry::is_symmetric( pose ) ) {
			protocols::moves::symmetry::SymPackRotamersMover pack1( fa_scorefxn_, taskstd );
			pack1.apply( pose );
		} else {
			protocols::moves::PackRotamersMover pack1( fa_scorefxn_, taskstd );
			pack1.apply( pose );
		}

	}


	void trim_target_pose( Pose & query_pose, protocols::loops::Loops &loops , core::Size max_gaplength ) {
		using namespace protocols::comparative_modeling;
		using namespace core::sequence;
		using namespace core::options;
		using namespace core::options::OptionKeys;

		utility::vector1< bool > to_trim(query_pose.total_residue(), false) ;
		protocols::loops::Loops new_loops;

		//
		for (int i=1; i<=loops.size(); ++i) {
			if (loops[i].size() > max_gaplength ) {
				for (int j=loops[i].start(); j<= loops[i].stop(); ++j) {
					to_trim[j] = true;
				}
			} else {
				new_loops.push_back( loops[i] );
			}
		}

		// # residues in new pose
		Size new_nres = 0;
		for ( Size i = 1; i <= query_pose.total_residue(); ++i ) {
			if (!to_trim[i]) new_nres++;
		}

		bool need_to_remap = true;
		if (new_nres == query_pose.total_residue()) return;  // nothing to do
		if (new_nres < 2) {
			std::cerr << "Error: not enough aligned residues! trying to continue" << std::endl;
			return;
		}

		// build new pose, generate seq mapping
		core::Size old_root = query_pose.fold_tree().root();
		core::pose::Pose new_query_pose;
		Size out_ctr=1;
		bool add_by_jump = true;
		SequenceMapping new_mapping(new_nres, query_pose.total_residue());
		SequenceMapping new_invmapping(query_pose.total_residue() , new_nres);
		for ( Size i = 1; i <= query_pose.total_residue(); ++i ) {
			if (!to_trim[i]) {
				if (add_by_jump) {
					new_query_pose.append_residue_by_jump( query_pose.residue(i), new_query_pose.total_residue(), "", "", true );
					add_by_jump = !query_pose.residue(i).is_polymer();
				} else if ( !query_pose.residue(i).is_polymer() ) {
					new_query_pose.append_residue_by_jump( query_pose.residue(i), new_query_pose.total_residue(), "", "", true );
					add_by_jump = true;
				} else {
					new_query_pose.append_residue_by_bond( query_pose.residue(i), false );
				}
				//for ( Size j = 1; j <= new_query_pose.residue(out_ctr).natoms(); ++j ) {
					//new_query_pose.set_xyz( core::id::AtomID(j,out_ctr), query_pose.xyz( core::id::AtomID(j,i) ) );
				//}
				new_mapping[out_ctr] = i;
				new_invmapping[i] = out_ctr;
				out_ctr++;
			} else {
				add_by_jump = true;
			}
		}
		core::kinematics::FoldTree f = new_query_pose.fold_tree();
		if (new_invmapping[old_root] != 0)
			f.reorder( new_invmapping[old_root] );

		// if pose is rooted on VRT update jump point
		if ( new_query_pose.residue( f.root() ).aa() == core::chemical::aa_vrt ) {
			// find residue closest to center-of-mass
			numeric::xyzVector< core::Real > massSum(0.0,0.0,0.0);
			int nAtms=0, nres=new_query_pose.total_residue();
			for ( int i=1; i<= nres; ++i ) {
				core::conformation::Residue const & rsd( new_query_pose.residue(i) );
				if (rsd.aa() == core::chemical::aa_vrt) continue;
				for ( Size j=1; j<= rsd.nheavyatoms(); ++j ) {
					core::conformation::Atom const & atom( rsd.atom(j) );
					massSum += atom.xyz();
					nAtms++;
				}
			}
			massSum /= nAtms;

			// try to avoid putting the vrt too close to termini
			int i_min = 1;
			int r_start = (int)std::floor(   nres/3 );
			int r_end   = (int)std::ceil ( 2*nres/3 );
			core::Real d_min = 99999, this_d;
			for ( int i=r_start; i<=r_end; ++i ) {
				core::conformation::Residue const & rsd( new_query_pose.residue(i) );
				if (!rsd.is_protein() ) continue;

				core::conformation::Atom const & atom( rsd.atom("CA") );
				this_d = (atom.xyz() - massSum).length();
				if (this_d < d_min) {
					d_min = this_d;
					i_min = i;
				}
			}

			f.renumber_jumps();
			f.slide_jump( 1, new_invmapping[old_root], i_min );
			f.reorder( new_invmapping[old_root] );
		}
		new_query_pose.fold_tree( f );

		// remap loops
		for (int i=1; i<=new_loops.size(); ++i) {
			new_loops[i].set_start( new_invmapping[new_loops[i].start()] );
			new_loops[i].set_stop( new_invmapping[new_loops[i].stop()] );
		}
		loops = new_loops;

		// remap fragments
		utility::vector1< core::fragment::FragSetOP > new_frag_libs;
		for (int i=1; i<=frag_libs_.size(); ++i) {
			// empty_clone
			core::fragment::FragSetOP new_frag_set( frag_libs_[i]->empty_clone() );

			// iterate over frames, clone if mapped
			for ( core::fragment::FrameIterator f=frag_libs_[i]->begin(); f != frag_libs_[i]->end(); ++f ) {
				core::Size start_res = f->start();
				if ( new_mapping[ start_res ] != 0 ) {
					core::fragment::FrameOP new_f = f->clone_with_frags();
					new_f->align(new_mapping);
					new_frag_set->add( new_f );
				}
			}
			new_frag_libs.push_back( new_frag_set );
		}
		frag_libs_ = new_frag_libs;

		query_pose = new_query_pose;
	}

	void apply_bbg( Pose &pose ) {
		using namespace core::options;
		using namespace core::options::OptionKeys;
		using namespace protocols::moves;
		using namespace core::pack::task;

		protocols::moves::CanonicalSamplingMoverOP csm(new CanonicalSamplingMover);
		csm->set_scorefunction(fapatt_scorefxn_);

		//setup sidechain mover
		SidechainMCMoverOP scm(new SidechainMCMover());
		TaskFactoryOP main_task_factory = new TaskFactory;
		operation::RestrictToRepackingOP rtrop = new operation::RestrictToRepacking;
		main_task_factory->push_back( rtrop );
		scm->set_temperature( csm->get_temp() );
		scm->set_scorefunction( *fapatt_scorefxn_ );
		scm->setup( fapatt_scorefxn_ );
		scm->set_task_factory(main_task_factory);
		scm->set_prob_uniform(0.75);
		scm->set_prob_withinrot(0.0);
		scm->set_prob_random_pert_current(0.5);
		scm->set_preserve_detailed_balance( csm->detailed_balance() );
		scm->set_ntrials( 1000 );

		csm->add_mover(scm,0.5);

		//add move-map to bbg-mover
		core::kinematics::MoveMapOP movemap = new core::kinematics::MoveMap;
		movemap->set_bb( true );
		movemap->set_chi( true );
		movemap->set_jump( true );

		BBG8T3AMoverOP bbg8t3mover = new BBG8T3AMover();
		bbg8t3mover->movemap(movemap);
		csm->add_mover(bbg8t3mover,0.5);

		csm->apply( pose );
	}

	void apply_autorb( Pose &pose ) {
		using namespace core::options;
		using namespace core::options::OptionKeys;
		using namespace protocols::moves;
		using namespace core::pack::task;

		MoverOP autorb( new protocols::RBSegment::AutoRBMover );
		autorb->apply( pose );
	}

	void apply_relax( Pose &pose ) {
		using namespace protocols::loops;
		using namespace protocols::jd2;
		using namespace core::options;
		using namespace core::options::OptionKeys;

		bool debug = option[ OptionKeys::MR::debug ]();

		if ( option[ OptionKeys::symmetry::symmetry_definition ].user() )  {
			protocols::moves::symmetry::SetupForSymmetryMover pre_mover;
			pre_mover.apply( pose );
		} else {
			core::pose::addVirtualResAsRoot( pose );
		}

		if ( option[ OptionKeys::edensity::mapfile ].user() ) {
			protocols::moves::MoverOP dens_dock( new protocols::electron_density::SetupForDensityScoringMover );
			dens_dock->apply( pose );
		}

		protocols::relax::RelaxProtocolBaseOP relax_prot = protocols::relax::generate_relax_from_cmd();
		relax_prot->set_current_tag( get_current_tag() );
		relax_prot->apply( pose );
	}

	void apply_repackminimize( Pose &pose ) {
		using namespace protocols::loops;
		using namespace protocols::jd2;
		using namespace core::options;
		using namespace core::options::OptionKeys;

		core::pack::task::TaskFactoryOP main_task_factory = new core::pack::task::TaskFactory;
		main_task_factory->push_back( new core::pack::task::operation::InitializeFromCommandline );
		main_task_factory->push_back( new core::pack::task::operation::RestrictToRepacking );
		if ( option[ packing::resfile ].user() ) {
			main_task_factory->push_back( new core::pack::task::operation::ReadResfile );
		}
		protocols::moves::PackRotamersMoverOP pack_mover = new protocols::moves::PackRotamersMover;
		pack_mover->task_factory( main_task_factory );
		pack_mover->score_function( fapatt_scorefxn_ );

		protocols::moves::SequenceMoverOP seq_mover = new protocols::moves::SequenceMover;
		seq_mover->add_mover( pack_mover );
	
		core::kinematics::MoveMapOP mm = new core::kinematics::MoveMap;
		if ( option[ relax::bb_move ].user() ) {
			mm->set_bb  ( option[ relax::bb_move ]() );
		} else {
			mm->set_bb  ( true );
		}
		if ( option[ relax::chi_move ].user() ) {
			mm->set_chi( option[ relax::chi_move ]() );
		} else {
			mm->set_chi ( true );
		}
		if ( option[ relax::jump_move ].user() ) {
			mm->set_jump( option[ relax::jump_move ]() );
		} else {
			mm->set_jump( true );
		}

		protocols::moves::MinMoverOP min_mover = new protocols::moves::MinMover(
			mm, fapatt_scorefxn_, "dfpmin_armijo_nonmonotone", 0.01, true );
		seq_mover->add_mover( min_mover );

		// do it
		seq_mover->apply( pose );
	}

	void apply_loopmodel( Pose &pose ) {
		using namespace protocols::loops;
		using namespace protocols::jd2;
		using namespace core::options;
		using namespace core::options::OptionKeys;

		bool debug = option[ OptionKeys::MR::debug ]();

		// set up initial loop build
		core::Size nres = pose.total_residue();
		while (!pose.residue(nres).is_polymer()) nres--;

		// pack all missing SCs
		pack_missing_sidechains( pose );

		// setup for symmetry
		if ( option[ OptionKeys::symmetry::symmetry_definition ].user() )  {
			protocols::moves::symmetry::SetupForSymmetryMover pre_mover;
			pre_mover.apply( pose );
		} else {
			core::pose::addVirtualResAsRoot( pose );
		}
		if ( option[ OptionKeys::edensity::mapfile ].user() ) {
			protocols::moves::MoverOP dens_dock( new protocols::electron_density::SetupForDensityScoringMover );
			dens_dock->apply( pose );
		}

		if (debug) pose.dump_pdb( "pre_loopmodel.pdb" );

		// if a loopfile is given, use it
		// otherwise, use model
		protocols::loops::Loops to_rebuild;
		if ( option[ OptionKeys::loops::loop_file ].user() ) {
			to_rebuild.read_loop_file( option[ OptionKeys::loops::loop_file ]()[1] );
		} else {
			// autoselect
			if (fapatt_scorefxn_->get_weight(core::scoring::patterson_cc) > 0) {
				to_rebuild = protocols::electron_density::findLoopFromPatterson( pose, 10, std::ceil(pose.total_residue()/40), false );
			} else {
				to_rebuild = protocols::electron_density::findLoopFromDensity( pose, 0.3, -1, 1 );
			}
		}
		to_rebuild.choose_cutpoints( pose );

		// now do loopbuilding
		LoopRelaxMoverOP lr_mover( new LoopRelaxMover );
		lr_mover->loops( to_rebuild );
		lr_mover->scorefxns( cenpatt_scorefxn_, fapatt_scorefxn_ );
		lr_mover->frag_libs( frag_libs_ );
		lr_mover->relax( option[ OptionKeys::loops::relax ]() );
		lr_mover->remodel( option[ OptionKeys::loops::remodel ]() );
		lr_mover->cmd_line_csts( true );
		lr_mover->rebuild_filter( option[ OptionKeys::cm::loop_rebuild_filter ]() );
		lr_mover->n_rebuild_tries( option[ OptionKeys::cm::max_loop_rebuild ]() );
		lr_mover->copy_sidechains( true );
		lr_mover->set_current_tag( get_current_tag() );
		lr_mover->apply(pose);
		remove_cutpoint_variants( pose );

		// score pose
		if (pose.is_fullatom())
			(*fapatt_scorefxn_)(pose);
		else
			(*cen_scorefxn_)(pose);
	}

	void apply_threading( Pose &pose, bool superfast=false ) {
		using namespace protocols::loops;
		using namespace protocols::jd2;
		using namespace core::options;
		using namespace core::options::OptionKeys;

		bool debug = option[ OptionKeys::MR::debug ]();

		protocols::moves::SwitchResidueTypeSetMover to_centroid("centroid");
		protocols::moves::SwitchResidueTypeSetMover to_fullatom("fa_standard");

		ThreadingJobCOP job = dynamic_cast< ThreadingJob const*  >(
			JobDistributor::get_instance()->current_job()->inner_job().get() );
		if ( !job ) {
			utility_exit_with_message(
				"CORE ERROR: You must use the ThreadingJobInputter with the LoopRelaxThreadingMover "
				"- did you forget the -in:file:template_pdb option?" );
		}

		// set up initial loop build
		core::Size nres = pose.total_residue();
		while (!pose.residue(nres).is_polymer()) nres--;
		Loops my_loops( job->loops( nres ) );

		if ( option[ MR::max_gaplength_to_model ].user() ) {
			trim_target_pose( pose, my_loops, option[ MR::max_gaplength_to_model ]() );
		}

		// find disulfides conserved from template
		pose.conformation().detect_disulfides();

		// pack all missing SCs
		pack_missing_sidechains( pose );

		// setup for symmetry
		if ( option[ OptionKeys::symmetry::symmetry_definition ].user() )  {
			protocols::moves::symmetry::SetupForSymmetryMover pre_mover;
			pre_mover.apply( pose );
		} else {
			core::pose::addVirtualResAsRoot( pose );
		}

		if ( option[ OptionKeys::edensity::mapfile ].user() ) {
			protocols::moves::MoverOP dens_dock( new protocols::electron_density::SetupForDensityScoringMover );
			dens_dock->apply( pose );
		}

		if (debug) pose.dump_pdb( "pre_loopmodel.pdb" );

		my_loops.choose_cutpoints( pose );

		if ( superfast ) {  // no loop modeling
			// repack/minimize
			apply_repackminimize( pose );
		} else if (option[ MR::max_gaplength_to_model ]() == 0) {  // no loop modeling
			// fastrelax only
			apply_relax( pose );
		} else {
			LoopRelaxMoverOP lr_mover( new LoopRelaxMover );
			lr_mover->scorefxns( cen_scorefxn_, fapatt_scorefxn_ );   // centroid threading doesn't use density score
			lr_mover->frag_libs( frag_libs_ );
			lr_mover->loops( my_loops );
			lr_mover->relax( option[ OptionKeys::loops::relax ]() );
			lr_mover->remodel( option[ OptionKeys::loops::remodel ]() );
			lr_mover->cmd_line_csts( true );
			lr_mover->rebuild_filter( option[ OptionKeys::cm::loop_rebuild_filter ]() );
			lr_mover->n_rebuild_tries( option[ OptionKeys::cm::max_loop_rebuild ]() );
			lr_mover->copy_sidechains( true );
			lr_mover->set_current_tag( get_current_tag() );

			lr_mover->apply( pose );
			remove_cutpoint_variants( pose );
		}

		// rescore pose
		if (pose.is_fullatom())
			(*fapatt_scorefxn_)(pose);
		else
			(*cen_scorefxn_)(pose);
	}

	void apply_makefrags( Pose &pose ) {
		using namespace protocols::loops;
		using namespace protocols::jd2;
		using namespace core::options;
		using namespace core::options::OptionKeys;
		using namespace core::fragment;
		using namespace utility::file;

		ThreadingJobCOP job = dynamic_cast< ThreadingJob const*  >(
			JobDistributor::get_instance()->current_job()->inner_job().get() );
		if ( !job ) {
			utility_exit_with_message(
				"CORE ERROR: You must use the ThreadingJobInputter with the LoopRelaxThreadingMover "
				"- did you forget the -in:file:template_pdb option?" );
		}

		// set up initial loop build
		core::Size nres = pose.total_residue();
		while (!pose.residue(nres).is_polymer()) nres--;
		Loops my_loops( job->loops( nres ) );

		if ( !option[ MR::max_gaplength_to_model ].user() ) return;   // nothing to do

		trim_target_pose( pose, my_loops, option[ MR::max_gaplength_to_model ]() );

		// dump remapped fragfiles
		utility::vector1<int> frag_sizes( option[ OptionKeys::loops::frag_sizes ] );
		FileVectorOption frag_files( option[ OptionKeys::loops::frag_files ] );
		runtime_assert( frag_sizes.size() == frag_libs_.size());
		for (int i=1; i<=frag_libs_.size(); ++i) {
			// don't dump 1mers
			if (frag_sizes[i] == 1 || frag_files[i] == std::string("none") ) continue;
			FragmentIO().write_data( option[ OptionKeys::out::file::frag_prefix ]+ ObjexxFCL::string_of(frag_sizes[i],0) , *frag_libs_[i] );
		}
	}

	void apply( Pose & pose ) {
		using namespace core::options;
		using namespace core::options::OptionKeys;

		std::string mode = option[ OptionKeys::MR::mode ]();
		if ( mode == "cm" ) {
			apply_threading( pose );
		} else if ( mode == "fastcm" ) {
			apply_threading( pose, true );
		} else if ( mode == "cmfrags" ) {
			apply_makefrags( pose ); //
		} else if ( mode == "loopmodel" ) {
			apply_loopmodel( pose ); //
		} else if ( mode == "relax" ){
			apply_relax( pose );
		} else if ( mode == "bbg" ) {
			apply_bbg( pose ); //
		} else if ( mode == "autorb" ) {
			apply_autorb( pose ); //
		} else {
			TR.Error << "UNKNOWN MODE " << mode << std::endl;
		}
	}
};

///////////////////////////////////////////////////////////////////////////////

void*
my_main( void* ) {
	using namespace protocols::moves;
	using namespace core::options;
	using namespace core::options::OptionKeys;

	SequenceMoverOP seq( new SequenceMover() );
	seq->add_mover( new MRMover() );

	try{
		protocols::jd2::JobDistributor::get_instance()->go( seq );
	} catch ( utility::excn::EXCN_Base& excn ) {
		std::cerr << "Exception: " << std::endl;
		excn.show( std::cerr );
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////////

int
main( int argc, char * argv [] ) {
	protocols::moves::CanonicalSamplingMover::register_options();

	// initialize option and random number system
	NEW_OPT(MR::max_gaplength_to_model, "max gaplength to rebuild", true);
    NEW_OPT(MR::mode, "mode", "cm");
	NEW_OPT(MR::debug, "output debug pdbs?", false);
	protocols::init( argc, argv );

	protocols::viewer::viewer_main( my_main );
}
