// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/core/chemical/Residue_.cc
/// @brief implementation of implementation class for abstract class Residue
/// @author Phil Bradley

// Unit headers
#include <core/chemical/Patch.hh>

// Package Headers
// Commented by inclean daemon #include <core/chemical/PatchOperation.hh>


// ObjexxFCL headers

// Numeric headers


// Utility headers


// C++ headers
// Commented by inclean daemon #include <sstream>
#include <fstream>

//Auto Headers
#include <core/util/Tracer.hh>


namespace core {
namespace chemical {

static util::Tracer tr("core.chemical");

/// @brief the string used to generate new residue names
std::string const patch_linker( "_p:" );

std::string
residue_type_base_name( ResidueType const & rsd_type )
{
	return rsd_type.name().substr( 0, rsd_type.name().find( patch_linker ) );
}

std::string
residue_type_all_patches_name( ResidueType const & rsd_type )
{
	Size spos = rsd_type.name().find( patch_linker );
	if( spos < rsd_type.name().length() ) return rsd_type.name().substr( spos );
	else return "";
}


/// @brief handy function, return the first word from a line
std::string
tag_from_line( std::string const & line )
{
	std::string tag;
	std::istringstream l( line );
	l >> tag;
	if ( l.fail() ) return "";
	else return tag;
}

/// @brief create a PatchCase from input lines
/// @details add selector_ from lines enclosed by "BEGIN_SELECTOR" and "END_SELECTOR".\n
/// add operations_ from each input line containing a single operation

PatchCaseOP
case_from_lines(
	utility::vector1< std::string > const & lines
)
{
	PatchCaseOP pcase( new PatchCase() );

	bool in_selector( false );
	for ( uint i=1; i<= lines.size(); ++i ) {
		std::string const tag( tag_from_line( lines[i] ) );

		if ( tag == "BEGIN_SELECTOR" ) {
			assert( !in_selector );
			in_selector = true;
		} else if ( tag == "END_SELECTOR" ) {
			in_selector = false;
		} else if ( in_selector ) {
			pcase->selector().add_line( lines[i] );
		} else {
			PatchOperationOP operation( patch_operation_from_patch_file_line( lines[i] ) );
			if ( operation ) pcase->add_operation( operation );
		}
	}

	return pcase;
}

/// @details First clone the base ResidueType. Then patching for this case is done by applying all the operations. Finally
///	 call finalize() to update all primary and derived data for the new ResidueType
ResidueTypeOP
PatchCase::apply( ResidueType const & rsd_in ) const
{
	ResidueTypeOP rsd( rsd_in.clone() );

	for ( utility::vector1< PatchOperationOP >::const_iterator iter= operations_.begin(),
					iter_end= operations_.end(); iter != iter_end; ++iter ) {
		bool const fail( (*iter)->apply( *rsd ) );
		if ( fail ) return 0;
	}
	rsd->finalize();
	return rsd;
}

/// @detals	- first read in all lines from the file, discarding # comment lines
/// - parse input lines for Patch name and variant types (NAME, TYPES)
///	- parse input lines for general ResidueSelector defined for this Patch (BEGIN_SELECTOR, END_SELECTOR)
///	- parse input lines to create each case acoordingly (BEGIN_CASE, END_CASE)
/// @note keep the order to avoid triggering parsing errors
void
Patch::read_file( std::string const & filename )
{
	// clear old data
	name_ = "";
	types_.clear();
	selector_.clear();
	cases_.clear();
	replaces_residue_type_ = false;

	utility::vector1< std::string > lines;
	{ // read the lines file
		std::ifstream data( filename.c_str() );
		std::string line;
		while ( getline( data,line ) ) {
			std::string const tag( tag_from_line( line ) );
			if ( tag.size() && tag[0] != '#' ) lines.push_back( line );
		}
	}

	// misc parsing
	for ( uint i=1; i<= lines.size(); ++i ) {
		std::istringstream l(lines[i]);
		std::string tag;
		l >> tag;
		if ( tag == "NAME" ) {
			l >> name_;
		} else if ( tag == "TYPES" ) {
			VariantType t;
			l >> t;
			while ( !l.fail() ) {
				types_.push_back( t );
				l >> t;
			}
		} else if ( tag == "REPLACE_RES_TYPE" ) {
			replaces_residue_type_ = true;
		}
	}

	// build the residue selector
	{
		bool in_selector( false );
		for ( uint i=1; i<= lines.size(); ++i ) {
			std::string tag( tag_from_line( lines[i] ) );
			if ( tag == "BEGIN_CASE" ) {
				break;
			} else if ( tag == "BEGIN_SELECTOR" ) {
				assert( !in_selector );
				in_selector = true;
			} else if ( tag == "END_SELECTOR" ) {
				assert( in_selector );
				in_selector = false;
			} else if ( in_selector ) {
				selector_.add_line( lines[i] );
			}
		}
	}

	// get the cases
	utility::vector1< std::string > case_lines;
	bool in_case( false );
	while ( !lines.empty() ) {
		// look for a case
		std::string tag( tag_from_line( lines[1] ) );
		if ( tag == "BEGIN_CASE" ) {
			assert( case_lines.empty() );
			assert( !in_case );
			in_case = true;
		} else if ( tag == "END_CASE" ) {
			PatchCaseOP new_case( case_from_lines( case_lines ) );
			if ( new_case ) cases_.push_back( new_case );
			case_lines.clear();
			in_case = false;
		} else if ( in_case ) case_lines.push_back( lines[1] );

		lines.erase( lines.begin() );
	}
}

/// @details loop through the cases in this patch and if it is applicable to this ResidueType, the corresponding patch
/// operations are applied to create a new variant type of the basic ResidueType. The new types's name and its
/// variant type info are updated together with all other primary and derived ResidueType data.
ResidueTypeOP
Patch::apply( ResidueType const & rsd_type ) const
{
	if ( !applies_to( rsd_type ) ) return 0; // I don't know how to patch this residue

	using namespace util;
	static core::util::Tracer core_chemical("core.chemical");

	for ( utility::vector1< PatchCaseOP >::const_iterator iter= cases_.begin(),
					iter_end = cases_.end(); iter != iter_end; ++iter ) {

		if ( (*iter)->applies_to( rsd_type ) ) {
			// this patch case applies to this rsd_type
			ResidueTypeOP patched_rsd_type( (*iter)->apply( rsd_type ) );
			if ( patched_rsd_type ) {
				// patch succeeded!
				if ( !replaces_residue_type_ ) {
					for ( utility::vector1< VariantType >::const_iterator iter=types_.begin(),
									iter_end = types_.end(); iter != iter_end; ++iter ) {
						patched_rsd_type->add_variant_type( *iter );
					}
					patched_rsd_type->name( patched_rsd_type->name()+patch_linker+name_ );
				}
				tr.Debug << "successfully patched: " << rsd_type.name() << " to: " << patched_rsd_type->name() << std::endl;
				return patched_rsd_type;
			}
		}
	}
	return 0;
}


} // chemical
} // core
