// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 sw=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file core/chemical/disulfide_util.cc
/// @brief A collection of procedures for manipulating disulfide bonds
/// @author Spencer Bliven <blivens@u.washington.edu>
/// @date 4/30/2009

// Unit Headers
#include <core/chemical/disulfide_util.hh>

// Project Headers
// Package Headers
#include <core/types.hh>
#include <core/pose/Pose.hh>

// AUTO-REMOVED #include <core/conformation/Interface.hh>
#include <core/conformation/Conformation.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/ResidueFactory.hh>
#include <core/conformation/util.hh>
#include <core/chemical/ChemicalManager.hh>
// AUTO-REMOVED #include <core/chemical/util.hh>
#include <core/chemical/ResidueTypeSet.hh>

#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/RotamerSampleOptions.hh>
#include <core/pack/pack_rotamers.hh>
#include <core/kinematics/MoveMap.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>

#include <core/util/Tracer.hh>

// Utility Headers
// AUTO-REMOVED #include <utility/vector1.hh>

// C++ headers
// AUTO-REMOVED #include <utility>

//Auto Headers
#include <core/chemical/VariantType.hh>


namespace core {
namespace chemical {

using namespace core;
using namespace std;

using utility::vector1;
using core::pose::Pose;
using core::pose::PoseOP;
using namespace core::conformation;
using core::pack::task::PackerTaskOP;
using core::scoring::ScoreFunctionOP;
using core::kinematics::MoveMap;
using core::kinematics::MoveMapOP;
static core::util::Tracer TR( "core.chemical.disulfide_util" );


/// @brief Replace a CYS with a CYD or vice-versa for changing disulfide bonds.
/// @param[in] index Position of the residue to replace.
/// @param[in] cys_type_name3 The 3-letter name of the cys type to use: e.g. CYS
///  or CYD.
/// @param[inout] conf The conformation to modify
/// @details Substitutes a residue with the given cys type, keeping as many of
///  the existing atom positions as possible.  If the original residue has a
///  disulfide variant it will be removed, otherwise a disulfide variant will
///  be added.  Should work with any ResidueTypeSet that has the proper
///  disulfide variants.  If the replacement fails for any reason a warning
///  will be printed.
/// @return true if the replacement was successful, false otherwise.
bool change_cys_state( Size const index, std::string cys_type_name3, Conformation & conf ) {
	// Cache information on old residue.
	Residue const & res( conf.residue( index ) );
	chemical::ResidueTypeSet const & residue_type_set = res.type().residue_type_set();

	// make sure we're working on a cys
	if ( res.aa() != chemical::aa_cys ) {
		TR.Warning << "WARNING: change_cys_state() was called on non-cys residue " << index << ", skipping!" << std::endl;
		return false;
	}

	// Get the residue type set of the desired new residue type.  Unfortunately
	// variant types for residues define in different parameter files aren't
	// yet handled via functions such as ResidueTypeSet::get_residue_type_with_variant_*
	// functions, so we need the name3 string to grab the possible residue types and
	// look through them manually to find the right one.
	chemical::ResidueTypeCAPs const & possible_types = residue_type_set.name3_map( cys_type_name3 );

	// Track the variant types of the old residue type.  We want the
	// new residue to have the same variant type as the old.
	utility::vector1< chemical::VariantType > variant_types = res.type().variant_types();

	// check and handle disulfide state
	if ( res.has_variant_type( chemical::DISULFIDE ) && cys_type_name3 == "CYS" ) {
		// if the old residue has DISULFIDE variant type then we are removing a
		// disulfide, so remove the variant type from the list
		variant_types.erase( std::find( variant_types.begin(), variant_types.end(), chemical::DISULFIDE ) );
	} else {
		if ( cys_type_name3 == "CYD" )	variant_types.push_back( chemical::DISULFIDE );	// creating a disulfide
	}

	// Run through all possible new residue types.
	for ( chemical::ResidueTypeCAPs::const_iterator
		type_iter = possible_types.begin(), type_end = possible_types.end();
		type_iter != type_end; ++type_iter )
	{
		bool perfect_match( true ); // indicates this type has all the same variant types as the old residue
		for ( Size kk = 1; kk <= variant_types.size(); ++kk ) {
			if ( ! (*type_iter)->has_variant_type( variant_types[ kk ] ) ) {
				perfect_match = false;
				break;
			}
		}

		if ( perfect_match ) { // Do replacement.
			ResidueOP new_res = ResidueFactory::create_residue( **type_iter, res, conf );
			copy_residue_coordinates_and_rebuild_missing_atoms( res, *new_res, conf );
			conf.replace_residue( index, *new_res, false );

			return true;
		}
	}

	// If we are here then a residue type match wasn't found; issue error message.
	TR.Error << "ERROR: Couldn't find a " << cys_type_name3 << " equivalent for residue " << index << "." <<std::endl;
	return false;
}

/// @brief Introduce cysteines at the specified location and define a
///  disulfide bond between them.
/// @details Does not do the repacking & minimization required to place the
///  disulfide correctly.
void form_disulfide(Pose & pose, Size lower_res, Size upper_res)
{
	// Verify we're dealing with a FA pose
	runtime_assert( pose.is_fullatom() );

	ResidueTypeSetCAP restype_set =
		ChemicalManager::get_instance()->residue_type_set( FA_STANDARD );

	// Break existing disulfide bonds to lower
	if( pose.residue(lower_res).aa() == chemical::aa_cys &&
			pose.residue( lower_res ).has_variant_type( chemical::DISULFIDE ) &&
			pose.residue_type( lower_res ).has_atom_name( "SG" ) // full atom residue
			)
	{
		Size const connect_atom( pose.residue( lower_res ).atom_index( "SG" ) );
		Size other_res( 0 );
		Size conn(0);
		for ( conn = pose.residue( lower_res ).type().n_residue_connections(); conn >= 1; --conn ) {
			if( Size( pose.residue(lower_res).type().residue_connection(conn).atomno() ) == connect_atom ) {
				other_res = pose.residue( lower_res ).connect_map( conn ).resid();
				break;
			}
		}
		if ( other_res == 0 ) {
			TR.Error << "Error: Residue " << lower_res << " was disulfide bonded but had no partner" << std::endl;
			utility_exit();
		}

		if(other_res == upper_res) {
			// Already a disulfide bond
			runtime_assert_msg(pose.residue( upper_res ).connect_map( conn ).resid() == lower_res,
				"Error: Disulfide bond wasn't reciprical");
			return;
		}

		// Break the disulfide bond to upper_res
		bool result = change_cys_state( other_res, "CYS", pose.conformation() );
		runtime_assert_msg(result,"Error converting CYD->CYS");
	}
	else {
		ResidueOP lower_cyd = ResidueFactory::create_residue(
			restype_set->name_map("CYD"), pose.residue(lower_res),
			pose.conformation());
		copy_residue_coordinates_and_rebuild_missing_atoms(
			pose.residue(lower_res), *lower_cyd, pose.conformation() );
		pose.replace_residue(lower_res, *lower_cyd, false /*backbone already oriented*/);
	}
	// Break existing disulfide bonds to upper
	if( pose.residue(upper_res).aa() == chemical::aa_cys &&
			pose.residue( upper_res ).has_variant_type( chemical::DISULFIDE ) &&
			pose.residue_type( upper_res ).has_atom_name( "SG" ) // full atom residue
			)
	{
		Size const connect_atom( pose.residue( upper_res ).atom_index( "SG" ) );
		Size other_res( 0 );
		Size conn(0);
		for ( conn = pose.residue( upper_res ).type().n_residue_connections(); conn >= 1; --conn ) {
			if( Size( pose.residue(upper_res).type().residue_connection(conn).atomno() ) == connect_atom ) {
				other_res = pose.residue( upper_res ).connect_map( conn ).resid();
				break;
			}
		}
		if ( other_res == 0 ) {
			TR.Warning << "Warning: Residue " << upper_res << " was disulfide bonded but had no partner" << std::endl;
			utility_exit();
		}

		// Break the disulfide bond to lower_res
		bool result = change_cys_state( other_res, "CYS", pose.conformation());
		runtime_assert_msg(result,"Error converting CYD->CYS");
	}
	else {
		ResidueOP upper_cyd = ResidueFactory::create_residue(
			restype_set->name_map("CYD"), pose.residue(upper_res),
			pose.conformation());
		copy_residue_coordinates_and_rebuild_missing_atoms(
			pose.residue(upper_res), *upper_cyd, pose.conformation() );
		pose.replace_residue(upper_res, *upper_cyd, false /*backbone already oriented*/);
	}

	// Both residues are now CYD
	runtime_assert(pose.residue(lower_res).name() == "CYD" && pose.residue(upper_res).name() == "CYD" );

	//form the bond between the two residues
	pose.conformation().declare_chemical_bond(lower_res,"SG",upper_res,"SG");

}

/// @details A convenience function for calling \c rebuild_disulfide() with
///  only a single disulfide bond
void rebuild_disulfide( Pose & pose, Size lower_res, Size upper_res,
		PackerTaskOP packer_task, ScoreFunctionOP packer_score,
		MoveMapOP mm, ScoreFunctionOP minimizer_score )
{
	vector1< pair<Size,Size> > disulfides;
	disulfides.push_back(std::make_pair(lower_res,upper_res));
	rebuild_disulfide(pose, disulfides, packer_task, packer_score, mm, minimizer_score);
}


/// @details
/// @pre The two residues specified should already be cysteines with a bond
///  defined between the SG atoms. This can be accomplished by first calling
///  form_disulfide().
///
///  To provide the most flexibility, this function does not restrict
///  the degrees of freedom in the repacking and minimization steps. This is
///  needed in some cases where the backbone position is not optimal for making
///  a disulfide bond. However, if this function needs to be reasonably fast
///  you should prevent repacking on all residues except the two targets.
///
/// @param pose[in,out] The pose to modify with a disulfides
/// @param lower_res[in] The first residue of the disulfide
/// @param upper_res[in] The second residue of the disulfide
/// @param packer_task[in] A task to control repacking. Defaults to repacking
///  lower_res \& upper_res if ommitted or NULL.
/// @param packer_score[in] A scoring function to use while repacking. Defaults
///  to score12 if ommitted or NULL.
/// @param mm[in] A MoveMap to control minimization. Defaults to full degrees
///  of freedom if ommitted or NULL.
/// @param minimizer_score[in] The score to use for minimization. Defaults to
///  packer_score if ommitted or NULL.
void rebuild_disulfide( core::pose::Pose & pose,
	utility::vector1<std::pair<core::Size, core::Size> > disulfides,
	core::pack::task::PackerTaskOP packer_task,
	core::scoring::ScoreFunctionOP packer_score,
	core::kinematics::MoveMapOP mm,
	core::scoring::ScoreFunctionOP minimizer_score )
//void rebuild_disulfide( Pose & pose, vector1<pair<Size, Size> > const& disulfides,
//		PackerTaskOP packer_task, ScoreFunctionOP packer_score,
//		MoveMapOP mm, ScoreFunctionOP minimizer_score )
{
	// Quick lookup of whether a residue is a disulfide or not
	vector1<bool> is_disulf(pose.total_residue(), false);

	for(vector1<pair<Size, Size> >::const_iterator
			disulf(disulfides.begin()), end_disulf(disulfides.end());
			disulf != end_disulf; ++disulf)
	{
		is_disulf[disulf->first] = true;
		is_disulf[disulf->second] = true;

		// Verify precondition
		if( ! is_disulfide_bond(pose, disulf->first, disulf->second) ) {
			TR.Error << "Disulfide bond required between " << disulf->first
				<< " and " << disulf->second << "." << std::endl;
			utility_exit();
		}
	}

	// Set up any NULL parameters
	if( !packer_task ) {
		packer_task = core::pack::task::TaskFactory::create_packer_task( pose );
		packer_task->initialize_from_command_line().or_include_current( true );
		packer_task->restrict_to_repacking();

		// Restrict repacking to the targets
		for( Size i(1); i <= pose.total_residue(); ++i )
		{
			if( !is_disulf[i] ) {
				packer_task->nonconst_residue_task(i).prevent_repacking();
			}
		}
	}
	if( !packer_score ) {
		packer_score = scoring::getScoreFunction();
	}
	if( !mm ) {
		mm = MoveMapOP(new MoveMap);
		mm->set_bb( true );
		mm->set_chi( true );
	}
	if( !minimizer_score ) {
		minimizer_score = packer_score;
	}

	// Extend rotamers for the disulfide
	for(vector1<pair<Size, Size> >::const_iterator
		disulf(disulfides.begin()), end_disulf(disulfides.end());
		disulf != end_disulf; ++disulf)
	{
		packer_task->nonconst_residue_task(disulf->first).and_extrachi_cutoff( 0 );
		packer_task->nonconst_residue_task(disulf->second).and_extrachi_cutoff( 0 );
		packer_task->nonconst_residue_task(disulf->first).or_ex1_sample_level(
			pack::task::EX_SIX_QUARTER_STEP_STDDEVS);
		packer_task->nonconst_residue_task(disulf->second).or_ex1_sample_level(
			pack::task::EX_SIX_QUARTER_STEP_STDDEVS);
	}

	// REPACK
	(*packer_score)(pose); // structure must be scored before rotamer_trials can be called
	pack::pack_rotamers(pose, *packer_score, packer_task );

	using namespace core::optimization;
	AtomTreeMinimizer().run( pose, *mm, *minimizer_score,
			MinimizerOptions( "dfpmin_armijo_nonmonotone", 0.01, true/*nblist*/, false/*deriv_check*/ ) );

	// update score
	pose.update_residue_neighbors();
	(*minimizer_score)( pose );

}

/// @brief Find whether there is a disulfide defined between two residues
///
/// @details We define a disulfide to exist between a pair of residues iff
///  -# They are both cysteines
///  -# They are bonded by their sidechains
bool
is_disulfide_bond( pose::Pose const& pose, Size residueA_pos, Size residueB_pos)
{
	Residue const& A = pose.residue(residueA_pos);
	Residue const& B = pose.residue(residueB_pos);

	if( !A.is_protein() || !B.is_protein() )
		return false;

	//both Cys or CysD
	if( A.type().name1() != 'C' || B.type().name1() != 'C' )
		return false;

	//bonded
	Size a_connect_atom;
	if( A.type().has_atom_name( "SG" ) )
		a_connect_atom = A.atom_index( "SG" );
	else {
		runtime_assert( A.type().has_atom_name( "CEN" ) ); //should be fa or centroid
		a_connect_atom = A.atom_index( "CEN" );
	}
	for ( Size connection = A.type().n_residue_connections(); connection >= 1; --connection ) {
		//check if A bonded to B
		if ( (Size) A.type().residue_connection( connection ).atomno() == a_connect_atom && //bond to sg, not the backbone
				A.connect_map( connection ).resid() == residueB_pos ) { //bonded to B
			return true;
		}
	}

	return false;
}

/// @brief Generate a list of all disulfide bonds in the pose
void disulfide_bonds( pose::Pose const& pose, vector1< pair<Size,Size> > & disulfides )
{
	for( Size i=1; i<= pose.total_residue(); ++i)
	{
		Residue const& res(pose.residue(i));

		// Skip things besides CYD
		if( !(res.aa() == aa_cys && res.has_variant_type(chemical::DISULFIDE) ))
			continue;
		Size connect_atom( 0);
		if( res.type().has_atom_name( "SG" ))
			connect_atom = res.atom_index( "SG" );
		else if( res.type().has_atom_name( "CEN" ))
			connect_atom = res.atom_index( "CEN" );
		else {
			TR.Warning << "Warning: unable to establish which atom to use for the disulfide to residue "
				<< i << std::endl;
			continue;
		}

		Size other_res(0);
		Size conn;
		for( conn = pose.residue( i ).type().n_residue_connections(); conn >= 1; --conn ) {
			if( Size( pose.residue( i ).type().residue_connection(conn).atomno() ) == connect_atom ) {
				other_res = pose.residue( i ).connect_map( conn ).resid();
				break;
			}
		}
		if ( other_res == 0 ) {
			TR.Error << "Error: Residue " << i << " was disulfide bonded but had no partner" << std::endl;
			utility_exit();
		}

		// Output the pair once
		if( i < other_res ) {
			disulfides.push_back( std::make_pair(i, other_res) );
		}
	}
}

} // ProteinInterfaceDesign
} // devel
