// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragment/picking/FragmentPicker.cc
/// @brief  Fragment picker - the core part of picking machinery
/// @author Dominik Gront (dgront@chem.uw.edu.pl)

// unit headers
#include <core/fragment/picking/FragmentPicker.hh>

#include <core/fragment/picking/VallProvider.hh>
#include <core/fragment/picking/VallChunk.hh>
#include <core/fragment/picking/VallResidue.hh>
#include <core/fragment/picking/VallChunkFilter.hh>
#include <core/fragment/picking/FragmentCandidate.hh>
#include <core/fragment/picking/CandidatesCollector.hh>
#include <core/fragment/picking/GrabAllCollector.hh>
#include <core/fragment/picking/quota/QuotaSelector.hh>
#include <core/fragment/picking/quota/QuotaConfig.hh>
#include <core/fragment/picking/quota/QuotaCollector.hh>
#include <core/fragment/picking/quota/ABEGO_SS_Config.hh>
#include <core/fragment/picking/quota/ABEGO_SS_Pool.hh>
#include <core/fragment/picking/quota/SecondaryStructurePool.hh>
#include <core/fragment/picking/BoundedCollector.hh>
#include <core/fragment/picking/PdbIdChunkFilter.hh>
#include <core/fragment/picking/BestTotalScoreSelector.hh>
#include <core/fragment/picking/scores/FragmentScoringMethod.hh>
#include <core/fragment/picking/scores/FragmentScoreManager.hh>
#include <core/fragment/picking/scores/SecondarySimilarity.hh>
#include <core/fragment/picking/scores/PartialSecondarySimilarity.hh>
#include <core/fragment/picking/scores/ProfileScoreL1.hh>
#include <core/fragment/picking/scores/RamaScore.hh>
#include <core/fragment/picking/scores/CSScore.hh>
#include <core/fragment/picking/scores/ABEGO_SS_Score.hh>
#include <core/fragment/picking/scores/TorsionBinSimilarity.hh>
#include <core/fragment/picking/scores/FragmentScoreMap.hh>
#include <core/fragment/picking/TorsionBinIO.hh>

//#include <core/conformation/SecondaryStructure.hh>

// option key includes
#include <core/options/option.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/frags.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>

#include <core/sequence/util.hh>
#include <core/util/prof.hh>
#include <core/util/Tracer.hh>

#include <utility/exit.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
//#include <utility/io/izstream.hh>
#include <utility>
#include <sstream>
#include <iostream>
#include <fstream>

namespace core {
namespace fragment {
namespace picking {

using namespace core;
using namespace core::fragment;
using namespace core::fragment::picking;
using namespace core::fragment::picking::scores;
using namespace options;
using namespace options::OptionKeys;

static core::util::Tracer trPicker("core.fragments.picking.FragmentPicker");

FragmentPicker::~FragmentPicker() {}

void FragmentPicker::bounded_protocol() {

        pick_candidates();
	save_fragments();
}

void FragmentPicker::quota_protocol() {
	using namespace ObjexxFCL;

        pick_candidates();

	FragmentScoreManagerOP ms = get_score_manager();
	for (Size iFragSize = 1; iFragSize <= frag_sizes_.size(); ++iFragSize) { // Loop over various sizes of fragments
		Size fragment_size = frag_sizes_[iFragSize];
		std::string out_file_name = prefix_ + "." + string_of(fragment_size)
				+ "mers";
		utility::io::ozstream output(out_file_name);
		CandidatesCollectorOP storage = get_candidates_collector(fragment_size);
		quota::QuotaCollector *c =
			    dynamic_cast<quota::QuotaCollector*> (storage());
		if (c == 0) {
		    utility_exit_with_message("Cant' cast candidates' collector to QuotaCollector. Is quota set up correctly?");
		}

		log_25_.setup_summary(c);
		log_200_.setup_summary(c);

		std::ofstream out_file;
		if (option[frags::describe_fragments].user()) {
		    std::string describe_name = option[frags::describe_fragments]()+"."+string_of(fragment_size)+"mers";
		    out_file.open(describe_name.c_str());
		}

		for (Size iqpos = 1; iqpos <= query_positions_.size(); ++iqpos) {

			Size qPos = query_positions_[iqpos];
			if ( qPos > size_of_query() - fragment_size + 1 ) continue;

			utility::vector1<std::pair<FragmentCandidateOP, FragmentScoreMapOP> > dummy_input;	// we don't need them, but some parameter has to be passed to the method
			utility::vector1<std::pair<FragmentCandidateOP, FragmentScoreMapOP> > out;

			    quota::QuotaSelector selector(n_frags_,qPos, c );

			    selector.select_fragments(dummy_input,out);
			    output << "position: " << I(12, qPos) << " neighbors:   " << I(10,
					out.size()) << std::endl << std::endl;
			    for (Size fi = 1; fi <= out.size(); ++fi) {
				out[fi].first->print_fragment(output);
				output << std::endl;
			    }
			    if( ms->if_late_scoring_for_zeros() )  {
				for (Size fi = 1; fi <= out.size(); ++fi)
				    ms->score_zero_scores(out[fi].first,out[fi].second);
			    }
			    if (option[frags::describe_fragments].user()) {
				ms->describe_fragments(out, out_file);
			    }
		}
		log_25_.write_summary();
		log_200_.write_summary();
//		storage->print_report(trPicker.Debug, ms);
		output.close();
	}
}

void FragmentPicker::keep_all_protocol() {
	using namespace ObjexxFCL;

	CompareTotalScore comparator(get_score_manager());
	for (Size iFragSize = 1; iFragSize <= frag_sizes_.size(); ++iFragSize) { // Loop over various sizes of fragments
		Size fragment_size = frag_sizes_[iFragSize];

	    std::ofstream out_file;
	    if (option[frags::describe_fragments].user()) {
		std::string describe_name = option[frags::describe_fragments]()+"."+string_of(fragment_size)+"mers";
		out_file.open(describe_name.c_str());
	    }

	    std::string out_file_name = prefix_ + "." + string_of(fragment_size)
				+ "mers";
	    utility::io::ozstream output(out_file_name);
	    CandidatesCollectorOP storage = get_candidates_collector(fragment_size);

	    for (Size iqpos = 1; iqpos <= query_positions_.size(); ++iqpos) {

		Size qPos = query_positions_[iqpos];
		if ( qPos > size_of_query() - fragment_size + 1 ) continue;

		utility::vector1<std::pair<FragmentCandidateOP, FragmentScoreMapOP> >	out;
		pick_candidates(qPos,fragment_size);
		utility::vector1<std::pair<FragmentCandidateOP, FragmentScoreMapOP> >  candidates = storage->get_candidates(qPos);
		std::sort(candidates.begin(),candidates.end(),comparator);
		selector_->select_fragments(candidates, out);
		if(out.size() == 0) continue;
		output << "position: " << I(12, qPos) << " neighbors:   " << I(10,
					out.size()) << std::endl << std::endl;
		FragmentScoreManagerOP ms = get_score_manager();
		if( ms->if_late_scoring_for_zeros() )  {
			for (Size fi = 1; fi <= out.size(); ++fi)
				ms->score_zero_scores(out[fi].first,out[fi].second);
		}
		for (Size fi = 1; fi <= out.size(); ++fi) {
			out[fi].first->print_fragment(output);
			output << std::endl;
		}
		if (option[frags::describe_fragments].user()) {
			get_score_manager()->describe_fragments(out, out_file);
		}
		trPicker.Info << "Collected candidates of size "<<fragment_size<<" at pos"<<qPos<<std::endl;
		storage->clear();
		trPicker.Debug<< storage->count_candidates()<<" candidates left in a sink after flushing"<<std::endl;
	    }
	    output.close();
	    out_file.close();
	}
	trPicker.Info<<std::endl;
}

void FragmentPicker::pick_candidates() {

	PROF_START( core::util::FRAGMENTPICKING );
	// Size n_total = chunks_->size()/100;

	scores::FragmentScoreMapOP empty_map = scores_->create_empty_map();
	for (Size i = 1; i <= chunks_->size(); i++) { // loop over provided chunks
		VallChunkOP chunk = chunks_->at(i); // For each chunk from a provider...
		if (chunk->size() < max_frag_size_) // This fragment is too short
			continue;
		bool flag = true;
		for (Size iFilter = 1; iFilter <= filters_.size(); iFilter++) {
			if ((flag = filters_[iFilter]->test_chunk(chunk)) == false) {
				trPicker.Debug << "Chunk: " << chunk->get_pdb_id()
						<< " didn't pass a filter" << std::endl;
				break;
			}
		}
		if (!flag)
			continue;
		trPicker.Debug << "Processing sequence from vall: "
				<< chunk->get_sequence() << std::endl;

		// cache the new chunk
		scores_->do_caching(chunk);

		for (Size iFragSize = 1; iFragSize <= frag_sizes_.size(); ++iFragSize) { // Loop over various sizes of fragments
			Size fragment_size = frag_sizes_[iFragSize];
			if (chunk->size() < fragment_size) // This fragment is too short
				continue;

			CandidatesCollectorOP sink = candidates_sink_[fragment_size];
			trPicker.Debug << "Picking fragments of size "<<fragment_size<<
				" at "<<query_positions_.size()<<" query positions"<<std::endl;
			for (Size iqpos = 1; iqpos <= query_positions_.size(); ++iqpos) { // loop over positions in a query

			    Size iPos = query_positions_[iqpos];
			    if ( iPos > size_of_query() - fragment_size + 1 ) continue;

				// split chunk into fragment candidates and score them
				for (Size j = 1; j <= chunk->size() - fragment_size + 1; j++) {
					FragmentCandidateOP f = new FragmentCandidate(iPos, j,
							chunk, fragment_size);
					if (scores_->score_fragment_from_cache(f, empty_map)) {
						std::pair<FragmentCandidateOP,scores::FragmentScoreMapOP> p(f,empty_map);
						if(sink->add(p))
						    empty_map  = scores_->create_empty_map();
/*
						scores::FragmentScoreMapOP new_map = empty_map->clone();
						std::pair<FragmentCandidateOP,
								scores::FragmentScoreMapOP> p(f, new_map);
						sink->add(p);
*/
					}
				}
			} // all query positions done
		} // all fragment sizes done
		scores_->clean_up();
		trPicker.Debug << chunk->get_pdb_id() << " done" << std::endl;
		if( (i*100) % chunks_->size() == 0 ) trPicker.Info << (i*100) / chunks_->size()
		    << "% done at "<<chunk->get_pdb_id()<< std::endl;
		trPicker.flush();
	} // all chunks done
	PROF_STOP( core::util::FRAGMENTPICKING );
}

double FragmentPicker::total_score(scores::FragmentScoreMapOP f) {

	utility::vector1<Real> components = f->get_score_components();
	utility::vector1<Real> weights = scores_->get_weights();
	Real total = 0.0;
	for (Size i = 1; i <= components.size(); i++)
		total += components.at(i) * weights.at(i);

	return total;
}


void FragmentPicker::read_ss_files(utility::vector1<std::string> sec_str_input) {

    trPicker.Debug << sec_str_input.size() / 2
		<< " secondary structure assignment(s):\n";
    for (Size i = 1; i <= sec_str_input.size(); i += 2) {
    	trPicker.Debug << i / 2 << " " << sec_str_input[i]
			<< " file will be loaded under \"" << sec_str_input[i + 1]
			<< "\" name\n";
	read_ss_file(sec_str_input[i], sec_str_input[i + 1]);
    }
    trPicker.Debug << std::endl;
}

void FragmentPicker::read_ss_file(std::string const & file_name,
		std::string prediction_name) {

	utility::io::izstream data( file_name.c_str() );
	 if ( !data ) {
	    data.close();
	    utility_exit_with_message( "Can't read secondary structure file: "+file_name );
	}

	std::string line, l1, l2, l3, l4, l5;
	getline( data, line );
	data.close();

	std::istringstream line_stream( line );
	line_stream >> l1 >> l2 >> l3 >> l4 >> l5;

	if ( (l1 == "#") && (l2 == "PSIPRED") && (l3 == "VFORMAT")
			 && (l4 == "(PSIPRED") ) {
		read_psipred_ss2( file_name, prediction_name);
	} else {
		if ( (l1 == "REMARK") && (l2 == "Neural") && (l3 == "network")
			 && (l4 == "secondary") && (l5 == "structure") ) {
			read_talos_ss( file_name, prediction_name);
		} else {
			utility_exit_with_message( "Can't identify secondary structure file type (needs vertical psipred_ss2 or talos+ pred.ss): "+file_name );
		}
	}
}

void FragmentPicker::read_psipred_ss2(std::string const & file_name,
		std::string prediction_name) {

	core::conformation::SecondaryStructureOP ss_profile =
			new core::conformation::SecondaryStructure();
	ss_profile->read_psipred_ss2(file_name);

	std::string query_ss_as_string;
	for (Size i = 1; i <= ss_profile->total_residue(); i++)
		query_ss_as_string += ss_profile->secstruct(i);

	query_ss_as_string_[prediction_name] = query_ss_as_string;
	query_ss_profile_[prediction_name] = ss_profile;
}

void FragmentPicker::read_talos_ss(std::string const & file_name,
		std::string prediction_name) {

	core::conformation::SecondaryStructureOP ss_profile =
			new core::conformation::SecondaryStructure();
	ss_profile->read_talos_ss(file_name);

	std::string query_ss_as_string;
	for (Size i = 1; i <= ss_profile->total_residue(); i++)
		query_ss_as_string += ss_profile->secstruct(i);

	query_ss_as_string_[prediction_name] = query_ss_as_string;
	query_ss_profile_[prediction_name] = ss_profile;
}

void FragmentPicker::add_query_ss(std::string query_secondary,
		std::string prediction_name) {

	core::conformation::SecondaryStructureOP ss_profile =
			new core::conformation::SecondaryStructure();
	ss_profile->extend(query_secondary.length());

	for (Size i = 1; i <= query_secondary.length(); ++i) {
		char ss = query_secondary[i - 1];
		if (ss == 'E')
			ss_profile->set_fractions(i, 0.0, 1.0, 0.0);
		else if (ss == 'L')
			ss_profile->set_fractions(i, 0.0, 0.0, 1.0);
		else
			ss_profile->set_fractions(i, 1.0, 0.0, 0.0);
	}
	query_ss_as_string_[prediction_name] = query_secondary;
	query_ss_profile_[prediction_name] = ss_profile;
}

void FragmentPicker::save_fragments() {
	using namespace ObjexxFCL;

	FragmentScoreManagerOP ms = get_score_manager();
	CompareTotalScore comparator(ms);
	for (Size iFragSize = 1; iFragSize <= frag_sizes_.size(); ++iFragSize) { // Loop over various sizes of fragments

		Size fragment_size = frag_sizes_[iFragSize];

		std::ofstream out_file;
		if (option[frags::describe_fragments].user()) {
			std::string describe_name = option[frags::describe_fragments]()+"."+string_of(fragment_size)+"mers";
			out_file.open(describe_name.c_str());
		}

		std::string out_file_name = prefix_ + "." + string_of(fragment_size)
				+ "mers";
		utility::io::ozstream output(out_file_name);
		CandidatesCollectorOP storage = get_candidates_collector(fragment_size);
		for (Size qPos = 1; qPos <= size_of_query(); ++qPos) {
			utility::vector1<std::pair<FragmentCandidateOP, FragmentScoreMapOP> >
					out;
			if(storage->get_candidates(qPos).size() == 0) continue;
			selector_->select_fragments(storage->get_candidates(qPos), out);
			output << "position: " << I(12, qPos) << " neighbors:   " << I(10,
					out.size()) << std::endl << std::endl;
			for (Size fi = 1; fi <= out.size(); ++fi) {
				out[fi].first->print_fragment(output);
				output << std::endl;
			}
			if( ms->if_late_scoring_for_zeros() )  {
				for (Size fi = 1; fi <= out.size(); ++fi)
				    ms->score_zero_scores(out[fi].first,out[fi].second);
			}
			if (option[frags::describe_fragments].user()) {
				ms->describe_fragments(out, out_file);
			}
		}
		storage->print_report(trPicker.Info, get_score_manager());
		output.close();
		out_file.close();
	}
}

void FragmentPicker::save_candidates() {
	using namespace ObjexxFCL;

	for (Size iFragSize = 1; iFragSize <= frag_sizes_.size(); ++iFragSize) { // Loop over various sizes of fragments
		Size fragment_size = frag_sizes_[iFragSize];
		std::string out_file_name = prefix_ + "." + string_of(fragment_size)
				+ "mers";
		utility::io::ozstream output(out_file_name);
		CandidatesCollectorOP storage = get_candidates_collector(fragment_size);

		for (Size qPos = 1; qPos <= size_of_query(); ++qPos) {
			utility::vector1<std::pair<FragmentCandidateOP, FragmentScoreMapOP> >
					out(storage->get_candidates(qPos));
			output << "position: " << I(12, qPos) << " neighbors:   " << I(10,
					out.size()) << std::endl << std::endl;
			for (Size fi = 1; fi <= out.size(); ++fi) {
				out[fi].first->print_fragment(output);
				output << std::endl;
			}
		}
		storage->print_report(trPicker.Debug, get_score_manager());
		output.close();
	}
}

void FragmentPicker::pick_candidates(Size i_pos,Size frag_len) {

	scores::FragmentScoreMapOP empty_map = scores_->create_empty_map();
	for (Size i = 1; i <= chunks_->size(); i++) { // loop over provided chunks
		VallChunkOP chunk = chunks_->at(i); // For each chunk from a provider...
		if (chunk->size() < frag_len) // This fragment is too short
			continue;
		bool flag = true;
		for (Size iFilter = 1; iFilter <= filters_.size(); iFilter++) {
			if ((flag = filters_[iFilter]->test_chunk(chunk)) == false) {
				trPicker.Debug << "Chunk: " << chunk->get_pdb_id()
						<< " didn't pass a filter" << std::endl;
				break;
			}
		}
		if (!flag)
			continue;
		trPicker.Debug << "Processing sequence from vall: "
				<< chunk->get_sequence() << std::endl;

		CandidatesCollectorOP sink = candidates_sink_[frag_len];

	    	// split chunk into fragment candidates and score them
		for (Size j = 1; j <= chunk->size() - frag_len + 1; j++) {
			FragmentCandidateOP f = new FragmentCandidate(i_pos, j,
							chunk, frag_len);
			if (scores_->score_fragment(f, empty_map)) {
			    scores::FragmentScoreMapOP new_map = empty_map->clone();
			    std::pair<FragmentCandidateOP,
				scores::FragmentScoreMapOP> p(f, new_map);
			    sink->add(p);
			}
		} // All chunk locations done
		trPicker.Debug << chunk->get_pdb_id() << " done" << std::endl;
		trPicker.Debug << sink->count_candidates()<<" candidates stored at pos. "
		    <<i_pos<<", "<<sink->count_candidates()<<" in total"<< std::endl;
		trPicker.flush();
	} // all chunks done
}


void FragmentPicker::parse_command_line() {

	//## -------- setup query profile
	if (option[in::file::checkpoint].user()) {
		core::sequence::SequenceProfileOP q_prof(
				new core::sequence::SequenceProfile);
		trPicker.Info << "reading a query profile from: "
				<< option[in::file::checkpoint]() << std::endl;
		q_prof->read_from_checkpoint(option[in::file::checkpoint]());
		set_query_seq(q_prof);
		trPicker.Info << "picking fragments for query profile: "
				<< get_query_seq_string() << std::endl;
	}
	if (option[in::file::pssm].user()) {
		core::sequence::SequenceProfileOP q_prof(
				new core::sequence::SequenceProfile);
		trPicker.Info << "reading a query profile from: "
				<< option[in::file::pssm]()[1] << std::endl;
		q_prof->read_from_file(option[in::file::pssm]()[1], 1.0);
		set_query_seq(q_prof);
		trPicker.Info << "picking fragments for query profile: "
				<< get_query_seq_string() << std::endl;
	}

	//Fasta file trumps sequence profile as far as query_seq_string_ is concerned
	if (option[in::file::fasta].user()) {
		std::string q_seq = core::sequence::read_fasta_file(
				option[in::file::fasta]()[1])[1]->sequence();
		trPicker.Info << "reading a query sequence from: "
				<< option[in::file::fasta]()[1] << std::endl;

		set_query_seq(q_seq);
		trPicker.Info << "picking fragments for query sequence: "
				<< get_query_seq_string() << std::endl;
	}

	// --------- setup query secondary structure
	if (option[frags::ss_pred].user()) {
		utility::vector1<std::string> sec_str_input(option[frags::ss_pred]());
		read_ss_files(sec_str_input);
	}
	//---------- setup chunk filters
	if (option[frags::allowed_pdb].user()) {
		AllowPdbIdFilterOP allow = new AllowPdbIdFilter();
		allow->load_pdb_id_from_file(option[frags::allowed_pdb]());
		add_chunk_filter(allow);
		trPicker.Info << "Allowed PDB chains:\n";
		allow->show_pdb_ids(trPicker.Info);
	}

	if (option[frags::denied_pdb].user()) {
		DenyPdbIdFilterOP deny = new DenyPdbIdFilter();
		deny->load_pdb_id_from_file(option[frags::denied_pdb]());
		add_chunk_filter(deny);
		trPicker.Info << "Excluded PDB chains:\n";
		deny->show_pdb_ids(trPicker.Info);
	}

	// ##--------- setup VALL
	PROF_START( core::util::FRAGMENTPICKING_READ_VALL );
	if (option[in::file::vall].user()) {
		read_vall(option[in::file::vall]());
	}
	PROF_STOP( core::util::FRAGMENTPICKING_READ_VALL );

	// -------- fragment sizes
	if (option[frags::frag_sizes].user()) {
		utility::vector1<Size> frag_sizes_tmp = option[frags::frag_sizes]();
		for (Size i = 1; i <= frag_sizes_tmp.size(); ++i) {
		    if(frag_sizes_tmp[i] > max_frag_size_)
			    max_frag_size_ = frag_sizes_tmp[i];
		    frag_sizes_.push_back(frag_sizes_tmp[i]);
		}
	} else {
		max_frag_size_ = 9;
		frag_sizes_.push_back(3);
		frag_sizes_.push_back(9);
	}
	trPicker.Info << "Will pick fragments of size:";
	for (Size i = 1; i <= frag_sizes_.size(); ++i)
		trPicker.Info << frag_sizes_[i] << " ";
	trPicker.Info << std::endl;

	//---------- setup scoring scheme
	trPicker.Info << "Creating fragment scoring scheme" << std::endl;
	FragmentScoreManagerOP scoring = get_score_manager();
	if (option[frags::scoring::config].user()) {
		scoring->create_scores(option[frags::scoring::config](), this);
	}

	// -------- how many fragments and candidates
	n_frags_ = option[frags::n_frags]();
	n_candidates_ = option[frags::n_candidates]();

	if (n_frags_ > n_candidates_) n_candidates_ = n_frags_;

	trPicker.Info << "Picking " << n_frags_ << " fragments based on "
			<< n_candidates_ << " candidates" << std::endl;

	//-------- this comparator is used both for collecting and selecting fragments
	CompareTotalScore comparator(get_score_manager());

	//-------- collector & selector set up
	if (option[frags::quota_protocol].user() || option[frags::picking::quota_config_file].user()) {
	// This setup is a bit more complicated when user needs quota.
	// The quota version of this code was moved into a separate method
	    parse_quota_command_line();
	// This setup is a bit more complicated, when user needs quota. The quota version of this code was moved into a separate method
	} else {
    	    if (option[frags::keep_all_protocol].user()) {
		for (Size i = 1; i <= frag_sizes_.size(); ++i) {
		    CandidatesCollectorOP collector = new GrabAllCollector(size_of_query());
		    set_candidates_collector(frag_sizes_[i], collector);
		    trPicker.Info << "Collector for fragment size: " << frag_sizes_[i]
				<< " set to: GrabAllCollector" << std::endl;
		}
	    } else {
		for (Size i = 1; i <= frag_sizes_.size(); ++i) {
	    	    CandidatesCollectorOP collector = new BoundedCollector<
				CompareTotalScore> (size_of_query(), n_candidates_,
				comparator,get_score_manager()->count_components());
		    set_candidates_collector(frag_sizes_[i], collector);
		    trPicker.Info << "Collector for fragment size: " << frag_sizes_[i]
				<< " set to: BoundedCollector" << std::endl;
		}
	    }
	    //-------- Selecting fragments from candidates
	    if (option[frags::picking::selecting_rule].user()) {
		std::string type = option[frags::picking::selecting_rule]();
		if (type.compare("BestTotalScoreSelector")==0) {
			selector_ = new BestTotalScoreSelector(n_frags_, scores_);
			trPicker.Info << "Fragment selector: BestTotalScoreSelector"
					<< std::endl;
		} else {
		    utility_exit_with_message("[ERROR]: unknown fragment selecting rule: " + type + "!");
		}
	    } else {
		selector_ = new BestTotalScoreSelector(n_frags_, scores_);
		trPicker.Info << "Fragment selector: BestTotalScoreSelector" << std::endl;
    	    }
	}
    	// # ---------- output file prefix:
	if (option[out::file::frag_prefix].user()) {
		prefix_ = option[out::file::frag_prefix]();
	}

	if (option[frags::picking::query_pos].user()) {
	    set_picked_positions( option[frags::picking::query_pos]() );
	}

	show_scoring_methods(trPicker);
	trPicker << std::endl;
}

void FragmentPicker::set_up_ss_abego_quota() {

	std::string quota_config_file("UNKNOWN-QUOTA-CONFIG_FILE");
        if (option[frags::picking::quota_config_file].user())
            quota_config_file = option[frags::picking::quota_config_file]();
        quota::ABEGO_SS_Config q_config(quota_config_file);

	utility::vector1<Size> components;
	utility::vector1<Real> weights;
	utility::vector1<Real> scoring_weights = scores_->get_weights();
	for (Size i = 1; i <= scores_->count_components(); ++i) {
		ABEGO_SS_Score *s0 =
			dynamic_cast<ABEGO_SS_Score*> (scores_->get_component(i).get());
		if (s0 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s0->get_id()] );
		}
		ProfileScoreL1 *s1 =
			dynamic_cast<ProfileScoreL1*> (scores_->get_component(i).get());
		if (s1 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s1->get_id()] );
		}

		RamaScore *s2 =
			dynamic_cast<RamaScore*> (scores_->get_component(i).get());
		if (s2 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s2->get_id()] );
		}

		CSScore *s3 =
			dynamic_cast<CSScore*> (scores_->get_component(i).get());
		if (s3 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s3->get_id()] );
		}

		SecondarySimilarity *s4 =
			dynamic_cast<SecondarySimilarity*> (scores_->get_component(i).get());
		if (s4 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s4->get_id()] );
		}
	}

	trPicker.Debug<<"Scoring scheme for ABEGO_SS quota pool sorting is:";
	for(Size l=1;l<=weights.size();l++)
	    trPicker.Debug<<"\n\t"<<components[l]<<"\t"<<weights[l];
	trPicker.Debug<<std::endl;
	Size buffer_factor = 5;
	for(Size f=1;f<=frag_sizes_.size();f++) {
	    quota::QuotaCollectorOP collector = new quota::QuotaCollector( size_of_query(), frag_sizes_[f] );
	    set_candidates_collector(frag_sizes_[f],collector);
	    Size middle = frag_sizes_[f] / 2 + 1;
	    assert( size_of_query() == q_config.size() ); // Test if the abego-ss table has the same size as the query sequence
	    for(Size j=1;j<=size_of_query()-frag_sizes_[f]+1;j++) {
		trPicker.Trace<<"Creating "<<q_config.n_columns()<<" quota pools at pos "<<j<<std::endl;
		for(Size i=1;i<=q_config.n_columns();i++) {
		    Real f = q_config.probability(j+middle-1,i);
		    quota::QuotaPoolOP p = new quota::ABEGO_SS_Pool(n_candidates_,q_config.get_pool_name(i),
			    q_config.get_pool_bins((i)),components,weights,f,scores_->count_components(),buffer_factor);
		    collector->add_pool(j,p);
		}
	    }
	}
}

void FragmentPicker::set_up_quota_nnmake_style() {

	std::string quota_config_file("UNKNOWN-QUOTA-CONFIG_FILE");
        if (option[frags::picking::quota_config_file].user())
            quota_config_file = option[frags::picking::quota_config_file]();
        quota::QuotaConfig q_config(quota_config_file);

	utility::vector1<Size> components;
	utility::vector1<Real> weights;
	components.push_back( 0 );		// the free entry in the vector is for secondary structure score (only one for each pool)
	weights.push_back( 0.0 );		// score weight for SecondarySimilarity; will be changed later
	components.push_back( 0 );		// this free entry in the vector is for RamaScore
	weights.push_back( 0.0 );		// score weight for RamaScore; will be changed later
	utility::vector1<Real> scoring_weights = scores_->get_weights();
	for (Size i = 1; i <= scores_->count_components(); ++i) {
		ProfileScoreL1 *s1 =
			dynamic_cast<ProfileScoreL1*> (scores_->get_component(i).get());
		if (s1 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s1->get_id()] );
		}

/****** RamaScore is a special case, dispatched below
		RamaScore *s2 =
			dynamic_cast<RamaScore*> (scores_->get_component(i).get());
		if (s2 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s2->get_id()] );
		}
*********/
		CSScore *s3 =
			dynamic_cast<CSScore*> (scores_->get_component(i).get());
		if (s3 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s3->get_id()] );
		}
		ABEGO_SS_Score *s4 =
			dynamic_cast<ABEGO_SS_Score*> (scores_->get_component(i).get());
		if (s4 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s4->get_id()] );
		}
		TorsionBinSimilarity *s5 =
			dynamic_cast<TorsionBinSimilarity*> (scores_->get_component(i).get());
		if (s5 != 0) {
			components.push_back( i );
			weights.push_back( scoring_weights[s5->get_id()] );
		}
	}

	utility::vector1<core::conformation::SecondaryStructureOP> predictions;
	utility::vector1<Real> ss_weights;
	std::map<std::string, core::conformation::SecondaryStructureOP>::iterator it;
	Real weight = 1.0 / ((Real) query_ss_profile_.size());
	for ( it=query_ss_profile_.begin() ; it != query_ss_profile_.end(); it++ ) {
	    predictions.push_back((*it).second);
	    ss_weights.push_back(weight);
	}
//	core::conformation::SecondaryStructureOP avg_ss = new core::conformation::SecondaryStructure(predictions,ss_weights);

	for(Size f=1;f<=frag_sizes_.size();f++) {
	    quota::QuotaCollectorOP collector = new quota::QuotaCollector( size_of_query(), frag_sizes_[f] );
	    set_candidates_collector(frag_sizes_[f],collector);

// --------- This part puts RamaScore into quota scoring; each Rama is based on a certain SS prediction and this part of the code
// --------- dispatches each Rama into a proper pool
	    for (Size i = 1; i <= scores_->count_components(); ++i) {
		RamaScore *sr =
			dynamic_cast<RamaScore*> (scores_->get_component(i).get());
		if (sr != 0) {
			std::string & name = sr->get_prediction_name();
			if( ! q_config.is_valid_quota_pool_name( name ) )
				continue;
			components[2] = i;
			weights[2] = scoring_weights[sr->get_id()];
			trPicker.Warning<<"RamaScore with ID "<<sr->get_id()<<" named "<<name<<
				" has been attached to its quota pool with weight "<<weights[2]<<std::endl;
		}
	    }
// ---------- end of RamaScore dispatch

// Create secondary structure pools (if any)
	    for (Size i = 1; i <= scores_->count_components(); ++i) {
		SecondarySimilarity *s =
			dynamic_cast<SecondarySimilarity*> (scores_->get_component(i).get());

		//PartialSecondarySimilarity is a variant of SecondarySimilarity, this means they're not compatible
		//So what is it compatible with???
//		if (s == 0) {
//			PartialSecondarySimilarity *s =
//				dynamic_cast<PartialSecondarySimilarity*> (scores_->get_component(i).get());
//		}

		if (s != 0) {
			std::string & name = s->get_prediction_name();
			if( ! q_config.is_valid_quota_pool_name( name ) )
				continue;
			components[1] = i;
			weights[1] = scoring_weights[s->get_id()];
			Size size = (Size)(q_config.get_fraction( name ) * n_candidates_);
			if( size == 0 ) {
			      trPicker.Warning<<"Config file couldn't provide quota fraction for the pool named "
			              <<name<<". Skipping the pool"<<std::endl;
			      continue;
			}
			collector->attach_secondary_structure_pools(q_config.get_fraction( name ) ,
			    get_query_ss( name ),name,n_candidates_,components,weights,scores_->count_components());
//			    avg_ss,name,n_candidates_,components,weights,scores_->count_components());
		}
	    }
	}
}

void FragmentPicker::parse_quota_command_line() {

	set_up_quota_nnmake_style();

	if (option[frags::picking::query_pos].user()) {
	    set_picked_positions( option[frags::picking::query_pos]() );
	}
}

void FragmentPicker::read_vall( utility::vector1< std::string > const & fns ) {
	chunks_ = new VallProvider();
	chunks_->vallChunksFromLibraries(fns);
}

void FragmentPicker::read_vall( std::string const & fn ) {
	chunks_ = new VallProvider();
	chunks_->vallChunksFromLibrary(fn);
}

void FragmentPicker::set_picked_positions(Size from,Size to) {

    query_positions_.clear();
    for(Size i=from;i<=to;i++)
	query_positions_.push_back( i );
}

void FragmentPicker::set_picked_positions(utility::vector1<Size> q_positions) {

    query_positions_.clear();
    for(Size i=1;i<=q_positions.size();i++)
	query_positions_.push_back( q_positions[i] );

}


Size QuotaDebug::max_pools() {

    return tags_.size();
}

void QuotaDebug::write_summary() {

    trPicker<< "Quota report: difference between the total expected and total picked foreach pool"<<std::endl;
    trPicker<< "This table is for first "<<nFrags_<<" fragments"<<std::endl;
    trPicker<< "Negative value says that was picked more that expected."<<std::endl;
    trPicker<< this->str()<<std::endl;
    this->str("");
}

void QuotaDebug::log(Size frag_len,Size q_pos,utility::vector1<Real> data) {

    *this  << std::setw(4)<<q_pos<< std::setw(4)<<frag_len;
    for(Size i=1;i<=data.size();i++)
	if(data[i]<1000000)
	    *this  << std::setw(10)<<std::setprecision(3)<<data[i];
	else
	    *this  << std::setw(10)<<"  --- ";
    *this  << std::endl;
}

void QuotaDebug::setup_summary(quota::QuotaCollector* collector_) {

    Size last_tag = 0;
    for(Size i=1;i<=collector_->query_length();++i) {
	for(Size j=1;j<=collector_->count_pools(i);++j) {
	    if( tag_map_.find(collector_->get_pool(i,j)->get_pool_name())==tag_map_.end() ) {
		tags_.push_back(collector_->get_pool(i,j)->get_pool_name());
		last_tag++;
		tag_map_[collector_->get_pool(i,j)->get_pool_name()] = last_tag;
	    }
	}
    }

    *this <<"\n#len pos ";
    for(Size i=1;i<=tags_.size();i++) {
	*this << std::setw(10)<<tags_[i];
    }
    *this<<std::endl;
}

QuotaDebug FragmentPicker::log_25_(25);
QuotaDebug FragmentPicker::log_200_(200);

} // picking
} // fragment
} // core
