// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite && is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragment/picking/scores/FragmentScoreManager.hh
/// @brief
/// @author Dominik Gront (dgront@chem.uw.edu.pl)


// package headers
#include <core/fragment/picking/scores/FragmentScoringMethod.hh>
#include <core/fragment/picking/FragmentCandidate.hh>
#include <core/fragment/picking/FragmentPicker.hh>
#include <core/fragment/picking/VallResidue.hh>
#include <core/fragment/picking/scores/FragmentScoreManager.hh>

#include <core/fragment/picking/scores/SequenceIdentity.hh>
#include <core/fragment/picking/scores/BFactor.hh>
#include <core/fragment/picking/scores/DisulfideIdentity.hh>
#include <core/fragment/picking/scores/DisulfideDistance.hh>
#include <core/fragment/picking/scores/ProlinePhiScore.hh>
#include <core/fragment/picking/scores/RamaScore.hh>
#include <core/fragment/picking/scores/SecondaryIdentity.hh>
#include <core/fragment/picking/scores/SecondarySimilarity.hh>
#include <core/fragment/picking/scores/TalosSSSimilarity.hh>
#include <core/fragment/picking/scores/PartialSecondarySimilarity.hh>
#include <core/fragment/picking/scores/TorsionBinSimilarity.hh>
#include <core/fragment/picking/scores/ProfileScore.hh>
#include <core/fragment/picking/scores/ProfileScoreL1.hh>
#include <core/fragment/picking/scores/ProfileScoreBlosum62.hh>
#include <core/fragment/picking/scores/ProfileScoreDistWeight.hh>
#include <core/fragment/picking/scores/FragmentCrmsd.hh>
#include <core/fragment/picking/scores/MidPsiOut.hh>
#include <core/fragment/picking/scores/MidPhiOut.hh>
#include <core/fragment/picking/scores/PhiPsiRmsd.hh>
#include <core/fragment/picking/scores/JCoupling.hh>
#include <core/fragment/picking/scores/PhiPsiSquareWell.hh>
#include <core/fragment/picking/scores/AtomPairConstraintsScore.hh>
#include <core/fragment/picking/scores/DihedralConstraintsScore.hh>
#include <core/fragment/picking/scores/InterbondAngleScore.hh>
#include <core/fragment/picking/scores/CSScore.hh>
#include <core/fragment/picking/scores/AmbigCSScore.hh>
#include <core/fragment/picking/scores/GunnCostScore.hh>
#include <core/fragment/picking/scores/ABEGO_SS_Score.hh>
#include <core/fragment/picking/scores/FragmentScoreMap.hh>


#include <utility/io/izstream.hh>

// C++
#include <algorithm>
#include <map>
#include <utility>

namespace core {
namespace fragment {
namespace picking {
namespace scores {

/// @brief creates an empty score map
/// @detailed this is the recommended way to create FragmentScoreMap objects since FragmentScoreManager knows exactly
/// what is the correct size of the map i.e. how many scoring terms have been registered.
FragmentScoreMapOP FragmentScoreManager::create_empty_map() {

	    return new FragmentScoreMap(score_weights_.size());
}


bool sort_scores(FragmentScoringMethodOP elem1, FragmentScoringMethodOP elem2) {
	return elem1->get_priority() > elem2->get_priority();
}

void FragmentScoreManager::add_scoring_method(
		core::fragment::picking::scores::FragmentScoringMethodOP scoring_term,
		Real weight) {

	std::map<FragmentScoringMethodOP, Real> weights;
	scores_.push_back(scoring_term);
	score_weights_.push_back(weight);

	for (Size i = 1; i <= scores_.size(); i++) {
		std::pair<FragmentScoringMethodOP, Real> p(scores_[i],
				score_weights_[i]);
		weights.insert(p);
	}

	sort(scores_.begin(), scores_.end(), sort_scores);

	for (Size i = 1; i <= scores_.size(); i++) {
		scores_[i]->set_id(i);
		score_weights_[i] = weights.find(scores_[i])->second;
	}
	width_.insert(std::pair<FragmentScoringMethodOP, Size>(scoring_term,
			default_width_));
	precision_.insert(std::pair<FragmentScoringMethodOP, Size>(scoring_term,
			default_precision_));
}

Real FragmentScoreManager::total_score(FragmentScoreMapOP f) {

	if (!f->was_modified())
		return f->get_most_recent_total_score();

	Real total = 0;
	utility::vector1<Real> &s = f->get_score_components();

	assert (s.size() == score_weights_.size());
	for (Size i = 1; i <= score_weights_.size(); i++)
		total += score_weights_[i] * s[i];
	f->recent_total_ = total;
	f->was_modified_ = false;

	return total;
}

void FragmentScoreManager::do_caching(VallChunkOP chunk) {

	for (Size iScore = 1; iScore <= scores_.size(); ++iScore) {
		CachingScoringMethod* score =
				dynamic_cast<CachingScoringMethod*> (scores_[iScore].get());
		if (score != 0)
			score->do_caching(chunk);
	}
}

void FragmentScoreManager::clean_up() {

	for (Size iScore = 1; iScore <= scores_.size(); ++iScore) {
		CachingScoringMethod* score =
				dynamic_cast<CachingScoringMethod*> (scores_[iScore].get());
		if (score != 0)
			score->clean_up();
	}
}

void FragmentScoreManager::show_scoring_methods(std::ostream& out) {

	out
			<< "Fragment scoring scheme used:\n-----------------------------------------\n";
	out << "id    fragment score method name   weight\n";
	for (Size i = 1; i <= scores_.size(); i++)
		out << std::setw(3) << scores_[i]->get_id() << " " << std::setw(30)
				<< scores_[i]->get_score_name() << std::setw(5)
				<< score_weights_[i] << "\n";
	out << "-----------------------------------------\n" << std::endl;
}

void FragmentScoreManager::describe_fragments(utility::vector1<std::pair<
		FragmentCandidateOP, scores::FragmentScoreMapOP> > const& pairs,
		std::ostream& out) {

	using namespace ObjexxFCL::fmt;

	bool if_quota = false;
	if( pairs[1].second->get_quota_score() < 999.98 ) if_quota = true;
        out << "#" << RJ(10, "query_pos ");
        out << RJ(10, "vall_pos ");
	out << RJ(6, "pdbid");
        out << " c ss ";
	utility::vector1<Size> w(scores_.size()+4);
	for (Size i = 1; i <= scores_.size(); i++) {
		w[i] = width_.find(scores_[i])->second;
		out << " " << scores_[i]->get_score_name();
		if (scores_[i]->get_score_name().length() > w[i])
			w[i] = scores_[i]->get_score_name().length();
	}
	if(if_quota) {
	    w[scores_.size()+1] = 10; // 10 characters for the total quota score
	    w[scores_.size()+2] = 9; // 9 characters for the total score
	    w[scores_.size()+3] = 10; // 10 characters for the pool name
	    out << " QUOTA_TOT TOTAL POOL_NAME FRAG_ID"<<std::endl;
	} else {
	    w[scores_.size()+1] = 9; // 9 characters for the total score
	    out << "  TOTAL  FRAG_ID"<<std::endl;
	}
	for (Size iF = 1; iF <= pairs.size(); ++iF) {

		FragmentCandidateOP fr = pairs[iF].first;
		FragmentScoreMapOP sc = pairs[iF].second;
		VallResidueOP r = fr->get_residue(1);
		out << " " << I(10, fr->get_first_index_in_query());
//		out << " " << I(10, fr->get_first_index_in_vall());
		out << " " << I(10, r->resi());
		out << " " << RJ(5, fr->get_pdb_id());
		out << " " << fr->get_chain_id();
		out << " " << fr->get_middle_ss();

		for (Size i = 1; i <= scores_.size(); i++) {
			Size p = precision_.find(scores_[i])->second;
			out << " " << F(w[i], p, sc->get_score_components()[i]);
		}
		if(if_quota) {
		    out << " " << F(w[scores_.size()+1],TOTAL_PRECISION,sc->get_quota_score());
		    out << " " << F(w[scores_.size()+1],TOTAL_PRECISION,total_score(sc));
		    out << " " << std::setw(w[scores_.size()+3])<<fr->get_pool_name();
		} else
		    out << F(w[scores_.size()+1],TOTAL_PRECISION,total_score(sc));

		assert ( fr->key() > 0 );
		assert ( fr->key() < 4000000 ); // Put your Vall's size here

		out << I(10, fr->key() ) << std::endl;

	}
}

bool FragmentScoreManager::score_fragment(FragmentCandidateOP candidate,
		FragmentScoreMapOP empty_map) {

	for (Size iScore = 1; iScore <= scores_.size(); iScore++) {
		if (!scores_[iScore]->score(candidate, empty_map))
			return false;
	}

	return true;
}

bool FragmentScoreManager::score_zero_scores(FragmentCandidateOP candidate,
                FragmentScoreMapOP empty_map) {

        for (Size iScore = 1; iScore <= scores_.size(); iScore++) {
                if( fabs(score_weights_[iScore]) < 0.000001 )
                	scores_[iScore]->score(candidate, empty_map);
	}

	return true;
}


bool FragmentScoreManager::score_fragment_from_cache(FragmentCandidateOP candidate,
		FragmentScoreMapOP empty_map) {

	for (Size iScore = 1; iScore <= scores_.size(); iScore++) {
		if(( zeros_score_later_ ) && ( fabs(score_weights_[iScore]) < 0.000001 ))
		    continue;
		CachingScoringMethod *s =
			dynamic_cast<CachingScoringMethod*> (scores_[iScore].get());
		if (s != 0) {
			if (!s->cached_score(candidate, empty_map))
				return false;
		} else {
			if (!scores_[iScore]->score(candidate, empty_map))
				return false;
		}
	}
	return true;
}

void FragmentScoreManager::register_score_maker(
		MakeFragmentScoringMethodOP scoring_term_maker) {
	registered_makers_[scoring_term_maker->get_score_name()]
			= scoring_term_maker;
}

FragmentScoreManager::FragmentScoreManager() {

	default_precision_ = 2;
	default_width_ = 6;
	register_score_maker(new MakeSecondaryIdentity());
	register_score_maker(new MakeSequenceIdentity());
	register_score_maker(new MakeBFactor());
	register_score_maker(new MakeDisulfideIdentity());
	register_score_maker(new MakeDisulfideDistance());
	register_score_maker(new MakeProlinePhiScore());
	register_score_maker(new MakeRamaScore());
	register_score_maker(new MakeProfileScore());
	register_score_maker(new MakeProfileScoreL1());
	register_score_maker(new MakeProfileScoreBlosum62());
	register_score_maker(new MakeProfileScoreDistWeight());
	register_score_maker(new MakeSecondarySimilarity());
	register_score_maker(new MakeTalosSSSimilarity());
	register_score_maker(new MakePartialSecondarySimilarity());
	register_score_maker(new MakeTorsionBinSimilarity());
	register_score_maker(new MakeFragmentCrmsd());
	register_score_maker(new MakeMidPhiOut());
	register_score_maker(new MakeMidPsiOut());
	register_score_maker(new MakePhiPsiRmsd());
	register_score_maker(new MakePhiPsiSquareWell());
	register_score_maker(new MakeJCoupling());
	register_score_maker(new MakeAtomPairConstraintsScore());
	register_score_maker(new MakeDihedralConstraintsScore());
	register_score_maker(new MakeCSScore());
	register_score_maker(new MakeAmbigCSScore());
	register_score_maker(new MakeTorsionBinSimilarity());
	register_score_maker(new MakeInterbondAngleScore());
	register_score_maker(new MakeGunnCostScore());
	register_score_maker(new MakeABEGO_SS_Score());
	zeros_score_later_ = true;
}

void FragmentScoreManager::create_scoring_method(
		std::string const & score_name, Size priority, Real weight,
		Real lowest, bool use_lowest, FragmentPickerOP picker, std::string config_line) {

	std::map<std::string, MakeFragmentScoringMethodOP>::iterator m =
			registered_makers_.find(score_name);
	if (m == registered_makers_.end()) {
		utility_exit_with_message("[ERROR]: unknown score type " + score_name
				+ "!");
	}

	add_scoring_method(m->second->make(priority, lowest, use_lowest, picker, config_line),
			weight);
}

void FragmentScoreManager::create_scores(std::string const & file_name,
		FragmentPickerOP picker) {

	utility::io::izstream input(file_name.c_str());
	if (!input) {
		utility_exit_with_message("[ERROR]: can't open file " + file_name + "!");
	}

	std::string line;
	std::string score_name;
	Size priority;
	Real weight;
	std::string low_string;
	Real lowest;
	bool use_lowest;
	std::string remark("");

	while (getline(input, line)) {
		if (line[0] == '#')
			continue;

		std::istringstream line_stream(line);
		line_stream >> score_name >> priority >> weight >> low_string;

		if (low_string == "-") {
			lowest = 0.0;
			use_lowest = false;
		} else {
			std::istringstream low_stream(low_string);
			//lowest = static_cast<Real>(low_string);
			low_stream >> lowest;
			use_lowest = true;
		}

		remark = "";
		if (line_stream.good()) {
			line_stream >> remark;
		}

//		std::cout << line << std::endl;
//		std::cout << score_name << " " << priority << " " << weight << " " << low_string << " " << use_lowest << " " << remark << std::endl;

		create_scoring_method(score_name, priority, weight, lowest, use_lowest, picker, remark);
	}
}

} // scores
} // picking
} // fragment
} // core


