// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/grid/Pockets/PocketGrid.hh
/// @brief  core::grid::Pockets::PocketGrid header
/// @author David Johnson
/// @author Ragul Gowthaman

#ifndef INCLUDED_core_grid_Pockets_PocketGrid_HH
#define INCLUDED_core_grid_Pockets_PocketGrid_HH

#include <utility/pointer/ReferenceCount.hh>
#include <core/grid/Pockets/PocketGrid.fwd.hh>
#include <core/types.hh>
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.fwd.hh>
#include <core/scoring/constraints/XYZ_Func.fwd.hh>

#include <utility/vector1_bool.hh>
#include <list>
#include <string>


namespace core {
namespace grid {
namespace Pockets {

///@
class PCluster
{

	friend class PocketGrid;
	friend class NonPlaidFingerprint;
	friend class QuaternionNonPlaidFingerprint;

public:

  PCluster(core::Size x, core::Size y, core::Size z, core::Size step_);
  PCluster(const PCluster& old);
  ~PCluster() {};
  int size() const {return points_.size();};
  bool testNeighbor(PCluster & c2);
  bool isClose(PCluster const & c2) const;
  bool isTarget() const {return target;};

private:
  typedef struct {
    core::Size x;
    core::Size y;
    core::Size z;
  } Cxyz;
  int count_;
    std::list < Cxyz > points_;
  bool target;
  core::Size maxX, minX, maxY, minY, maxZ, minZ, step;

}; //PCluster

///@
class PClusterSet
{

	friend class PocketGrid;
	friend class NonPlaidFingerprint;
	friend class QuaternionNonPlaidFingerprint;

public:
  PClusterSet();
  PClusterSet& operator= (const PClusterSet& old);
  void clear();
  void add (core::Size x, core::Size y, core::Size z, core::Size step);
  void findClusters();
  core::Real getLargestClusterSize( core::Real const & stepSize, core::Real const & minClusterSize );
  core::Real getNetClusterSize( core::Real const & stepSize, core::Real const & minClusterSize );
  core::Size size() { return clusters_.size(); }

private:
  std::list <PCluster> clusters_;

}; //PClusterSet

///@
class PocketGrid : public utility::pointer::ReferenceCount
{

	//	friend class FingerprintBase;
	//	friend class QuaternionFingerprintBase;
	friend class NonPlaidFingerprint;
	friend class QuaternionNonPlaidFingerprint;

private:
  enum PtType {EMPTY, PROTEIN, TARGET, HSURFACE, PSURFACE, POCKET, PO_SURF, PO_BURIED,T_SURFACE, TP_POCKET, TP_SURF, TP_BURIED, PO_EDGE, TP_EDGE,PSP};
	std::vector < std::vector < std::vector <PtType> > > grid_;
	std::vector < std::vector < std::vector <core::Size> > > pockets_;
  //std::vector < conformation::Atom > atoms_;
  //core::Size numAtoms_;
  core::Size xdim_, ydim_, zdim_;
  core::Real xcorn_;
  core::Real ycorn_;
  core::Real zcorn_;
  core::Real stepSize_;
  core::Real maxLen_;
  std::string tag_;
  void init();
  void setup_default_options();
  void newSearch(core::Size thr1, core::Size thr2, core::Size max1, core::Size max2, bool psp=false, bool sps=false);
  void searchX(core::Size thr, bool psp=false);
  void searchY(core::Size thr, bool psp=false);
  void searchZ(core::Size thr, bool psp=false);
  void searchD1(core::Size thr, bool psp=false);
  void searchD2(core::Size thr, bool psp=false);
  void searchD3(core::Size thr, bool psp=false);
  void searchD4(core::Size thr, bool psp=false);
  bool fill(core::Size x, core::Size y,core::Size z);
  bool touchesSolvent(core::Size x, core::Size y,core::Size z) const;
  bool touchesSS(core::Size x, core::Size y,core::Size z) const;
  bool touchesPS(core::Size x, core::Size y,core::Size z) const;
  bool touchesSurface(core::Size x, core::Size y,core::Size z, bool polar, bool either=false) const;
  core::Size pdbno_;
  PClusterSet clusters_;

  Real size_x_;
  Real size_y_;
  Real size_z_;
  Real spacing_;
  Real limit_x_;
  Real limit_y_;
  Real limit_z_;
  bool restrictSize_;
  Real probe_rad_;
  Real surf_score_;
  Real surf_dist_;
  Real bur_score_;
  Real bur_dist_;
  bool side_chains_only_;
  bool markpsp_;
  bool marksps_;
  Real minPockSize_;
  Real maxPockSize_;

public:
  PocketGrid();
  PocketGrid(const PocketGrid& gr);
  PocketGrid& operator= (const PocketGrid& gr);

  PocketGrid( conformation::Residue const & central_rsd );
  PocketGrid( std::vector< conformation::ResidueOP > const & central_rsd );

  PocketGrid( core::Real const & xc, core::Real const & yc, core::Real const & zc, core::Size x, core::Size y, core::Size z, core::Real const & stepSize=1, bool psp=false, bool sps=false);

  PocketGrid( core::Real const & xc, core::Real const & yc, core::Real const & zc, core::Size x, core::Real const & stepSize=1, bool psp=false, bool sps=false);
  PocketGrid( core::Vector const & center, core::Size x, core::Real const & stepSize=1, bool psp=false, bool sps=false);
  PocketGrid( core::Vector const & center, core::Size const & x, core::Size const & y, core::Size const & z, core::Real const & stepSize=1, bool psp=false, bool sps=false);

  ~PocketGrid() {};

  void initialize( core::Real const & xc, core::Real const & yc, core::Real const & zc, core::Size x, core::Size y, core::Size z, core::Real const & stepSize=1, bool psp=false, bool sps=false);

  void initialize( core::Real const & xc, core::Real const & yc, core::Real const & zc, core::Size x, core::Real const & stepSize=1, bool psp=false, bool sps=false);
  void initialize( core::Vector const & center, core::Size x, core::Real const & stepSize=1, bool psp=false, bool sps=false);
  void initialize( core::Vector const & center, core::Size const & x, core::Size const & y, core::Size const & z, core::Real const & stepSize=1, bool psp=false, bool sps=false);
  void initialize( conformation::Residue const & central_rsd, core::Size const & x, core::Size const & y, core::Size const & z, core::Real const & stepSize=1, bool psp=false, bool sps=false);
  void initialize( std::vector< conformation::ResidueOP > const & central_rsd, core::Size const & x, core::Size const & y, core::Size const & z, core::Real const & stepSize=1, bool psp=false, bool sps=false);
  void clear();

  void recenter(core::Real const & xc, core::Real const & yc, core::Real const & zc);
	void recenter( conformation::Residue const & central_rsd );
	void recenter( std::vector< conformation::ResidueOP > const & central_rsds );
  void recenter( core::Vector const & center);

  void findPockets(core::Size thr, core::Size max);
  void findPSP(core::Size thr, core::Size max);
  void findSPS(core::Size thr, core::Size max);

  void mark (core::Real x, core::Real y, core::Real z, core::Real const & vdWd, core::Real const & buffer, bool polar=false, bool targetResi=false);
  void mark (core::Vector const & center, core::Real const & vdWd, core::Real const & buffer, bool polar=false, bool targetResi=false);

  void clearSmallPockets(core::Size minsize);
  void findClusters();

  void dumpGridToFile();
  void fillTargetPockets();
  void print();
  core::Real targetPocketVolume(core::Real const & surf_sc, core::Real const & bur_sc) const ;
  core::Real largestTargetPocketVolume();
  void markPocketDepth(core::Real const & surf_d, core::Real const & bur_d);
  void markEdgeDepth(core::Real const & surf_d, core::Real const & bur_d);
  core::Real netTargetPocketVolume();
  void markDepth(core::Size x, core::Size y, core::Size z, core::Real const & surf_d, core::Real const & bur_d);
  bool markOneEdgeDepth(core::Size x, core::Size y, core::Size z, core::Real const & surf_d, core::Real const & bur_d, bool isTarget);
  core::Real targetPocketSolventSurface() const ;
  core::Real targetPocketProteinSurface() const ;
  core::Real targetPocketHydrophobicProteinSurface() const ;
  core::Real targetPocketPolarProteinSurface() const ;
  core::Real targetPocketHeuristicScore() const ;
  bool isTooSmall() const;
  core::Vector whatIsTooSmall() const;

	bool autoexpanding_pocket_eval( conformation::Residue const & central_rsd, scoring::constraints::XYZ_Func const & xyz_func, Size const total_residues );
	//bool autoexpanding_pocket_eval( std::vector< conformation::Residue > const & central_rsd, scoring::constraints::XYZ_Func const & xyz_func, Size const total_residues );
	bool autoexpanding_pocket_eval( conformation::Residue const & central_rsd, core::pose::Pose const & inPose);
	bool autoexpanding_pocket_eval( std::vector< conformation::ResidueOP > const & central_rsd, core::pose::Pose const & inPose);

  static std::vector< conformation::ResidueOP > getRelaxResidues( core::pose::Pose const & input_pose, std::string const & resids );

}; //class grid

}//Pockets
}//grid
}//core

#endif
