// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
///
/// @brief
/// @author

// Unit headers
#include <core/io/pdb/pose_io.hh>

#include <core/types.hh>

#include <core/pose/Pose.hh>
#include <core/pose/util.hh>

#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/AA.hh>
#include <core/chemical/ResidueType.hh>
// AUTO-REMOVED #include <core/chemical/VariantType.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/Conformation.hh>
// AUTO-REMOVED #include <core/conformation/ResidueFactory.hh>
// AUTO-REMOVED #include <core/chemical/residue_io.hh>
#include <core/kinematics/FoldTree.hh>

#include <core/scoring/Energies.hh>

// AUTO-REMOVED #include <core/id/AtomID_Map.Pose.hh>
// AUTO-REMOVED #include <core/id/AtomID_Mask.hh>

#include <ObjexxFCL/format.hh>
// AUTO-REMOVED #include <ObjexxFCL/ObjexxFCL.hh>


#include <core/io/pdb/pdb_dynamic_reader.hh>
#include <core/io/pdb/file_data.hh>

#include <core/util/Tracer.hh>
#include <core/options/option.hh>

// option key includes

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/inout.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>

// Utility headers
#include <utility/exit.hh>
#include <utility/string_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

//Auto Headers
#include <core/chemical/AtomTypeSet.hh>
#include <core/id/AtomID_Map.hh>
#include <core/kinematics/Jump.hh>

//Auto using namespaces
namespace ObjexxFCL { namespace fmt { } } using namespace ObjexxFCL::fmt; // AUTO USING NS
//Auto using namespaces end




//#include <fstream>

/// A temporary copy of the pose_from_pdb code from the demo directory.
/// Will be phased out in favor of file_data routines soon.
///

namespace core {
namespace io {
namespace pdb {


/// special Tracer instance acting as special param for all traced_dump_pdb functions
core::util::Tracer TR_dump_pdb_dummy( "core.io.pdb.pose_io.dump_pdb_dummy" );

core::util::Tracer TR("core.io.pose_io");

using utility::vector1;

/// @brief Writes
void
write_additional_pdb_data(
	std::ostream & out,
	pose::Pose const & pose,
	FileData const &,
	bool write_fold_tree
)
{
	if ( write_fold_tree | options::option[ options::OptionKeys::inout::fold_tree_io ].user() ) {
		out << "REMARK " << pose.fold_tree();
	}
	if ( options::option[ options::OptionKeys::out::file::pdb_parents]() ) {
		std::string value;
		bool has_parents = core::pose::get_comment( pose, "parents", value );
		if( has_parents ){
			out << "REMARK PARENT    " << value.substr(0,5) << std::endl;
		}
	}
}

//
void
read_additional_pdb_data(
	std::string const & s,
	pose::Pose & pose,
	FileData const &,
	bool read_fold_tree
)
{

	if ( (!read_fold_tree) && (!options::option[ options::OptionKeys::inout::fold_tree_io ].user()) ) return;

	// split on newlines
	utility::vector1< std::string > lines;
	Size start=0, i=0;
	while(start < s.size()) {
		if( s[i] == '\n' || s[i] == '\r' /* || i==s.size()-1 */) {
			lines.push_back( std::string(s.begin()+start, s.begin()+i) );
			start = i+1;
		}
		i++;
		if( i == s.size() ) {
			lines.push_back( std::string(s.begin()+start, s.begin()+i) );
			break;
		}
	}

	//
	for ( Size i=1; i<= lines.size(); ++i ) {
		std::string const & line( lines[i] );

		// look for fold_tree info
		if ( line.size() >= 16 && line.substr(0,16) == "REMARK FOLD_TREE" ) {
			std::istringstream l( line );
			std::string tag;
			kinematics::FoldTree f;
			l >> tag >> f;
			if ( !l.fail() && Size(f.nres()) == pose.total_residue() ) {
				TR << "setting foldtree from pdb file: " << f << std::endl;
				pose.fold_tree( f );
			} else {
				TR.Fatal << "pose_io:: foldtree io failure: " << line << ' ' << pose.total_residue()
					<< ' ' << f << std::endl;
				utility_exit();
			}
		}
	}

}

void
pose_from_pdb(
	pose::Pose & pose,
	chemical::ResidueTypeSet const & residue_set,
	std::string const & filenames_string,
	bool read_fold_tree
)
{
	std::vector<std::string> filenames= utility::split(filenames_string);


	std::string res;

	std::vector<std::string>::const_iterator begin= filenames.begin();
	for(; begin != filenames.end(); ++begin){
		utility::io::izstream file( *begin );
		if (!file) {
			TR.Error << "PDB File:" << *begin << " not found!" << std::endl;
			utility_exit_with_message( "Cannot open PDB file \"" + *begin + "\"" );
		} else {
			TR.Debug << "read file: " << *begin << std::endl;
		}
		utility::slurp( file, res );
	}

	//fpd If the conformation is not of type core::Conformation, reset it
	if ( !pose.conformation().same_type_as_me( conformation::Conformation(), true ) ) {
		pose.set_new_conformation( new conformation::Conformation() );
	}

	core::io::pdb::FileData fd = core::io::pdb::PDB_DReader::createFileData(res);
	if ( fd.filename == "" ) {
		fd.filename = utility::join(filenames, "_");
	}
	fd.build_pose(pose, residue_set);

	// set secondary structure for centroid PDBs
	if ( residue_set.name() == core::chemical::CENTROID ) {
		core::pose::set_ss_from_phipsi( pose );
	}

	// check for foldtree info
	read_additional_pdb_data( res, pose, fd, read_fold_tree);
}

void
pose_from_pdb(
	core::pose::Pose & pose,
	std::string const & filename,
	bool read_fold_tree
) {
	using core::options::option;
	using namespace chemical;
	using namespace core::options::OptionKeys;

	ResidueTypeSetCAP residue_set(
		option[ in::file::centroid_input ].value() ?
		ChemicalManager::get_instance()->residue_type_set( CENTROID ) :
		ChemicalManager::get_instance()->residue_type_set( FA_STANDARD )
	);

	pose_from_pdb( pose, *residue_set, filename, read_fold_tree );
}

utility::vector1< core::pose::PoseOP > poseOPs_from_pdbs(
	utility::vector1< std::string > const & filenames,
	bool read_fold_tree
) {
	using namespace chemical;
	ResidueTypeSetCAP residue_set
		( ChemicalManager::get_instance()->residue_type_set( FA_STANDARD ) );

	return poseOPs_from_pdbs( *residue_set, filenames, read_fold_tree );
}

utility::vector1< core::pose::Pose > poses_from_pdbs(
	utility::vector1< std::string > const & filenames,
	bool read_fold_tree
) {
	using namespace chemical;
	ResidueTypeSetCAP residue_set
		( ChemicalManager::get_instance()->residue_type_set( FA_STANDARD ) );

	return poses_from_pdbs( *residue_set, filenames, read_fold_tree );
}

utility::vector1< core::pose::Pose > poses_from_pdbs(
	chemical::ResidueTypeSet const & residue_set,
	utility::vector1< std::string > const & filenames,
	bool read_fold_tree
) {
	using namespace chemical;

	using std::string;
	using utility::vector1;
	using core::pose::Pose;

	vector1< Pose > poses;
	typedef vector1< string >::const_iterator vec_it;
	for ( vec_it it = filenames.begin(), end = filenames.end(); it != end; ++it ) {
		Pose pose;
		pose_from_pdb( pose, residue_set, *it, read_fold_tree );
		poses.push_back( pose );
	}

	return poses;
}


utility::vector1< core::pose::PoseOP > poseOPs_from_pdbs(
	chemical::ResidueTypeSet const & residue_set,
	utility::vector1< std::string > const & filenames,
	bool read_fold_tree
) {
	using namespace chemical;

	using std::string;
	using utility::vector1;
	using core::pose::Pose;

	vector1< pose::PoseOP > poses;
	typedef vector1< string >::const_iterator vec_it;
	for ( vec_it it = filenames.begin(), end = filenames.end(); it != end; ++it ) {
		pose::PoseOP pose = new pose::Pose;
		pose_from_pdb( *pose, residue_set, *it, read_fold_tree );
		poses.push_back( pose );
	}

	return poses;
}


void
pose_from_pdb(
	utility::vector1< pose::Pose > & poses,
	std::string const & filename,
  bool read_fold_tree
)
{
	using namespace chemical;
	ResidueTypeSetCAP residue_set(
		ChemicalManager::get_instance()->residue_type_set( FA_STANDARD )
	);
	pose_from_pdb( poses, *residue_set, filename, read_fold_tree );
}


void
pose_from_pdb(
	utility::vector1< pose::Pose > & poses,
	chemical::ResidueTypeSet const & residue_set,
	std::string const & filename,
	bool read_fold_tree
)
{
	// Size fsize;
	pose::Pose pose;
	std::string all_lines, sub_lines;

	utility::io::izstream file( filename );
	if (!file) {
		TR.Error << "File:" << filename << " not found!" << std::endl;
		utility_exit_with_message( "Cannot open file " + filename );
	} else {
		TR.Debug << "read file: " << filename << std::endl;
	}

	utility::slurp( file, all_lines );
	// Hacky way to make sure that we find all models. I blame society.
	all_lines = "\n" + all_lines;

	Size pos1 = 0;
	Size pos2 = 0;
	Size n_models = 0;

	// count the number of poses will be reading
	while( pos1 != std::string::npos )
	{
		pos1 = all_lines.find( "\nMODEL ", pos1 );
		if ( pos1 != std::string::npos )
		{
			++n_models;
			++pos1;
		}
	}

	TR.Debug << "Reading " << n_models << " poses." << std::endl;
	// make space for all of our poses
	if ( n_models == 0 ) n_models = 1;
	poses.reserve( poses.size() + n_models );

	pos1 = 0;

	pos1 = all_lines.find( "\nMODEL ", pos1 );
	if ( pos1 != std::string::npos ) {
		pos2 = 0;
		while( pos2 != std::string::npos ) {
			// set pos1 == "M" in MODEL
			++pos1;

			// pos2 = position of newline character, start somewhere after pos1
			pos2 = all_lines.find( "\nMODEL ", pos1);
			sub_lines = all_lines.substr( pos1, pos2-pos1 ) ;
			pos1 = pos2;

			core::io::pdb::FileData fd = core::io::pdb::PDB_DReader::createFileData( sub_lines );
			fd.filename = filename;
			fd.build_pose(pose, residue_set);

			// check for foldtree info
			read_additional_pdb_data( sub_lines, pose, fd, read_fold_tree);
			poses.push_back( pose );
		}
	} else {
		core::io::pdb::FileData fd = core::io::pdb::PDB_DReader::createFileData( all_lines );
		if ( fd.filename == "" ) {
			fd.filename = filename;
		}
		fd.build_pose(pose, residue_set);
		// check for foldtree info
		read_additional_pdb_data( all_lines, pose, fd, read_fold_tree);
		poses.push_back( pose );
	}
}

void
pose_from_pdbstring(
	pose::Pose & pose,
	std::string const & pdbcontents,
	std::string const & filename
)
{
	FileData fd = PDB_DReader::createFileData( pdbcontents );
	fd.filename = filename;
	chemical::ResidueTypeSetCAP residue_set
		( chemical::ChemicalManager::get_instance()->residue_type_set( chemical::FA_STANDARD ) );
	fd.build_pose( pose, *residue_set);

}


void
pose_from_pose(
		pose::Pose & new_pose,
		pose::Pose const & old_pose,
		utility::vector1< core::Size > const & residue_indices
){
	using namespace chemical;
	ResidueTypeSetCAP residue_set(
		ChemicalManager::get_instance()->residue_type_set( FA_STANDARD )
	);
	pose_from_pose( new_pose, old_pose, *residue_set,  residue_indices);
}

void
pose_from_pose(
		pose::Pose & new_pose,
		pose::Pose const & old_pose,
		chemical::ResidueTypeSet const & residue_set,
		utility::vector1< core::Size > const & residue_indices
){
	FileData fd;
	std::string data;
	fd.init_from_pose( old_pose, residue_indices );
	fd.build_pose(new_pose, residue_set);
}

void
centroid_pose_from_pdb(
	pose::Pose & pose,
	std::string const & filename,
	bool read_fold_tree
)
{
	using namespace chemical;
	ResidueTypeSetCAP residue_set
		( ChemicalManager::get_instance()->residue_type_set( CENTROID ) );

	pose_from_pdb( pose, *residue_set, filename, read_fold_tree);
}


void
dump_pdb(
	pose::Pose const & pose,
	std::ostream & out,
	id::AtomID_Mask const & mask,
	std::string const & tag
) {
	Size const nres( pose.total_residue() );

	Size number(0);

	static std::string const chains( " ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890" );

	out << "MODEL     " << tag << "\n";
	for ( Size i=1; i<= nres; ++i ) {
		conformation::Residue const & rsd( pose.residue(i) );
		for ( Size j=1; j<= rsd.natoms(); ++j ) {
			conformation::Atom const & atom( rsd.atom(j) );

 			if ( ! mask[ id::AtomID( j,i ) ] ) continue;

			//skip outputing virtual atom unless specified
			if ( !options::option[ options::OptionKeys::out::file::output_virtual ]() &&
				(int)atom.type() == (int)rsd.atom_type_set().n_atomtypes() ) continue;

			++number;
			runtime_assert( rsd.chain() < chains.size() ); // silly restriction
			char const chain( chains[ rsd.chain() ] );
			out << "ATOM  " << I(5,number) << ' ' << rsd.atom_name(j) << ' ' <<
				rsd.name3() << ' ' << chain << I(4,rsd.seqpos() ) << "    " <<
				F(8,3,atom.xyz()(1)) <<
				F(8,3,atom.xyz()(2)) <<
				F(8,3,atom.xyz()(3)) <<
				F(6,2,1.0) << F(6,2,1.0) << '\n';
		}
	}
	out << "ENDMDL\n";
}


///////////////////////////////////////////////////////////////////////////////
void
dump_bfactor_pdb(
	pose::Pose const & pose,
	id::AtomID_Map< Real > const & bfactor,
	std::ostream & out,
	std::string const & tag
)
{
	int const nres( pose.total_residue() );

	int number(0);

	static std::string const chains( " ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890" );

	if(tag!="NO_MODEL_LINE_IN_OUTPUT") out << "MODEL     " << tag << "\n";
	for ( int i=1; i<= nres; ++i ) {
		conformation::Residue const & rsd( pose.residue(i) );
		for ( Size j=1; j<= rsd.natoms(); ++j ) {
			conformation::Atom const & atom( rsd.atom(j) );

			++number;
			runtime_assert( rsd.chain() < chains.size() ); // silly restriction
			char const chain( chains[ rsd.chain() ] );
			out << "ATOM  " << I(5,number) << ' ' << rsd.atom_name(j) << ' ' <<
				rsd.name3() << ' ' << chain << I(4,rsd.seqpos() ) << "    " <<
				F(8,3,atom.xyz()(1)) <<
				F(8,3,atom.xyz()(2)) <<
				F(8,3,atom.xyz()(3)) <<
				F(6,2,1.0) << F(6,2, bfactor[ id::AtomID(j,i) ] ) << '\n';
		} // 	for ( int i=1; i<= nres; ++i )
	} // 	for ( Size j=1; j<= rsd.natoms(); ++j )
	if(tag!="NO_MODEL_LINE_IN_OUTPUT") out << "ENDMDL\n";
} // dump_bfactor_pdb


///////////////////////////////////////////////////////////////////////////////
void
dump_pdb_residue(
	conformation::Residue const & rsd,
	Size & atom_number,
	std::ostream & out
) {

	static std::string const chains( " ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890" );

	for ( Size j=1; j<= rsd.natoms(); ++j ) {
		conformation::Atom const & atom( rsd.atom(j) );

		++atom_number;
		runtime_assert( rsd.chain() < chains.size() ); // silly restriction
		char const chain( chains[ rsd.chain() ] );
		out << "ATOM  " << I(5,atom_number) << ' ' << rsd.atom_name(j) << ' ' <<
			rsd.name3() << ' ' << chain << I(4,rsd.seqpos() ) << "    " <<
			F(8,3,atom.xyz()(1)) <<
			F(8,3,atom.xyz()(2)) <<
			F(8,3,atom.xyz()(3)) <<
			F(6,2,1.0) << F(6,2,0.0) << '\n';
	}
}



///////////////////////////////////////////////////////////////////////////////
void
dump_pdb(
	pose::Pose const & pose,
	std::ostream & out,
	std::string const & tag
) {
	pose.dump_pdb(out, tag);
}


void
dump_pdb(
	pose::Pose const & pose,
	std::string const & filename,
	std::string const & tag
) {
	pose.dump_pdb(filename, tag);
	/*
	std::ofstream out( filename.c_str() );
	dump_pdb( pose, out );
	out.close();
	*/
}


/// @brief dump_pdb depending on visibility of tracer
/// @param[in] tr   output performed if tracer is visible or if passed dummy
///  tracer core::io::pdb::TR_dump_pdb_dummy
void
traced_dump_pdb(
	core::util::Tracer const & tr,
	pose::Pose const & pose,
	std::ostream & out,
	std::string const & tag
)
{
	if ( ( &tr ) != ( &core::io::pdb::TR_dump_pdb_dummy ) ) {
		if ( !tr.visible() ) {
			return;
		}
	}

	pose.dump_pdb( out, tag );
}


/// @brief dump_pdb depending on visibility of tracer
/// @param[in] tr   output performed if tracer is visible or if passed dummy
///   tracer core::io::pdb::TR_dump_pdb_dummy
void
traced_dump_pdb(
	core::util::Tracer const & tr,
	pose::Pose const & pose,
	std::string const & filename,
	std::string const & tag
)
{
	if ( ( &tr ) != ( &core::io::pdb::TR_dump_pdb_dummy ) ) {
		if ( !tr.visible() ) {
			return;
		}
	}

	pose.dump_pdb( filename, tag );
}


// the old way: build termini variants after appending non-terminus residues onto the pose in from the pdb...

// void
// pose_from_pdb(
// 	pose::Pose & pose,
// 	chemical::ResidueTypeSet const & residue_set,
// 	std::string const & filename
// )
// {
// 	//using namespace core;
// 	using namespace conformation;

// 	typedef numeric::xyzVector< Real > Vector;

// 	// reset current data
// 	pose.clear();

// 	Coords coords;
// 	Strings resids, sequence, pose_resids;

// 	read_pdb( filename, resids, sequence, coords );

// 	int const nres_pdb( resids.size() );

// 	char prev_chain('?');


// 	for ( int i=1; i<= nres_pdb; ++i ) {
// 		std::string const pdb_name( sequence[i] );
// 		std::string const resid( resids[i] );
// 		runtime_assert( resid.size() == 6 );
// 		char const chain( resid[5] );

// 		ResidueCoords const & xyz( coords.find( resid )->second );

// 		ResidueTypeCAPs const & rsd_type_list( residue_set.name3_map( pdb_name ) );
// 		if ( rsd_type_list.empty() ) {
// 			std::cout << "Unrecognized aa: " << pdb_name << '\n';
// 			continue;
// 		}

// 		// look for perfect match:
// 		bool matched( false );
// 		for ( Size j=1; j<= rsd_type_list.size(); ++j ) {
// 			ResidueType const & rsd_type( *(rsd_type_list[j]) );

// 			if ( rsd_type.is_terminus() ) continue; // no termini at this stage

// 			int rsd_missing(0), xyz_missing(0);

// 			for ( Size k=1; k<= rsd_type.natoms(); ++k ) {
// 				if ( xyz.count( rsd_type.atom_name(k) ) == 0 ) ++xyz_missing;
// 			}

// 			for ( ResidueCoords::const_iterator iter=xyz.begin(), iter_end=xyz.end(); iter!= iter_end; ++iter ) {
// 				if ( !rsd_type.has( iter->first ) ) ++rsd_missing;
// 			}

// 			if ( rsd_missing ) continue;

// 			std::cout << "match: " << i << ' ' << rsd_type.name() << ' ' << xyz_missing << std::endl;

// 			matched = true;

// 			// found a perfect match! fill in the coords
// 			ResidueOP new_rsd( ResidueFactory::create_residue( rsd_type ) );

// 			for ( ResidueCoords::const_iterator iter=xyz.begin(), iter_end=xyz.end(); iter!= iter_end; ++iter ) {
// 				new_rsd->atom( iter->first ).xyz( iter->second );
// 			}


// 			if ( chain != prev_chain && pose.total_residue() ) {
// 				pose.append_residue( new_rsd, true, pose.total_residue() );
// 			} else {
// 				pose.append_residue( new_rsd );
// 			}
// 			pose_resids.push_back( resid );

// 			break;
// 		} // j=1,rsd_type_list.size()


// 		if ( !matched ) {
// 			// unforgiving for testing purposes
// 			std::cout << "Unrecognized residue: " << pdb_name << std::endl;
// 			utility_exit();
// 		}

// 		// handle termini
// 		if ( chain != prev_chain ) {
// 			prev_chain = chain;
// 			int const seqpos( pose.total_residue() );
// 			if ( seqpos > 1 ) {
// 				pose.conformation().insert_chain_ending( seqpos - 1 );
// 				// make previous residue a terminus
// 				make_upper_terminus( pose, residue_set, seqpos-1 );
// 			}
// 			make_lower_terminus( pose, residue_set, seqpos );

// 		}

// 	} // i=1,nres_pdb

// 	make_upper_terminus( pose, residue_set, pose.total_residue() );

// 	// now handle missing atoms
// 	id::AtomID_Mask missing( false );

// 	id::initialize( missing, pose ); // dimension the missing-atom mask

// 	for ( Size i=1; i<= pose.total_residue(); ++i ) {
// 		ResidueCoords const & xyz( coords.find( pose_resids[i] )->second );
// 		Residue const & rsd( pose.residue(i) );
// 		for ( Size j=1; j<= rsd.natoms(); ++j ) {
// 			if ( xyz.count( rsd.atom_name(j) ) == 0 ) missing[ id::AtomID( j, i ) ] = true;
// 		}
// 	}

// 	pose.conformation().fill_missing_atoms( missing );
// }

// @brief Write energies information into an output stream (e.g. the tail of a pdb file)
void extract_scores(
	core::pose::Pose const & pose,
	utility::io::ozstream & out
)
{
// 	if(!pose.energies().energies_updated()){
// 		out << "Pose's energies were not current, PDBJobOutputter will not force update" << std::endl;
// 		return;
// 	}

	//This is shamelessly refactored from the older JobDistributor; Jobdistributors.hh:1018; SVN 25940
	// APL: Moving this job-independent code into a central location.
	// Which score terms to use
	core::scoring::EnergyMap weights = pose.energies().weights();
	typedef utility::vector1<core::scoring::ScoreType> ScoreTypeVec;
	ScoreTypeVec score_types;
	for(int i = 1; i <= core::scoring::n_score_types; ++i) {
		core::scoring::ScoreType ii = core::scoring::ScoreType(i);
		if ( weights[ii] != 0 ) score_types.push_back(ii);
	}
	// This version is formatted for easy parsing by R, Excel, etc.
	out << "# All scores below are weighted scores, not raw scores.\n";
	out << "#BEGIN_POSE_ENERGIES_TABLE " << out.filename() << "\n";
	out << "label";
	for(ScoreTypeVec::iterator ii = score_types.begin(), end_ii = score_types.end(); ii != end_ii; ++ii)
		out << " " << name_from_score_type(*ii);
	out << " total\n";
	out << "weights";
	for(ScoreTypeVec::iterator ii = score_types.begin(), end_ii = score_types.end(); ii != end_ii; ++ii)
		out << " " << weights[*ii];
	out << " NA\n";
	out << "pose";
	core::Real pose_total = 0.0;
	if ( pose.energies().energies_updated() ) {
		for(ScoreTypeVec::iterator ii = score_types.begin(), end_ii = score_types.end(); ii != end_ii; ++ii) {
			core::Real score = (weights[*ii] * pose.energies().total_energies()[ *ii ]);
			out << " " << score;
			pose_total += score;
		}
		out << " " << pose_total << "\n";
		for(core::Size j = 1, end_j = pose.total_residue(); j <= end_j; ++j) {
			core::Real rsd_total = 0.0;
			out << pose.residue(j).name() << "_" << j;
			for(ScoreTypeVec::iterator ii = score_types.begin(), end_ii = score_types.end(); ii != end_ii; ++ii) {
				core::Real score = (weights[*ii] * pose.energies().residue_total_energies(j)[ *ii ]);
				out << " " << score;
				rsd_total += score;
			}
			out << " " << rsd_total << "\n";
		}
	}
	out << "#END_POSE_ENERGIES_TABLE " << out.filename() << "\n";
}


} // namespace pdb
} // namespace io
} // namespace core
