// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/kinematics/util.hh
/// @brief  Kinematics utility functions
/// @author Phil Bradley


#ifndef INCLUDED_core_kinematics_util_HH
#define INCLUDED_core_kinematics_util_HH


// Package headers
#include <core/kinematics/tree/Atom.fwd.hh>
#include <core/kinematics/Edge.fwd.hh>
#include <core/kinematics/FoldTree.fwd.hh>
#include <core/kinematics/AtomTree.fwd.hh>
#include <core/kinematics/AtomPointer.fwd.hh>
#include <core/id/AtomID.fwd.hh>
// AUTO-REMOVED #include <core/id/AtomID_Map.hh> // FWD doesnt get AtomMap for slice
#include <core/id/DOF_ID_Map.fwd.hh>
#include <core/id/DOF_ID_Mask.fwd.hh>
#include <core/id/TorsionID.fwd.hh>
#include <core/kinematics/MoveMap.fwd.hh>
// AUTO-REMOVED #include <core/kinematics/types.hh>

// Rosetta headers
#include <core/pose/Pose.fwd.hh>
#include <core/conformation/Residue.fwd.hh>

// ObjexxFCL headers
// AUTO-REMOVED #include <ObjexxFCL/FArray.all.fwd.hh>
#include <ObjexxFCL/FArray1D.fwd.hh>

// Numeric headers
// AUTO-REMOVED #include <numeric/xyzMatrix.fwd.hh>

//Auto Headers
#include <core/types.hh>
#include <ostream>




namespace core {
namespace kinematics {

void
build_tree(
	FoldTree const & fold_tree,
	conformation::ResidueCAPs const & residues,
	AtomPointer2D & atom_pointer
);

/// @brief build a sub atom-tree for a jump edge and attach it to main atom-tree
void
build_jump_edge(
	Edge const & edge,
	conformation::ResidueCAPs const & residues,
	AtomPointer2D & atom_pointer
);

/// @brief build a sub atom-tree for a polymer edge and attach it to main atom-tree
void
build_polymer_edge(
	Edge const & edge,
	conformation::ResidueCAPs const & residues,
	AtomPointer2D & atom_pointer
);

/// @brief build a sub atom-tree for a chemical edge and attach it to main atom-tree
void
build_chemical_edge(
	kinematics::Edge const & edge,
	conformation::ResidueCAPs const & residues,
	AtomPointer2D & atom_pointer
);


/// @brief  build the tree of atoms for this residue, anchored at root_atomno
void
build_residue_tree(
	int const root_atomno,
	conformation::Residue const & rsd,
	AtomPointer1D & atom_ptr,
	bool const root_atom_is_jump_atom
	//bool const keep_1st_child_position = false
);



/// @brief  build_residue_tree function that uses the foldtree info
/// @brief  also used in build_tree to build the residue tree for the root residue
void
build_residue_tree(
	conformation::ResidueCAPs const & residues,
	conformation::Residue const & rsd,
	FoldTree const & fold_tree,
	AtomPointer1D & atom_ptr
);


/// @brief creat an atom and add it to the residue atom-tree based on information stored in links.
tree::Atom*
add_atom(
	int const atomno,
	int const seqpos,
	utility::vector1< utility::vector1< Size > > const & links,
	AtomPointer1D & atom_ptr,
	bool const add_jump_atom
);


int
get_root_atomno(
	conformation::Residue const & rsd,
	int const dir // +1, -1, or "dir_jump"
);

/// @brief  Get the atom-index of the atom to which the residue at position seqpos should be anchored.
int
get_anchor_atomno(
	conformation::Residue const & anchor_rsd,
	Size const seqpos,
	FoldTree const & fold_tree
);


/// @brief get anchor atom to which the atom-tree of next residue in the edge is attached.
int
get_anchor_atomno(
	conformation::Residue const & rsd,
	int const dir // forward(1), backward(-1), or "dir_jump"
);


void
get_chemical_root_and_anchor_atomnos(
	conformation::Residue const & rsd_anchor,
	conformation::Residue const & rsd_root,
	Size & anchor_atom_no,
	Size & root_atom_no
);

/// @brief set up a map to look up TORSION_ID by DOF_ID (Map[DOF_ID] = TORISION_ID)
void
setup_dof_to_torsion_map(
	pose::Pose const & pose,
	id::DOF_ID_Map< id::TorsionID > & dof_map
);


/// @brief convert from allow-bb,allow-chi MoveMap to simple DOF_ID boolean mask needed by the minimizer
void
setup_dof_mask_from_move_map(
	MoveMap const & mm,
	pose::Pose const & pose,
	id::DOF_ID_Mask & dof_mask
);

int
pick_loopy_cutpoint(
	Size const n_res,
	ObjexxFCL::FArray1D_float const & cut_bias_sum
);

/// @brief set up a map to match mainchain atoms from residue1 to residue2
void
setup_corresponding_atoms(
	id::AtomID_Map< id::AtomID > & atom_map,
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2
);


tree::Atom*
setup_backrub_atom_tree(
	utility::vector1< id::AtomID > mainchain, // make our own local copy
	id::AtomID const & downstream_id, // mainchain child of last mainchain atom
	AtomPointer2D const & old_atom_pointer,
	utility::vector1< std::pair< Size, Size > > const & edges,
	Size const first_new_pseudo_residue
);


/// @brief  Use this routine to deduce atom indices of connect atoms in the tree
void
get_anchor_and_root_atoms(
	conformation::Residue const & anchor_rsd,
	conformation::Residue const & root_rsd,
	Edge const & edge,
	Size & anchor_atomno,
	Size & root_atomno
);



/// @brief  Helper function for conformation routines
void
replace_residue_in_atom_tree(
	conformation::Residue const & new_rsd,
	FoldTree const & fold_tree,
	conformation::ResidueCAPs const & residues,
	AtomTree & atom_tree
);

/// @brief  Inserts/ appends new residue subtree into an existing atomtree
/// @note  The foldtree must already have been changed to reflect the new residue
/// @note  The sequence position of the new residue is deduced from new_rsd.seqpos()
/// @note  This function handles renumbering of the atomtree if necessary
void
insert_residue_into_atom_tree(
	conformation::Residue const & new_rsd,
	FoldTree const & fold_tree,
	conformation::ResidueCAPs const & residues,
	AtomTree & atom_tree
);


/// @brief  Moves the first same-residue child of the jump atom corresponding to edge into first place in the child list
void
promote_sameresidue_child_of_jump_atom(
																			 Edge const & edge,
																			 conformation::ResidueCAPs const & residues,
																			 AtomPointer2D const & atom_pointer
																			 );


/// @brief  Moves the first same-residue child of the jump atom corresponding to edge into first place in the child list
void
promote_sameresidue_child_of_jump_atom(
																			 Edge const & edge,
																			 conformation::ResidueCAPs const & residues,
																			 AtomTree & atom_tree
																			 );
/// @brief prints something like this ***1***C***1*********2***C********3****C****2********3*****
void
simple_visualize_fold_tree( FoldTree const & fold_tree, std::ostream& out );

/// @brief prints something like this ***1***C***1*********2***C********3****C****2********3*****
///                                   **********xxxxxxxxxxxxx************************************
void
simple_visualize_fold_tree_and_movemap( FoldTree const & fold_tree,  MoveMap const& mm, std::ostream& out );

/// @brief prints something like this ***1***C***1*********2***C********3****C****2********3*****
///                                   **********xxxxxxxxxxxxx************************************
void
simple_visualize_fold_tree_and_movemap_bb_chi( FoldTree const & fold_tree, MoveMap const& mm, std::ostream& out );

///@brief linearizes (or defoliates, if you prefer) a FoldTree.  "default" FoldTrees produced by the PDB reader have all chains (peptide edges) starting from jumps relative to residue 1.  This code modifies the tree to instead have all the jumps be relative to the preceding edge.  It is not tested with ligands and will not work with "functional" jumps.  From A to B:
///A:FOLD_TREE  EDGE 1 78 -1  EDGE 1 79 1   EDGE 79 454 -1  EDGE 1 455 2    EDGE 455 540 -1  EDGE 1 541 3    EDGE 541 697 -1
///B:FOLD_TREE  EDGE 1 78 -1  EDGE 78 79 1  EDGE 79 454 -1  EDGE 454 455 2  EDGE 455 540 -1  EDGE 540 541 3  EDGE 541 697 -1
core::kinematics::FoldTree
linearize_fold_tree( core::kinematics::FoldTree const & tree );

} // namespace kinematics
} // namespace core


#endif // INCLUDED_core_kinematics_util_HH
