// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/InterchainEnergy.cc
/// @brief  Statistically derived rotamer pair potentials
/// @detailed For docking (or between chains) only those residues at the interface
///						and between the two interfaces need to be evaluated
/// @author Monica Berrondo

#ifndef INCLUDED_core_scoring_InterchainPotential_HH
#define INCLUDED_core_scoring_InterchainPotential_HH

#include <core/types.hh>

// Unit headers
#include <core/scoring/AtomVDW.fwd.hh>
#include <core/scoring/InterchainPotential.fwd.hh>
#include <core/scoring/EnvPairPotential.hh>

// Package headers
// AUTO-REMOVED #include <core/scoring/EnergyGraph.fwd.hh>
#include <core/conformation/Interface.hh>
#include <core/conformation/Residue.fwd.hh>

#include <core/util/datacache/CacheableData.hh>


// Project headers

// Utility headers
// AUTO-REMOVED #include <utility/vector1.hh>

#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// C++


namespace core {
namespace scoring {
using namespace conformation;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief Keep track of the interface information

class InterfaceInfo : public util::datacache::CacheableData {
public:

	///@brief Default constructor with rb jump = 1
	InterfaceInfo(): calculated_(false)
	{
		rb_jump_.push_back( 1 );
		distance_ = 6.0; //default
	}

	///@brief Constructor with arguments for non-default rb jump
	InterfaceInfo( Size rb_jump_in ): calculated_(false), distance_ (6.0)
	{
		rb_jump_.push_back( rb_jump_in );
		distance_ = 6.0; //default
	}

	///@brief Constructor with arguments for multiple jumps
	InterfaceInfo( utility::vector1_size rb_jump_in ): calculated_(false)
	{
		rb_jump_ = rb_jump_in;
		distance_ = 6.0; //default
	}

	InterfaceInfo( InterfaceInfo const & src );

	util::datacache::CacheableDataOP
	clone() const
	{
		return new InterfaceInfo( *this );
	}

	///@brief Removes all jumps from the interface calculation
	void clear_jumps(){ rb_jump_.clear(); }

	///@brief Adds another jump to the interface calculation, for example
	///for multi-body docking
	void add_jump( Size jump_in ){ rb_jump_.push_back( jump_in ); }

	///@brief Sets the distance cutoff for interface calculations
	void distance( Real distance_in ){ distance_ = distance_in; }

	///@brief Returns if interface calculation is up to date
	bool calculated() const { return calculated_; }

	///@brief Returns if interface calculation is up to date
	bool & calculated() { return calculated_; }

	///@brief Returns the number of jumps that are being used
	/// in interface calculations
	Size num_jump() const { return num_jump_; }

	///@brief Sets up InterfaceInfo members such as interface_list_
	///based on variables from construction
	void
	initialize();

	///@brief Returns whether a residue is at any of the interfaces
	bool
	is_interface( conformation::Residue rsd ) const ;

	///@brief Returns whether the two residues are considered a
	///residue pair at any of the interfaces
	bool
	is_pair(
		conformation::Residue rsd1,
		conformation::Residue rsd2
	) const;

	///@brief Returns the number of resides at the interface defined
	///by jump_num
	Size
	interface_nres( Size jump_num ) const;

	///@brief Calculates the interface for all jumps specified in
	///rb_jump_
	void
	calculate( pose::Pose const & pose);

	private:
	bool calculated_;
	Size num_jump_;
	Real distance_;

	utility::vector1_size rb_jump_;

	utility::vector1 < InterfaceOP > interface_list_;
};

class InterchainPotential : public EnvPairPotential {

public:
	InterchainPotential();

	void
	compute_interface( pose::Pose & pose ) const;

	void
	finalize( pose::Pose & pose ) const;

	///
	void
	evaluate_env_score(
		pose::Pose const & pose,
		conformation::Residue const & rsd,
		Real & env_score
	) const;

	///
	void
	evaluate_contact_score(
		pose::Pose const & pose,
		Real & contact_score
	) const;

	///
	void
	evaluate_pair_and_vdw_score(
		pose::Pose const & pose,
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		Real & pair_score,
		Real & vdw_score
	) const;

	// Commention out to make PyRosetta compile (undefined in .cc file)
	//Size interface_residues( pose::Pose const & pose ) const;

	InterfaceInfo const & interface_from_pose( pose::Pose const & ) const;
	InterfaceInfo & nonconst_interface_from_pose(  pose::Pose & ) const;

private: // data

	// const-ref to scoring database
	AtomVDW const & atom_vdw_;

	ObjexxFCL::FArray2D< Real > interchain_env_log_;
	ObjexxFCL::FArray2D< Real > interchain_pair_log_;
};

} // ns scoring
} // ns core

#endif
