// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/types.hh
/// @brief  core::scoring package type declarations
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


#ifndef INCLUDED_core_scoring_hbonds_create_poly_HH
#define INCLUDED_core_scoring_hbonds_create_poly_HH

#include <core/scoring/hbonds/constants.hh>

namespace core {
namespace scoring {
namespace hbonds {


//JSS Useful #defines to set up hbonds_geom scoring polynomials of the form
// name(/*in*/ double x, /*out*/ double &value, double &deriv);
// You shouldn't have to look at these; The idea is to auto-generate
// the calls to create_poly#() using matlab scripts to fit polynomials.
// POLY_CLIPVAL is the polynomial value outside of clip interval [xmin xmax]
// See hbonds_geom.cc for more comments.
#define POLY_CLIPVAL 0.0
#define create_poly8(name, xmin, xmax, c7, c6, c5, c4, c3, c2, c1, c0) inline void name(double x, double &value, double &deriv) { if ( (x<=xmin) || (x>=xmax) ) {value = POLY_CLIPVAL; deriv = 0.0; return; } value = deriv = c7; value = value*x+c6; deriv = deriv*x+value; value = value*x+c5; deriv = deriv*x+value; value = value*x+c4; deriv = deriv*x+value; value = value*x+c3; deriv = deriv*x+value; value = value*x+c2; deriv = deriv*x+value; value = value*x+c1; deriv = deriv*x+value; value = value*x+c0; }
#define create_poly7(name, xmin, xmax, c6, c5, c4, c3, c2, c1, c0) inline void name(double x, double &value, double &deriv) { if ( (x<=xmin) || (x>=xmax) ) {value = POLY_CLIPVAL; deriv = 0.0; return; } value = deriv = c6; value = value*x+c5; deriv = deriv*x+value; value = value*x+c4; deriv = deriv*x+value; value = value*x+c3; deriv = deriv*x+value; value = value*x+c2; deriv = deriv*x+value; value = value*x+c1; deriv = deriv*x+value; value = value*x+c0; }
#define create_poly6(name, xmin, xmax, c5, c4, c3, c2, c1, c0) inline void name(double x, double &value, double &deriv) { if ( (x<=xmin) || (x>=xmax) ) {value = POLY_CLIPVAL; deriv = 0.0; return; } value = deriv = c5; value = value*x+c4; deriv = deriv*x+value; value = value*x+c3; deriv = deriv*x+value; value = value*x+c2; deriv = deriv*x+value; value = value*x+c1; deriv = deriv*x+value; value = value*x+c0; }
#define create_poly5(name, xmin, xmax, c4, c3, c2, c1, c0) inline void name(double x, double &value, double &deriv) { if ( (x<=xmin) || (x>=xmax) ) {value = POLY_CLIPVAL; deriv = 0.0; return; } value = deriv = c4; value = value*x+c3; deriv = deriv*x+value; value = value*x+c2; deriv = deriv*x+value; value = value*x+c1; deriv = deriv*x+value; value = value*x+c0; }
#define create_poly4(name, xmin, xmax, c3, c2, c1, c0) inline void name(double x, double &value, double &deriv) { if ( (x<=xmin) || (x>=xmax) ) {value = POLY_CLIPVAL; deriv = 0.0; return; } value = deriv = c3; value = value*x+c2; deriv = deriv*x+value; value = value*x+c1; deriv = deriv*x+value; value = value*x+c0; }
#define create_poly3(name, xmin, xmax, c2, c1, c0) inline void name(double x, double &value, double &deriv) { if ( (x<=xmin) || (x>=xmax) ) {value = POLY_CLIPVAL; deriv = 0.0; return; } value = deriv = c2; value = value*x+c1; deriv = deriv*x+value; value = value*x+c0; }
#define create_poly2(name, xmin, xmax, c1, c0) inline void name(double x, double &value, double &deriv) { if ( (x<=xmin) || (x>=xmax) ) {value = POLY_CLIPVAL; deriv = 0.0; return; } value = deriv = c1; value = value*x+c0; }


// Since C++ does not yet have unnamed (lambda) functions in the standard,
// I use #define to set up named hbond scoring polynomials that are called
// as name(/*in*/ double x, /*out*/ double &value, double &deriv);
// The idea is to auto-generate the create_poly#() by matlab scripts that
// fit polynomials; if you enter coodinates by hand and have typos,
// the error message may be cryptic becaue of the #defines (but at least
// you'll get them -- the old fortran-style allowed disagreements between
// degree and # of coefficients without complaint.)
//
// Polynomial coefficients are listed from most to least significant:
// e.g., create_poly8(poly_name, xmin, xmax, coefs on x^7..x^0 );
// This is the same order as MATLAB so polynomials can be made by polyfit
// and displayed by x=xmin:0.01:xmax; plot(x,polyval(coeffs, x));
//
// Each score polynomial is clipped to an interval [xmin,xmax], which
// is currently important only to the distance polynomials.
// Outside the interval, the derivative = 0.0 and value = POLY_CLIPVAL;
// If you want the polynomial to always be non-positive, you can find the
// interval by combining roots(coeffs) information with MATLAB plots.
//
// Formerly there was a scale and offset applied at evaluation time;
// those must be built into the polynomial now (WYSIWYG),
// but this is easily done with the MATLAB polynomial fitting scripts.

// define polynomials for use in hbond score calculation
create_poly8(poly_AHdist_1, MIN_R, 2.8, 12.9376808600000, -221.0155722000000, 1604.3913040000000, -6409.3357730000000, 15200.8642500000000, -21375.0021600000000, 16475.9881100000000, -5361.5564400000000)
create_poly8(poly_AHdist_2, MIN_R, 2.8, 18.7240950300000, -317.6123713000000, 2286.8313450000000, -9051.6212090000000, 21249.4097200000000, -29551.2671600000000, 22511.3368900000000, -7235.5744970000000)
create_poly8(poly_AHdist_3, MIN_R, 2.745, 13.5898024400000, -224.0452428000000, 1568.9330940000000, -6044.2578470000000, 13820.1498000000000, -18730.9607600000000, 13912.9223800000000, -4361.9954250000000)
create_poly5(poly_AHdist_4, MIN_R, 2.5, 10.9872773800000, -100.2401419000000, 340.9733405000000, -511.6111233000000, 285.0061262000000)
create_poly5(poly_AHdist_5, MIN_R, 2.5, 7.0117355380000, -68.9996882900000, 251.8209310000000, -403.3593133000000, 238.7378958000000)
create_poly8(poly_cosAHD_1, MIN_xD, MAX_xD, 223.5268153000000, -757.7254095000000, 1019.5935080000000, -689.2232431000000, 240.1436064000000, -37.8411958300000, 0.8586890400000, 0.2781819850000)
create_poly8(poly_cosAHD_2, MIN_xD, MAX_xD, 213.1474671353920, -759.4777162626370, 1079.1258026902300, -780.9753519683800, 303.3394207482390, -61.7775612464580, 5.9636572265530, 0.0945417123440)
create_poly8(poly_cosAHD_3, MIN_xD, MAX_xD, 111.9877946000000, -380.3066184000000, 514.7650204000000, -352.4092342000000, 124.6219703000000, -19.9440194600000, 0.1493149790000, 0.6357717740000)
create_poly3(poly_cosAHD_4, MIN_xD, MAX_xD, -0.5625825030000, -0.7466826680000, 0.8092651710000)
create_poly3(poly_cosAHD_5, MIN_xD, MAX_xD, 0.0949628850000, -0.2543131720000, 0.0000000000000)
create_poly3(poly_cosAHD_6, MIN_xD, MAX_xD, -0.1001401440000, -1.1391390410000, 0.7392791860000)
create_poly3(poly_cosAHD_7, MIN_xD, MAX_xD, 0.0893802210000, -0.2075037760000, 0.0000000000000)
create_poly8(poly_cosBAH_1, MIN_xH, MAX_xH, 54.8066433100000, -196.8196655000000, 295.9418886000000, -232.1056020000000, 96.9912456500000, -20.6091836100000, 1.5731698160000, 0.0007454580000)
create_poly8(poly_cosBAH_2, MIN_xH, MAX_xH, 6.6658720788160, -26.4142064887520, 64.0514279060350, -82.5793977573480, 52.6279929055660, -17.2948159438680, 2.3375193142830, 0.2010997938560)
create_poly8(poly_cosBAH_3, MIN_xH, MAX_xH, 43.9448384700000, -144.3836033000000, 193.5865176000000, -132.4469355000000, 47.2813728800000, -8.9458880120000, -0.2270351350000, 0.7919029950000)
create_poly3(poly_cosBAH_4, MIN_xH, MAX_xH, 1.7209846440000, -1.8552545730000, 0.0000000000000)
create_poly3(poly_cosBAH_5, MIN_xH, MAX_xH, 0.4395982490000, -0.4446730760000, 0.0000000000000)
create_poly3(poly_cosBAH_6, MIN_xH, MAX_xH, 1.7614878420000, -1.8769594060000, 0.0000000000000)
create_poly7(poly_cosBAH_7, MIN_xH, MAX_xH, 37.7443160000000, -117.7316740000000, 143.0759275000000, -86.2258835000000, 26.7448175000000, -4.4699705000000, 0.3658455000000)

// evaluation routine that is used by default

void
eval_1d_polynomials_classic(
	HBEvalType hbe,
	double const & AHdist, // input geometric paramters
	double const & cosBAH,
	double const & cosAHD,
	double & Pr,           // value of AHdist polynomial
	double & dPr,          // deriv of AHdist polynomial
	double & PSxH,         // value of cosBAH polynomial (when AHdist < SWITCH_DIS)
	double & dPSxH,        // deriv of cosBAH polynomial (when AHdist < SWITCH_DIS)
	double & PSxD,         // value of cosAHD polynomial (when AHdist < SWITCH_DIS)
	double & dPSxD,        // deriv of cosAHD polynomial (when AHdist < SWITCH_DIS)
	double & PLxH,         // value of cosBAH polynomial (when AHdist >= SWITCH_DIS)
	double & dPLxH,	       // deriv of cosBAH polynomial (when AHdist >= SWITCH_DIS)
	double & PLxD,         // value of cosAHD polynomial (when AHdist >= SWITCH_DIS)
	double & dPLxD         // deriv of cosAHD polynomial (when AHdist >= SWITCH_DIS)
);

// experimental evaluation routine

void
eval_1d_polynomials(
	HBEvalType hbe,
	double const & AHdist, // input geometric paramters
	double const & cosBAH,
	double const & cosAHD,
	double & Pr,           // value of AHdist polynomial
	double & dPr,          // deriv of AHdist polynomial
	double & PSxH,         // value of cosBAH polynomial (when AHdist < SWITCH_DIS)
	double & dPSxH,        // deriv of cosBAH polynomial (when AHdist < SWITCH_DIS)
	double & PSxD,         // value of cosAHD polynomial (when AHdist < SWITCH_DIS)
	double & dPSxD,        // deriv of cosAHD polynomial (when AHdist < SWITCH_DIS)
	double & PLxH,         // value of cosBAH polynomial (when AHdist >= SWITCH_DIS)
	double & dPLxH,	       // deriv of cosBAH polynomial (when AHdist >= SWITCH_DIS)
	double & PLxD,         // value of cosAHD polynomial (when AHdist >= SWITCH_DIS)
	double & dPLxD         // deriv of cosAHD polynomial (when AHdist >= SWITCH_DIS)
);



} // namespace hbonds
} // namespace scoring
} // namespace core


#endif // INCLUDED_core_scoring_types_HH
