// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/SurfaceEnergy.cc
/// @author Ron Jacak


// Unit headers
#include <core/scoring/methods/SurfaceEnergy.hh>
#include <core/scoring/methods/SurfaceEnergyCreator.hh>

// Package headers
#include <core/scoring/SurfacePotential.hh>
//#include <core/scoring/ScoringManager.hh>

// Project headers
#include <core/pose/Pose.hh>
// AUTO-REMOVED #include <core/conformation/Residue.hh>
#include <core/scoring/Energies.hh>

// Utility headers
#include <core/util/Tracer.hh>

static core::util::Tracer TR("core.scoring.methods.SurfaceEnergy");

// C++ headers


namespace core {
namespace scoring {
namespace methods {


/// @details This must return a fresh instance of the SurfaceEnergy class,
/// never an instance already in use
methods::EnergyMethodOP
SurfaceEnergyCreator::create_energy_method(
	methods::EnergyMethodOptions const &
) const {
	return new SurfaceEnergy;
}

ScoreTypes
SurfaceEnergyCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( surface );
	return sts;
}


SurfaceEnergy::SurfaceEnergy() :
	parent( new SurfaceEnergyCreator )
{}


EnergyMethodOP
SurfaceEnergy::clone() const {
	return new SurfaceEnergy();
}


void
SurfaceEnergy::setup_for_scoring( pose::Pose & /*pose*/, ScoreFunction const & /*sf*/ ) const {
	// since this is a fake EnergyMethod, don't do anything here
	// is it even necessary to implement this method as empty?
}


void
SurfaceEnergy::residue_energy( conformation::Residue const & /*rsd*/, pose::Pose const & /*pose*/, EnergyMap & /*emap*/ ) const {

	// if this were a real term, the code here might look like the following
	//Real surface_score( 0.0 );
	//surface_score = evaluate_env_score( pose, rsd )
	//emap [ surface ] += surface_score

}

// SurfaceEnergy is non-pairwise decomposable, so it can only be calculated all at once, not on a residue by residue
// basis. For packing, it uses a specialized InteractionGraph that updates the score efficiently for substitutions.
void
SurfaceEnergy::finalize_total_energy( pose::Pose & pose, ScoreFunction const &, EnergyMap & totals ) const {

	// don't run if minimizing, non-differentiable
	if ( ! pose.energies().use_nblist() ) {
		core::Real surface_score;
		scoring::SurfacePotential::get_instance()->compute_pose_surface_energy( pose, surface_score );
		//TR << "surface score: " << surface_score << std::endl;
		totals[ surface ] = surface_score;
	}

}


}
}
}
