// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/core/sequence/sequtil.hh
/// @brief small bundle of utilities for dealing with sequences
/// @author James Thompson

#ifndef INCLUDED_core_sequence_util_HH
#define INCLUDED_core_sequence_util_HH

#include <core/types.hh>

#include <core/pose/Pose.hh>

#include <core/sequence/Sequence.fwd.hh>
#include <core/sequence/ScoringScheme.fwd.hh>
#include <core/sequence/SequenceMapping.fwd.hh>
#include <core/sequence/SequenceAlignment.fwd.hh>

#include <utility/vector1.fwd.hh>


namespace core {
namespace sequence {

	/// @brief helper function for reading a SequenceMapping from an alignment
	/// file.
	void
	read_alignment_file(
		std::string const & filename,
		std::string & seq1,
		std::string & seq2,
		sequence::SequenceMapping & mapping // from numbering in sequence 1 to numbering in sequence 2
	);

  /// @brief Read in sequences from a fasta-formatted file.
  utility::vector1< SequenceOP > read_fasta_file( std::string const & filename );
  utility::vector1< std::string > read_fasta_file_str( std::string const & filename );
  std::string read_fasta_file_return_str( std::string const & filename );

	/// @brief Read in a SequenceMapping from a file. File
	/// format is super-simple, it just contains single
	/// lines like this that claim that residue resi and
	/// resj are aligned: resi resj
	SequenceMapping simple_mapping_from_file( std::string const & filename );

	utility::vector1< SequenceAlignment > read_aln(
		std::string const & format,
		std::string const & filename
	);

	utility::vector1< SequenceOP > seqs_from_cmd_lines();

	/// @brief read generalized alignment format.
	utility::vector1< SequenceAlignment > read_general_aln(
		std::istream & input
	);

	utility::vector1< SequenceAlignment > read_grishin_aln_file(
	   std::string const & filename
	);

	utility::vector1< SequenceAlignment > read_general_aln_file(
		std::string const & filename
	);

	// @brief returns the number of correctly aligned positions in candidate_aln
	// relative to true_aln.
	core::Size n_correctly_aligned_positions(
		SequenceAlignment & candidate_aln,
		SequenceAlignment & true_aln
	);

	/// @brief takes the sequences in the provided vector1 and makes them match
	/// the alignment in aln_to_steal by matching gaps. This assumes that the
	/// ungapped sequences at index j in the vector1< SequenceOP > match the
	/// ungapped sequences at index j in aln_to_steal.
	SequenceAlignment steal_alignment(
		SequenceAlignment aln_to_steal,
		utility::vector1< SequenceOP > seqs
	);

	/// @brief Constructs a SequenceAlignment from the given SequenceMapping and
	/// the two sequences.
	SequenceAlignment mapping_to_alignment(
		SequenceMapping const & mapping,
		SequenceOP seq1,
		SequenceOP seq2
	);

	/// @brief Assuming that map1 maps sequence A to sequence B, and map2 maps
	/// sequence B to sequence C, this function returns the SequenceMapping
	/// representing the direct map of sequence A to sequence C.
	SequenceMapping transitive_map(
		SequenceMapping const & map1,
		SequenceMapping const & map2
	);

	/// @brief Generates a mapping of sequence 1 onto sequence 2 using dynamic
	/// programming with a simple scoring framework.
	core::sequence::SequenceMapping map_seq1_seq2(
		core::sequence::SequenceOP seq1,
		core::sequence::SequenceOP seq2
	);

	/// @brief Generate a naive sequence alignment between two sequences.
	core::sequence::SequenceAlignment align_naive(
		core::sequence::SequenceOP seq1,
		core::sequence::SequenceOP seq2
	);

	core::sequence::SequenceAlignment align_poses_naive(
		core::pose::Pose & pose1,
		core::pose::Pose & pose2
	);

	utility::vector1< Real >
	get_maximum_scores(
		core::sequence::ScoringSchemeOP ss,
		core::sequence::SequenceOP seq
	);

} // sequence
} // core

#endif
