// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   integration/make_Protein.cc
/// @brief  Create a Protein object from "classic" Rosetta data structures
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


// Unit headers
#include <integration/make_Protein.hh>

// Package headers

// Rosetta headers
#include <aaproperties_pack.h>
#include <after_opts.h>
#include <misc.h>
#include <param_aa.h>

// libRosetta headers
#include <core/conformation/Atom.hh>
#include <core/conformation/amino/keys/AtomKeys.hh>
#include <core/conformation/amino/backbone/keys/BackboneUnitKey.hh>
#include <core/conformation/amino/backbone/keys/BackboneUnitKeys.hh>
#include <core/conformation/internal/Protein_.hh>

// ObjexxFCL headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// Standard library headers
#include <cassert>


namespace integration {


/// @brief Create a Protein object from "classic" Rosetta data structures
core::conformation::ProteinOP
make_Protein(
	int const res_beg, // First residue in protein
	int const res_end  // Last residue in protein
)
{
	using namespace core::conformation;
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param_aa;

	// Create the empty protein on the heap                               // DO WE WANT A VERSION THAT RETURNS PROTEIN BY VALUE?  PROBABLY NOT: COPY CTORS ARE $$$ AND COMPLEX
	ProteinOP protein_p( new Protein_() );
	Protein & protein( *protein_p );

	// Add the residues
	for ( int i = res_beg; i <= res_end; ++i ) {
		using namespace core::conformation::amino;

		// Set BackboneUnit key
		BackboneUnitKey bu_key;
		if ( truefalseoption( "termini" ) ) { // Full termini              // THIS SHOULD BE A STAND-ALONE FUNCTION SINCE WE'LL PROBABLY NEED IT IN A FEW PLACES
			if ( res_beg == res_end ) { // Single residue "protein"
				bu_key = BackboneUnitKeys::NCT;
			} else if ( i == res_beg ) {
				bu_key = BackboneUnitKeys::NT;
			} else if ( i == res_end ) {
				bu_key = BackboneUnitKeys::CT;
			} else {
				bu_key = BackboneUnitKeys::Mid;
			}
		} else { // Lite termini
			if ( res_beg == res_end ) { // Single residue "protein"
				bu_key = BackboneUnitKeys::NCTL;
			} else if ( i == res_beg ) {
				bu_key = BackboneUnitKeys::NTL;
			} else if ( i == res_end ) {
				bu_key = BackboneUnitKeys::CTL;
			} else {
				bu_key = BackboneUnitKeys::Mid;
			}
		}

		// Add the amino acid residue to the protein
		AminoAcid & aa( protein.add( aa_name3( res(i) ), bu_key ) );

		// Set the atom positions
		for ( int j = 1, je = natoms( res(i), res_variant(i) ); j <= je; ++j ) {        // DOES natoms GET TERMINI ATOMS?  IF NOT WE NEED TO FIND/ASSIGN THOSE COORDS
			assert( AtomKeys::has( atom_name( j, res(i), res_variant(i) ) ) );                   // May need Nterm_atom_name_complete & Cterm_atom_name_complete for terminus atoms when -termini used ???
			amino::AtomKey const & atom_key( AtomKeys::key( atom_name( j, res(i), res_variant(i) ) ) );
			aa[ atom_key ].position( Atom::Position( &full_coord(1,j,i) ) ); // Assignment from pointer to contiguous (x,y,z) values
		}

	}

	return protein_p;
}


} // namespace integration
